package edu.mayo.bior.catalog.verification;

import edu.mayo.bior.catalog.CatalogDataSource;
import edu.mayo.bior.catalog.CatalogFileUtils;
import edu.mayo.bior.catalog.HumanBuildAssembly;
import edu.mayo.pipes.util.BiorProperties;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.io.BufferedReader;
import java.io.FileReader;
import java.io.IOException;
import java.util.*;

/**
 * Hold paths to files to help check Human Reference data
 */
public class HumanReferenceInfo
{
   private static Logger sLogger = LoggerFactory.getLogger(HumanReferenceInfo.class);

   // Attributes for positional catalog checking:
   private Map<String, Long> mChrSizeMap = null;
   private String mRefseqFileNm = null;
   private List<String> mExpectedChrOrderList = null;

   private String grch37ChrSizes;
   private String grch37ChrOrder;
   private String grch37RefSeq;

   private String grch38ChrSizes;
   private String grch38ChrOrder;
   private String grch38RefSeq;
   
   private BiorProperties biorProps;

   private void loadHumanRefAssemblyPaths() throws VerifierInputException
   {
	   try {
		   biorProps = new BiorProperties();

		   // GRCh37
		   grch37ChrSizes = biorProps.get(BiorProperties.Key.humanRefChrSizesGrch37File);
		   grch37ChrOrder = biorProps.get(BiorProperties.Key.humanRefChrOrderGrch37File);
		   grch37RefSeq   = biorProps.get(BiorProperties.Key.humanRefSeqGrch37File);
	
		   // GRCh38
		   grch38ChrSizes = biorProps.get(BiorProperties.Key.humanRefChrSizesGrch38File);
		   grch38ChrOrder = biorProps.get(BiorProperties.Key.humanRefChrOrderGrch38File);
		   grch38RefSeq   = biorProps.get(BiorProperties.Key.humanRefSeqGrch38File);
	   }
	   catch (IOException e)
	   {
		   String msg = "Could not load Human Reference Assembly paths from the bior.properties file. Msg: " + e.getMessage();
		   sLogger.error(msg);
		   throw new VerifierInputException(msg);
	   }
   }

   public HumanReferenceInfo(HumanBuildAssembly humanRefAssembly, MessageLogger logger) throws VerifierInputException
   {
	   loadHumanRefAssemblyPaths();
	   
      if (humanRefAssembly != null )
      {
         if (humanRefAssembly == HumanBuildAssembly.GRCh37)
         {
            mRefseqFileNm = grch37RefSeq;
            initHumanReferenceInfo(grch37ChrSizes, grch37ChrOrder);
         }
         else if (humanRefAssembly == HumanBuildAssembly.GRCh38)
         {
            mRefseqFileNm = grch38RefSeq;
            initHumanReferenceInfo(grch38ChrSizes, grch38ChrOrder);
         }
         else
         {
            logger.logWarning("Can't process genome assembly [" + humanRefAssembly +
                              "]. Will not be verifying chromosomal positions or reference allele values.");
         }
      }
      else
      {
         logger.logWarning("Don't have genome assembly. Will not be verifying chromosomal positions or " +
                           "reference allele values.");
      }

   }

   public boolean canCheck()
   {
      return getChrSizeMap() != null && getExpectedChrOrderList() != null && getExpectedChrOrderList() != null;
   }

   public Map<String, Long> getChrSizeMap()
   {
      return mChrSizeMap;
   }

   public String getRefseqFileNm()
   {
      return mRefseqFileNm;
   }

   public List<String> getExpectedChrOrderList()
   {
      return mExpectedChrOrderList;
   }

   private void initHumanReferenceInfo(String chromosomeSizesFileName, String chromosomeOrderFileFileName)
      throws VerifierInputException
   {
      if (chromosomeSizesFileName == null || chromosomeSizesFileName.length() <= 0)
      {
         throw new VerifierInputException("Chromosome-size file specified is not valid. Cannot verify catalog positional elements without this file.");
      }
      if (chromosomeOrderFileFileName == null || chromosomeOrderFileFileName.length() <= 0)
      {
         throw new VerifierInputException("Chromosome-order file specified is not valid. Cannot verify catalog positional elements without this file.");
      }

      mChrSizeMap = initChrSizes(chromosomeSizesFileName);
      if (mChrSizeMap == null || mChrSizeMap.size() == 0)
      {
         throw new VerifierInputException("Chromosome-size map is empty. Cannot verify catalog positional elements without this info. File specified: " + chromosomeSizesFileName);
      }

      mExpectedChrOrderList = initChrOrder(chromosomeOrderFileFileName);
      if (mExpectedChrOrderList == null || mExpectedChrOrderList.size() == 0)
      {
         throw new VerifierInputException("Chromosome-order list is null or empty. Cannot verify catalog positional elements without this info. File specified: " + chromosomeOrderFileFileName);
      }
   }

   private TreeMap<String, Long> initChrSizes(String genomeRefChrSizeFileNm)
   {
      if (genomeRefChrSizeFileNm == null || genomeRefChrSizeFileNm.length() <= 0)
      {
         return null;
      }

      TreeMap<String, Long> map = new TreeMap<String, Long>();

      BufferedReader r = null;
      try
      {
         r = new BufferedReader(new FileReader(genomeRefChrSizeFileNm));
         String n = null;
         while ((n = r.readLine()) != null)
         {
            String[] elems = n.trim().split("\t");
            if (elems.length != 2)
            {
               r.close();
               throw new Exception("Genome chromosome size file not in expected format. There should be only two columns in the file: " + genomeRefChrSizeFileNm);
            }
            String chr = null;
            Long iPos = null;
            try
            {
               chr = elems[0].trim().replace("chr", "");
            }
            catch (Exception e)
            {
               chr = null;
            }
            try
            {
               iPos = new Long(elems[1].trim());
            }
            catch (Exception e)
            {
               iPos = null;
            }
            if (chr != null && iPos != null)
            {
               map.put(chr, iPos);
            }
         }
      }
      catch (Throwable t)
      {
         return null;
      }
      finally
      {
         try
         {
            if (r != null)
            {
               r.close();
            }
         }
         catch (Exception e)
         { /**/ }
      }

      return map;
   }

   private List<String> initChrOrder(String orderedChrFileName)
      throws VerifierInputException
   {
      List<String> orderedChrExpecting = new ArrayList<String>();

      BufferedReader r = null;
      try
      {
         r = CatalogFileUtils.getBufferedReader(orderedChrFileName);
         String l;
         while ((l = r.readLine()) != null)
         {
            String ln = l.trim();
            if (ln.length() > 0)
            {
               String chrVal = ln;
               if( ln.startsWith("#") ) { // comment
            	   continue;
               } else if (ln.startsWith("chr")) {
                  chrVal = ln.replace("chr", "");
               }
               orderedChrExpecting.add(chrVal);
            }
         }
      }
      catch (IOException io)
      {
         throw new VerifierInputException("Unable to read and initialize chromosome order file: " + orderedChrFileName);
      }
      finally
      {
         try
         {
            if (r != null)
            {
               r.close();
            }
         }
         catch (Exception e)
         {
            // if exception on trying to close, just eat this for now
         }
      }

      return orderedChrExpecting;
   }
}
