package edu.mayo.bior.catalog.verification;

import com.google.gson.*;
import edu.mayo.bior.catalog.*;
import edu.mayo.pipes.history.ColumnMetaData;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.io.*;
import java.util.*;

/**
 * Do row by row validation on catalog rows.  Verify:
 * - JSON is correctly formatted (try to parse it)
 * - JSON type and count
 * - _landmark matches first col in catalog (if > 1 cols)
 * - _minBP matches 2nd col in catalog (if > 1 cols)
 * - _maxBP matches 3rd col in catalgo (if > 1 cols)
 * - _refAllele matches the ref allele in the reference assembly
 * - _minBP and _maxBP fall within the chromosomal range
 */
public class CatalogRowVerifier
{
   private static final Logger sLogger = LoggerFactory.getLogger(CatalogRowVerifier.class);

   // 5GB. If catalog file bigger than this, stop reading after past chunk
   private static final long FILE_SIZE_THRESHOLD = 5l * 1024 * 1024 * 1024;

   private static final String SPACE = " ";
   public static final String UNKNOWN_CHR = "UNKNOWN";
   private File mCatalogFile;
   private Map<String, ColumnMetaData> mColumnInfo;
   private MessageLogger mLogger;
   private HumanReferenceInfo mReferenceInfo;
   private CatalogJsonVerifier mJsonVerifier;
   // If we are verifying only 1 in X number of lines:
   private long mOneInXLinesToVerify;
   // If we are verifying only a small chunk of the file:
   private long mChunkStartLine = 1;
   private long mChunkNumLines = Long.MAX_VALUE;

   
   private long rowsInCatalog = 0;
   private long rowsVerified = 0;

   public CatalogRowVerifier(File catalogFile, CatalogDataSource dataSource,
                             Map<String, ColumnMetaData> catalogColumns,
                             MessageLogger logger, long oneInXLinesToVerify)
      throws VerifierInputException
   {
      this(catalogFile, dataSource, catalogColumns, logger, oneInXLinesToVerify, /*startLine=*/1, /*numLines=*/Long.MAX_VALUE);
   }


   public CatalogRowVerifier(File catalogFile, CatalogDataSource dataSource,
                             Map<String, ColumnMetaData> catalogColumns,
                             MessageLogger logger, long oneInXLinesToVerify, long startLine, long numLines)
      throws VerifierInputException
   {
      mCatalogFile = catalogFile;
      mColumnInfo = catalogColumns;
      mLogger = logger;
      mJsonVerifier = new CatalogJsonVerifier(catalogColumns, logger);

      HumanBuildAssembly humanBuildAssembly = (dataSource == null) ? null : dataSource.getHumanBuildAssembly();
      mReferenceInfo = new HumanReferenceInfo(humanBuildAssembly, logger);

      // For quick verify, only select 1 in X lines to verify
      mOneInXLinesToVerify = oneInXLinesToVerify;

      mChunkStartLine = startLine;
      mChunkNumLines = numLines;
   }

   /** Allow test cases to modify the values for # of occurrences so we can test the int vs long limit */
   protected void _test_setCatalogJsonVerifier(CatalogJsonVerifier verifier) {
	   this.mJsonVerifier = verifier;
   }
   
   protected void _test_setChunk(long chunkStartLine, long chunkNumLines) {
	   mChunkStartLine = chunkStartLine;
	   mChunkNumLines  = chunkNumLines;
   }
   
   protected void _test_setRowCounts(long rowsInCtg, long rowsVerified) {
	   this.rowsInCatalog = rowsInCtg;
	   this.rowsVerified  = rowsVerified;
   }
   
   /**
    * reportColumnUsage: Write out catalog column info and check if zero data occurrences
    * for columns.
    */
   public void reportColumnUsage()
   {
      if (!haveColumnInfo())
      {
         logWarning("Cannot report usage stats as we have no information from columns.tsv. ");
         return;
      }

      for (String colName : mColumnInfo.keySet())
      {
         ColumnMetaData catCol = mColumnInfo.get(colName);
         StringBuilder msgBuf = new StringBuilder();
         msgBuf.append(String.format("Column '%s'", colName));

         long numTimesInCatalog = mJsonVerifier.getNumberOfTimesKeySeenInCatalog(colName);
         if (numTimesInCatalog == 0)
         {
            msgBuf.append(" not found in the catalog. Consider removing it from the columns.tsv file.");
            logWarning(msgBuf.toString());
         }
         else
         {
            msgBuf.append(String.format(": number of occurrences in catalog: %d.", numTimesInCatalog));
            if (catCol.getCount().equals("."))
            {
               msgBuf.append(String.format(" Largest number of elements found for array column: %d.",
                  mJsonVerifier.getMaxValuesForKey(colName)));
            }
            logInfo(msgBuf.toString());
         }
      }
   }

   public void verify() {
      CatalogVariantVerifier variantVerifier = null;
      
      logSamplingAndChunkSizeStats();
      
      BufferedReader catalogRdr = CatalogFileUtils.getBufferedReader(mCatalogFile.getAbsolutePath());
      if (catalogRdr == null) {
         logCannotReadCatalogError();
         return;
      }

      // Set this for later checking to see if you will leave file early
      boolean fileBiggerThanThreshold = false;
      if (mCatalogFile.length() > FILE_SIZE_THRESHOLD)
      {
         fileBiggerThanThreshold = true;
         sLogger.debug(String.format("Catalog file length %d > %d. Can stop reading early if past chunk.",
            mCatalogFile.length(), FILE_SIZE_THRESHOLD));
      }

      String chrProcessing = null;
      try {
         boolean seen4TabixColumns = false;
         boolean seen1TabixColumn = false;
         boolean keepChecking = true;
         catalogRdr = CatalogFileUtils.getBufferedReader(mCatalogFile.getAbsolutePath());
         String curCatalogRow;

         boolean stoppedReading = false;
         while ((curCatalogRow = catalogRdr.readLine()) != null && keepChecking) {
            if (curCatalogRow.trim().startsWith("#")) {
               continue;
            }

            // bail out if past chunk
            if (fileBiggerThanThreshold && ChunkUtils.beyondTargetChunk(rowsInCatalog + 1, mChunkStartLine, mChunkNumLines))
            {
               stoppedReading = true;
               sLogger.info(String.format("Stopped reading catalog after reading %d lines - past chunk",
                  rowsInCatalog));
               break;
            }

            logMemoryUseAndRowCountSoFar(rowsInCatalog, rowsVerified);

            rowsInCatalog++;

            // If we are outside the chunk, or have chosen to sample a certain % of the rows and this is not one of them, then skip this row
            // Even if we are past the end of the chunk (in which case we could stop),
            // we want to keep going with the row count so we know how many total rows the catalog has
            if (shouldSkipRow(rowsInCatalog)) {
               continue;
            }

            rowsVerified++;


            CatalogTabixEntry currentRow = VerifyUtils.readCatalogRow(curCatalogRow, mLogger);
            if (currentRow == null) {
               // no need to log a message. Already done in readCatalogRow
               continue;
            }

            if (currentRow.is4Field()) {
               seen4TabixColumns = true;
            } else if (currentRow.is1Field()) {
               seen1TabixColumn = true;
            }

            if (seen4TabixColumns && seen1TabixColumn) {
               logError("Saw a mix of 4 columns and 1 columns in catalog. Stopping checking", VerifyErrorCodes.CATALOG_COLUMN_COUNT_INCONSISTENT);
               keepChecking = false;
               continue;
            }

            // Due to memory constraints, only load the reference sequence for the chromosome that we are validating right now.
            // If we hit a row where the previous chr we were processing is not the chr for the current row, configure a new
            // SeqLookup for this next chromosome, logging the stats for previous before we move on.
            if (chrProcessing == null || !currentRow.getChromosome().equals(chrProcessing)) {
               // log for previous chr before initialize ref sequence for next chr.
               logRefAlleleNotMatchingSummary(variantVerifier, chrProcessing);

               // init ref seq for next chr processing:
               chrProcessing = currentRow.getChromosome();
               variantVerifier = new CatalogVariantVerifier(mReferenceInfo, chrProcessing, mLogger);
            }

            // Finally verify the catalog row that was read:
            verifyCatalogRow(currentRow, variantVerifier);
         }

         logEndResults(variantVerifier, chrProcessing, rowsInCatalog, rowsVerified, stoppedReading);
      }
      // TODO - only an IOException is thrown here (at least declared). Do we want to be more restrictive?
      catch (IOException e) {
         logError("Problem in catalog verify: " + e.getMessage(), VerifyErrorCodes.CATALOG_ROW_VERIFY_EXCEPTION);
      }
   }


   private void logCannotReadCatalogError() {
	   String msg = String.format("Couldn't read catalog '%s' for unknown reason", mCatalogFile.getPath());
	   logError(msg, VerifyErrorCodes.CATALOG_FILE_UNREADABLE);
	   logError("Won't be able to run row-by-row verify. Check logs for more detail.", VerifyErrorCodes.CATALOG_FILE_UNREADABLE_NO_ROW_BY_ROW_VERIFICATION);
   }


   private void logSamplingAndChunkSizeStats() {
	   String samplingStr = mOneInXLinesToVerify == 1 ? "each row" : ("1 in " + mOneInXLinesToVerify + " rows");
	   String statusMsg = String.format("Verifying %s for '%s' starting at %s", samplingStr, mCatalogFile.getPath(), VerifyUtils.composeDateTime());
	   System.out.println(statusMsg);
	   logInfo(statusMsg);
	      
	   String endChunkPosition = (mChunkNumLines == 0  ||  mChunkNumLines == Long.MAX_VALUE)  ?  "END"  :  (mChunkNumLines + "");
	   String chunkStr = "Chunk (lines): " + mChunkStartLine + " - " + endChunkPosition;
	   logInfo(chunkStr);
   }

   private void logEndResults(CatalogVariantVerifier variantVerifier, String chrProcessing,
                              long rowsInCatalog, long rowsVerified, boolean stoppedReading) {
	   logMemoryUseAndRowCountSoFarForcePrint(rowsInCatalog, rowsVerified);

	   // log for last chromosome
	   logRefAlleleNotMatchingSummary(variantVerifier, chrProcessing);

	   String statusMsg = String.format("Verified %d of total %d rows of catalog, finished at %s.",
         rowsVerified, rowsInCatalog, VerifyUtils.composeDateTime());
      if (stoppedReading) {
         statusMsg = String.format("Verified %d of total ??? rows of catalog (read %d rows and stopped reading after " +
            "chunk due to large catalog size), finished at %s.",
            rowsVerified, rowsInCatalog, VerifyUtils.composeDateTime());
      }
	   System.out.println(statusMsg);
	   logInfo(statusMsg);
   }


   /**
    * Skip the current row if it is either before the chunk start line, OR if it is not one of the 1-in-X rows to sample
    * Input: 1-based row #.
    */
   private boolean shouldSkipRow(long currentRowInCatalog)
   {
      if (!ChunkUtils.inTargetChunk(currentRowInCatalog, mChunkStartLine, mChunkNumLines))
      {
         return true;
      }

      long zeroBasedRowInCatalog = (currentRowInCatalog - 1);
      boolean isOneInXRowToSample = zeroBasedRowInCatalog % mOneInXLinesToVerify == 0;
      return !isOneInXRowToSample;
   }
   

   private void logMemoryUseAndRowCountSoFar(long rowsCountedInCatalogSoFar, long rowsVerified) {
	   if (rowsCountedInCatalogSoFar % 100000 == 0 ) {
		   logMemoryUseAndRowCountSoFarForcePrint(rowsCountedInCatalogSoFar, rowsVerified);
	   }
   }
   
   private void logMemoryUseAndRowCountSoFarForcePrint(long rowsCountedInCatalogSoFar, long rowsVerified) {

      // Don't put out message if 0 rows counted so far. It's confusing.
      if (rowsCountedInCatalogSoFar == 0)
      {
         return;
      }

      // TODO - what is this gc() about and why is it done twice?
      System.gc();
	   sLogger.info("Rows counted: " + rowsCountedInCatalogSoFar + ".  Rows verified: " + rowsVerified);
	   System.gc();
	   double currentMemory = ((double) ((double) (Runtime.getRuntime().totalMemory() / 1024) / 1024)) - ((double) ((double) (Runtime.getRuntime().freeMemory() / 1024) / 1024));
	   sLogger.debug("Memory used currently: " + currentMemory + " MB at " + rowsCountedInCatalogSoFar + " rows processed");
   }


   private void logRefAlleleNotMatchingSummary(CatalogVariantVerifier variantVerifier, String chrProcessing)
   {
      if (chrProcessing != null && variantVerifier != null)
      {
         if (variantVerifier.getNumberRefAllelesNotMatchReferenceSequence() != 0)
         {
            int code = VerifyErrorCodes.SUMMARY_SOME_REF_ALLELES_NOT_MATCHING_REF_SEQ;
            if (VerifyUtils.isChrM(chrProcessing))
            {
               code = VerifyErrorCodes.SUMMARY_SOME_REF_ALLELES_NOT_MATCHING_REF_SEQ_CHROM_M;
            }
            String noMatchMessage = "Chromosome " + chrProcessing +
               ": _refAllele values that DO NOT match the reference sequence is [" +
               variantVerifier.getNumberRefAllelesNotMatchReferenceSequence() + "].";
            logError(noMatchMessage, code);
         }
      }
   }

   private void verifyCatalogRow(CatalogTabixEntry catalogRowEntry, CatalogVariantVerifier variantVerifier) throws NumberFormatException, IOException
   {
      JsonObject catalogRowJsonObj;
      try
      {
         // This is a verification by itself... badly formatted JSON will fail here in
         // some ways, but maybe not all (expected data type).
         catalogRowJsonObj = VerifyUtils.getJsonObject(catalogRowEntry.getJsonString());
      }
      catch (JsonParseException jse)
      {
         // TODO - improve message by getting more information from jse
         logError("Exception parsing json value for tabix row =" + catalogRowEntry.getJsonString(), VerifyErrorCodes.JSON_UNPARSEABLE_ROW_VERIFICATION);
         return;
      }

      verifyRowTabix(catalogRowEntry);

      // TODO - should we note errors that are too bad to recover from and not continue if we see those?
      mJsonVerifier.verify(catalogRowEntry.getJsonString());

      CatalogEntryGoldenJson catalogRowGolden = verifyRowGoldenJson(catalogRowJsonObj, catalogRowEntry);
      // if this is null, messages have been logged and no point in going on for this row
      if (catalogRowGolden == null)
      {
         return;
      }

      // Compare the two -- tabix and golden:
      verifyTabixToGoldenElements(catalogRowEntry, catalogRowGolden);
      variantVerifier.verify(catalogRowJsonObj, catalogRowGolden);
   }

   protected void verifyRowTabix(CatalogTabixEntry tabixEntry)
   {
      if (tabixEntry.is4Field() && mReferenceInfo.canCheck())
      {
         // 1.) Make sure _landmark, _minBP, and _maxBP are all present because
         //     positional catalogs must have them. ALTHOUGH, there can be non-positional rows
         //     within a positional catalog.

         verifyTabixPositionsInChrRange(tabixEntry);
      }
   }

   protected CatalogEntryGoldenJson verifyRowGoldenJson(JsonObject catalogRowJson, CatalogTabixEntry catalogRowEntry)
   {
      CatalogEntryGoldenJson catalogRowGolden;

      try
      {
         catalogRowGolden = new CatalogEntryGoldenJson(catalogRowJson);
      }
      catch (CatalogFormatException e)
      {
         logError(e.getMessage(), VerifyErrorCodes.CATALOG_GOLDEN_ATTRIBUTES_PARSE_ERROR);
         return null;
      }

      if (catalogRowEntry.is1Field())
      {
         return catalogRowGolden;
      }

      Set<String> validChrs = null;
      if (mReferenceInfo.getChrSizeMap() != null && mReferenceInfo.getChrSizeMap().size() > 0)
      {
         validChrs = mReferenceInfo.getChrSizeMap().keySet();
      }
      verifyGoldenJsonLandmark(catalogRowGolden, catalogRowJson, validChrs);

      verifyGoldenJsonBp(catalogRowGolden.getMinBP(), catalogRowJson, "_minBP", VerifyErrorCodes.JSON_GOLDEN_ATTRIBUTE_MINBP_NOT_SPECIFIED);
      verifyGoldenJsonBp(catalogRowGolden.getMaxBP(), catalogRowJson, "_maxBP", VerifyErrorCodes.JSON_GOLDEN_ATTRIBUTE_MAXBP_NOT_SPECIFIED);

      return catalogRowGolden;
   }

   private void verifyGoldenJsonLandmark(CatalogEntryGoldenJson catalogRowGolden, JsonObject catalogRowJson,
                                         Set<String> validChrValues)
   {
      String chr = catalogRowGolden.getChr();
      if (chr == null)
      {
         // TODO this shouldn't really have the knowledge of _landmark key name but it makes an effective error message
         logWarning("_landmark golden value is not in JSON String, but entry is positional. Json: " +
               catalogRowJson.toString());
         return;
      }

      if (UNKNOWN_CHR.equals(chr))
      {
         return;
      }

      if (validChrValues != null && validChrValues.size() > 0)
      {
         if (!validChrValues.contains(chr))
         {
            logWarning("_landmark value [" + chr + "] not found in configured valid chromosomes list: " +
               validChrValues.toString());
         }
      }

      if (chr.equals("X") || chr.equals("Y") || chr.equals("M") || !chr.matches("\\d+"))
      {
         // This is ok; nothing to check here right now.
      }
      else
      {
         // TODO - does this make sense that you would assume ahead of time the list that is passed in. Maybe the
         // TODO - check above against validChrValues is enough
         // TODO - maybe just check if you have an integer and it's human, it should be between 1 and 22

         // Rest of landmarks should only be numbers:
         //   ie. a landmark of "MT" would get here but error out once here, and that is ok for now.
         try
         {
            Integer iLandmark = new Integer(chr);
            if (iLandmark < 1 || iLandmark > 22)
            { // We don't store X, Y, or M as 23, 24, or 25:
               logError("_landmark value should be between 1 and 22 [" + chr +
                     "], but is not. Json: " + catalogRowJson.toString(),
                  VerifyErrorCodes.CHROMOSOME_NOT_IN_INTEGER_RANGE);
            }
         }
         catch (Exception e)
         {
            logError("_landmark value should be valid numeric chr value at this point [" + chr +
                  "], but could not create Integer object for value. Json : " + catalogRowJson.toString(),
               VerifyErrorCodes.CHROMOSOME_SHOULD_BE_INTEGER);
         }
      }

   }

   private boolean haveColumnInfo()
   {
      return mColumnInfo != null && mColumnInfo.size() > 0;
   }

   private void verifyGoldenJsonBp(Long bp, JsonObject catalogRowJson, String key, int errorCode)
   {
      if (bp == null)
      {
         logError(String.format("%s should be set. Json '%s'", key, catalogRowJson.toString()), errorCode);
      }
   }

   protected void verifyTabixToGoldenElements(CatalogTabixEntry tabixRowEntry, CatalogEntryGoldenJson goldenVariant)
   {
      if (tabixRowEntry.is1Field()) {
         return;
      }

      String jsonLandmarkChr = goldenVariant.getChr();
      if (!tabixRowEntry.getChromosome().equals(jsonLandmarkChr)) {
    	  String msg = String.format("Tabix chr [%s] not same as _landmark value [%s].  Row: %s",
    			  		 tabixRowEntry.getChromosome(), jsonLandmarkChr,  getFirstXChars(tabixRowEntry.getLine(), 100));
    	  logError(msg, VerifyErrorCodes.TABIX_CHROMOSOME_DIFFERENT_FROM_LANDMARK);
      }

      Long jsonGoldenIntMinBP = goldenVariant.getMinBP();
      if (jsonGoldenIntMinBP != null) {
         if (tabixRowEntry.getMinPosition().intValue() != jsonGoldenIntMinBP.intValue()) {
        	 String msg = String.format("Tabix start position [%d] not same as _minBP value [%d].  Row: %s",
                      		tabixRowEntry.getMinPosition(), jsonGoldenIntMinBP,  getFirstXChars(tabixRowEntry.getLine(), 100));
        	 logError(msg, VerifyErrorCodes.TABIX_START_POSITION_DIFFERENT_FROM_MINBP);
          }
      }

      Long jsonGoldenIntMaxBP = goldenVariant.getMaxBP();
      if (jsonGoldenIntMaxBP != null) {
         if (tabixRowEntry.getMaxPosition().intValue() != jsonGoldenIntMaxBP.intValue()) {
        	 String msg = String.format("Tabix stop position [%d] not same as _maxBP value [%d].  Row: %s",
        			 		tabixRowEntry.getMaxPosition(), jsonGoldenIntMaxBP, getFirstXChars(tabixRowEntry.getLine(), 100));
        	 logError(msg, VerifyErrorCodes.TABIX_STOP_POSITION_DIFFERENT_FROM_MAXBP);
         }
      }

      if (tabixRowEntry.getMaxPosition() < tabixRowEntry.getMinPosition()) {
    	  String msg = String.format("Tabix stop position [%d] less than tabix start position [%d]. Invalid catalog format.  Row: %s",
    			  		tabixRowEntry.getMaxPosition(), tabixRowEntry.getMinPosition(), getFirstXChars(tabixRowEntry.getLine(), 100));
          logError(msg, VerifyErrorCodes.TABIX_STOP_POSITION_LESS_THAN_START);
      }
      if (jsonGoldenIntMinBP != null && jsonGoldenIntMaxBP != null) {
         if (jsonGoldenIntMaxBP < jsonGoldenIntMinBP) {
        	 String msg = String.format("Golden _maxBP stop position [%d] less than golden _minBP start value [%d].  Invalid catalog format.  Row: %s",
        			 		jsonGoldenIntMaxBP, jsonGoldenIntMinBP, getFirstXChars(tabixRowEntry.getLine(), 100));
             logError(msg, VerifyErrorCodes.MAXBP_LESS_THAN_MINBP);
         }
      }
   }


   private Object getFirstXChars(String line, int numChars) {
	   if( line == null )
		   return "(null)";
	   if( line.length() <= numChars )
		   return line;
	   return line.substring(0, numChars) + "...";
   }


/**
    * verifyTabixPositionsInChrRange: If this catalog row is a positional tabix row within the catalog, make sure tabix and
    * json entries are valid based on this positional catalog's reference information (chromosome
    * size map). Chromosome size map set based datasource.properties file "Build" property.
    *
    * @param tabixEntry
    */
   private void verifyTabixPositionsInChrRange(CatalogTabixEntry tabixEntry)
   {

      // NOTE: There can be some non-positional catalog entries even in a positional
      //       catalog so we need to account for those where we don't have positional
      //       info for this catalog entry. The only supported non-positional format
      //       as of 4/19/2016:
      //			UNKNOWN	0	1
      //       But for a short time, we'll also need to support
      //          .	0	0    (omim)
      //       And possibly:
      //        	UNKNOWN	0	0  (because current bior_create_catalog does this)
      // TODO do we need to add . 0 0 and UNKNOWN 0 0 to isValidNonPositional?
      if (tabixEntry.isValidNonPositional())
      {
         return;
      }

      if (!tabixEntry.isPositional())
      {
         logError("Tabix entry is neither positional or non-positional. " +
               " Chr: " + tabixEntry.getChromosome() +
               " Tabix entry min position: " + tabixEntry.getMinPosition() +
               " Tabix entry max position: " + tabixEntry.getMaxPosition(),
            VerifyErrorCodes.TABIX_ENTRY_NOT_POSITIONAL_OR_NONPOSITIONAL);
         return;
      }

      // past special case, now just make sure valid positional elements:
      if (!inChromosomeRange(tabixEntry.getChromosome(), tabixEntry.getMinPosition()))
      {
         logError("Tabix start position not in valid chromosome size range. " +
               " Chr: " + tabixEntry.getChromosome() +
               " Chr Max Position: " + mReferenceInfo.getChrSizeMap().get(tabixEntry.getChromosome()) +
               " Tabix entry min position: " + tabixEntry.getMinPosition(),
            VerifyErrorCodes.TABIX_START_POSITION_NOT_IN_CHROMOSOME_RANGE);
      }
      if (!inChromosomeRange(tabixEntry.getChromosome(), tabixEntry.getMaxPosition()))
      {
         logError("Tabix stop position not in valid chromosome size range. " +
               " Chr: " + tabixEntry.getChromosome() +
               " Chr Max Position: " + mReferenceInfo.getChrSizeMap().get(tabixEntry.getChromosome()) +
               " Tabix entry min position: " + tabixEntry.getMaxPosition(),
            VerifyErrorCodes.TABIX_STOP_POSITION_NOT_IN_CHROMOSOME_RANGE);
      }
   }

   private boolean inChromosomeRange(String chr, long pos)
   {
      Long chrSize = mReferenceInfo.getChrSizeMap().get(chr);

      if (VerifyUtils.isEmpty(chr))
      {
         logError("Chromosome value is empty. Cannot check position [" + pos + "] for chromosome value.", VerifyErrorCodes.CHROMOSOME_MISSING);
         return false;
      }
      else if (chrSize == null)
      {
         logError("Chromosome value not found in chromosome size map: " + chr + " Cannot check position [" + pos + "] for chromosome value.", VerifyErrorCodes.CHROMOSOME_NOT_FOUND_IN_CHROM_SIZE_MAP);
         return false;
      }

      // Is a non-positional _minBP or _maxBP set to a 1?  I would think it would be initialized to -1
      if (pos >= 1 && pos <= chrSize)
      {
         return true;
      }
      else
      {
         return false;
      }
   }
   
   private void logInfo(String msg)
   {
      mLogger.logInfo(msg);
   }

   private void logWarning(String msg)
   {
      mLogger.logWarning(msg);
   }

   private void logError(String msg, int code)
   {
      mLogger.logError(msg, code);
   }


}
