package edu.mayo.bior.catalog.markdown.transformer.comparison;

import com.google.gson.Gson;
import com.google.gson.GsonBuilder;
import edu.mayo.bior.catalog.markdown.transformer.MarkdownTransformer;
import edu.mayo.bior.catalog.markdown.transformer.TransformException;

import java.io.File;
import java.io.IOException;
import java.util.ArrayList;
import java.util.List;

import static org.apache.commons.io.FileUtils.readFileToString;

/**
 * Transforms markdown content with comparison statistics information.
 */
public class ComparisonStatsImpl implements MarkdownTransformer {

    static final String NOT_AVAILABLE_MESG = "Information is not available";

    private List<MarkdownTransformer> transformers = new ArrayList<MarkdownTransformer>();

    /**
     * Constructor
     * @param comparisonJsonFile JSON file containing comparison stats data.  A NULL value is allowed
     *                           and should be utilized if there's no JSON file.
     */
    public ComparisonStatsImpl(File comparisonJsonFile) throws IOException {
        CatalogDiffStatistics stats = null;
        if (comparisonJsonFile != null) {
            String comparisonJson = readFileToString(comparisonJsonFile);
            stats = parseComparisonStatsJson(comparisonJson);
        }

        transformers.add(new CatalogTableImpl(stats));
        transformers.add(new CatalogVolumeImpl(stats));
        transformers.add(new CatalogRowUpdateImpl(stats));
        transformers.add(new CatalogTop5ColumnUpdateImpl(stats));
        transformers.add(new CatalogColumnChangesImpl(stats));
        transformers.add(new CatalogContentChangesImpl(stats));
    }

    @Override
    public String transform(String markdown) throws TransformException {
        for (MarkdownTransformer markdownTransformer: transformers) {
            markdown = markdownTransformer.transform(markdown);
        }
        return markdown;
    }

    /**
     * Parses JSON into a {@link CatalogDiffStatistics} object.
     * @param json The JSON to be parsed.
     * @return {@link CatalogDiffStatistics} object.
     */
    private CatalogDiffStatistics parseComparisonStatsJson(String json) {
        GsonBuilder builder = new GsonBuilder();
        Gson gson = builder.create();
        return gson.fromJson(json, CatalogDiffStatistics.class);
    }
}
