package edu.mayo.bior.catalog.index;

import java.io.BufferedReader;
import java.io.File;
import java.io.FileReader;
import java.io.IOException;
import java.sql.Connection;
import java.sql.PreparedStatement;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.sql.Statement;
import java.util.ArrayList;
import java.util.Properties;

import edu.mayo.pipes.JSON.lookup.lookupUtils.IndexUtils;
import edu.mayo.pipes.util.index.H2Connection;

public class IndexDatabaseCreator {
	/** Reads a bgzip catalog, getting a key from col and jsonPath, then creating an H2 database/index 
	 * @param bgzipPath  Full path to bgzip catalog file
	 * @param keyCol 1-based column where the json is located
	 * @param jsonPath  The json path to get the id that we will index (ex: "HGNC" for HGNC Id within the Genes catalog)
	 * @param outH2DbPath  The name of the H2 database that will be created
	 * @param isVerbose  If true, print all status messages while building the index 
	 * @throws IOException 
	 * @throws SQLException 
	 * @throws ClassNotFoundException */ 
	public void buildIndexH2(String bgzipPath, int keyCol, String jsonPath, String outH2DbPath, boolean isVerbose) throws SQLException, IOException, ClassNotFoundException {
		Connection dbConn = null;
		File tempTxtOut = null;
		try {
			// Always print this msg so user knows where to find index
			println(String.format("Creating index on key '%s', path '%s'",  jsonPath, outH2DbPath), true);
			
			println("-------------- Building Index --------------", isVerbose);
			
			// First remove the database file
			File h2DbFile = new File(outH2DbPath);
			if(h2DbFile.exists()) {
				println("Deleting file: " + h2DbFile.getCanonicalPath(), isVerbose);
				h2DbFile.delete();
			}
			
		    // NOTE: Indexes are saved to a text file first to avoid the huge memory locking issue
		    // (this occurred when reading from a bgzip file and trying to load directly to memory or a database
		    // but saving to files worked ok).
			
			// Save indexes to text file
			println("Saving indexes to temp text file...", isVerbose);
			//addZipIndexesToDb(bgzipFile, 3, false, "\t", dbConn);
		    tempTxtOut = new File(h2DbFile.getCanonicalFile().getParentFile(), "/tempIndex_" + jsonPath + ".txt");
		    IndexUtils indexUtils = new IndexUtils();
		    Properties props = indexUtils.zipIndexesToTextFile(new File(bgzipPath), "\t", keyCol, jsonPath, tempTxtOut, isVerbose);
	
		    // Throw exception if maxKeyLen is 0, because then it didn't index anything
		    int maxKeyLen = (Integer)(props.get(IndexUtils.IndexBuilderPropKeys.MaxKeyLen));
		    if( 0 == maxKeyLen ) {
		    	throw new IllegalArgumentException("There were no keys indexed!  Check your inputs and try again.");
		    }
		    
		    // Read all data from text and put into H2 database
		    H2Connection h2Conn = new H2Connection(h2DbFile);
		    dbConn = h2Conn.getConn();
		    println("Create database table...", isVerbose);
		    boolean isKeyAnInteger = (Boolean)(props.get(IndexUtils.IndexBuilderPropKeys.IsKeyColAnInt));
		    h2Conn.createTable(isKeyAnInteger, maxKeyLen, dbConn);

		    println("Add rows from text file to database...", isVerbose);
		    textIndexesToDb(dbConn, false, tempTxtOut, isVerbose);

			// Print the database
			//printDatabase(h2DbFile, isKeyAnInteger);

		    // Can create index before adding items, which has several benefits:
		    //  1) The indexing can take a long time, and can fail without warning
		    //  2) You don't get any feedback while it is indexing, whereas if you index first, 
		    //     you can see the progress even though it's a bit slower per line inserted
		    //  3) It takes about the same amount of time overall when < 20 million lines to :
		    // 	   index first and then insert (very fast index, but slower per-line insert)
		    //	   insert, then index at the end (faster per-line insert, but much longer indexing time)
		    // BUT, one major CON to this:
		    //  a) If the file is over 20 million lines, it gets considerably slower to insert each additional
		    //     line and may cause the script to hang around 40 million lines written.
		    println("Creating index on database...", isVerbose);
		    println("  Size of file before H2 index applied: " + h2DbFile.length(), isVerbose);
			h2Conn.createTableIndex(dbConn);

		    if( isVerbose )
		    	printDatabaseHeader(dbConn);
		    println("Num rows in database index: " + countDatabaseRows(h2DbFile), isVerbose);
		    println("Size of index file: " + h2DbFile.length(), isVerbose);
		    println("Done.", isVerbose);
		} finally {
			if(dbConn != null && ! dbConn.isClosed())
				dbConn.close();
			// Remove the temp text file
			if( tempTxtOut != null  &&  tempTxtOut.exists() )
				tempTxtOut.delete();
		}
	}
	
	private static void println(String s, boolean isVerbose) {
		if( isVerbose )
			System.out.println(s);
	}
		
	public static void printDatabaseHeader(Connection dbConn) throws SQLException {
		Statement stmt = null;
		ResultSet rs = null;
		try {
			stmt = dbConn.createStatement();
			rs = stmt.executeQuery("SELECT * FROM Indexer");
	    
			// Print the column names
			int numCols = rs.getMetaData().getColumnCount();
			System.out.println("Column headers:");
			for(int i=1; i <= numCols; i++)
				System.out.println(rs.getMetaData().getColumnName(i)  + " (" + rs.getMetaData().getColumnTypeName(i) + ")");
		} finally {
			if( rs != null )
				rs.close();
			if( stmt != null )
				stmt.close();
		}
	}
	
	/** Prints all rows in the database */
	public static void printDatabase(File h2DbFile) throws SQLException, ClassNotFoundException, IOException {
		System.out.println("\n\nPrinting table Indexer...");
		System.out.println(getTableAsString(h2DbFile));
	}
	
	public static String getTableAsString(File h2DbFile) throws SQLException, IOException {
		ArrayList<ArrayList<String>> table = getDatabaseTable(h2DbFile);
		StringBuilder str = new StringBuilder();
		for(int i=0; i < table.size(); i++) {
			str.append( (i==0 ? "" : i+")") );
			for(String col : table.get(i)) 
				str.append("\t" + col);
			str.append("\n");
		}
		return str.toString();
	}
	
	/** Return database key-pos table (including header row) 
	 * @throws SQLException */
	public static ArrayList<ArrayList<String>> getDatabaseTable(File h2DbFile) throws SQLException, IOException {
		ArrayList<ArrayList<String>> rows = new ArrayList<ArrayList<String>>();
		Connection dbConn = null;
		Statement stmt = null;
		ResultSet rs = null;
		try {
			dbConn = new H2Connection(h2DbFile).getConn();
			stmt = dbConn.createStatement();
			rs = stmt.executeQuery("SELECT * FROM Indexer ORDER BY Key");
	    
			// Print the column names
			int numCols = rs.getMetaData().getColumnCount();
			ArrayList<String> headerRow = new ArrayList<String>();
			for(int i=1; i <= numCols; i++)
				headerRow.add(rs.getMetaData().getColumnName(i));
			rows.add(headerRow);
			
			while(rs.next()) {
				ArrayList<String> row = new ArrayList<String>();
		    	String key = "" + rs.getObject(1);
		    	Long pos = rs.getLong(2);
		    	row.add(key);
		    	row.add("" + pos);
		    	rows.add(row);
		    }
		} finally {
			if( rs != null )
				rs.close();
			if( stmt != null )
				stmt.close();
			if( dbConn != null )
				dbConn.close();
		}
		return rows;
	}

	public static long countDatabaseRows(File h2DbFile) throws SQLException, ClassNotFoundException, IOException {
		Connection dbConn = new H2Connection(h2DbFile).getConn();
	    Statement stmt = dbConn.createStatement();
	    ResultSet rs = stmt.executeQuery("SELECT COUNT(*) FROM Indexer");
	    long count = 0;
	    while(rs.next()) {
	    	count = rs.getInt(1);
	    }
    	//System.out.println("# of rows in database = " + count);
	    rs.close();
	    stmt.close();
	    dbConn.close();
	    return count;
	}

	private void textIndexesToDb(Connection dbConn, boolean isKeyInteger, File tmpTxt, boolean isVerbose) throws NumberFormatException, SQLException, IOException {
		long numObjects = 0;
		long numBytesRead = 0;
		long MB = 1024L * 1024L;

		BufferedReader fin = new BufferedReader(new FileReader(tmpTxt));
		
		final String SQL = "INSERT INTO Indexer (Key, FilePos) VALUES (?, ?)";
		PreparedStatement stmt = dbConn.prepareStatement(SQL);
		dbConn.setAutoCommit(true);

		String line = null;
		while( (line = fin.readLine()) != null ) {
			numObjects++;
			String[] cols = line.split("\t");
			String key = cols[0];
			String pos = cols[1];
			// Key
			if(isKeyInteger)
				stmt.setLong(1, Integer.valueOf(key));
			else
				stmt.setString(1, key);
			// FilePos
			stmt.setLong(2, Long.valueOf(pos));
			stmt.execute();
			dbConn.commit();

			if( isVerbose && numObjects % 10000 == 0 ) {
				System.out.println(key + "    " + numObjects 
						+ ", avgBytesPerItem: " + (numBytesRead / numObjects) 
						+ ", MBs read: " + (numBytesRead/MB) + ", Mem (MBs): " + (new IndexUtils().getMemoryUse()/MB));
			}
		} while( line != null );

		fin.close();
		stmt.close();
		
		println("Num objects read: " + numObjects, isVerbose);
	}

}
