package edu.mayo.bior.buildcatalog;

import java.io.File;
import java.io.IOException;
import java.util.HashMap;
import java.util.List;

import edu.mayo.bior.catalog.Catalog;
import edu.mayo.bior.catalog.CatalogMetadataConstant;
import edu.mayo.bior.util.validation.CatalogFileValidator;
import edu.mayo.pipes.JSON.lookup.lookupUtils.IndexUtils;
import edu.mayo.pipes.history.ColumnMetaData;
import edu.mayo.pipes.history.ColumnMetaDataOperations;
import edu.mayo.bior.catalog.index.IndexDatabaseCreator;

public class IndexesStep implements BuildCatalogStepInterface {

	private BuildInfo mBuildInfo = null;
	private StepLogger mStepLogger = null;

	private List<String> mH2IndexList = null;

	private String mCatalogDirectoryPath = null;
	private String mCatalogFilePath = null;

	private String mColumnInfoFilePath = null;
	private HashMap<String,ColumnMetaData> mColumnInfoMap = null;

	public IndexesStep(BuildInfo buildInfo, StepLogger logger) {
		mBuildInfo = buildInfo;
		mStepLogger = logger;
	}

	public void execute() throws BuildCatalogStepInputException, BuildCatalogStepExecuteException {
		checkInput();
		
		if (mH2IndexList == null || mH2IndexList.size() <= 0) {
			return;
		}
		
		String indexDbPathOut = null;
		for (String columnNameToIndex : mH2IndexList) {
			
			try {
				indexDbPathOut = IndexUtils.getH2DbIndexPath(mCatalogFilePath, columnNameToIndex);
				IndexUtils.createParentDirectories(indexDbPathOut);
			} catch (IOException io) {
				throw new BuildCatalogStepExecuteException("Could not generate index name or create index subdirectory successfully for requested index column name: " + columnNameToIndex);
			}

			try {
				IndexDatabaseCreator indexer = new IndexDatabaseCreator();
				indexer.buildIndexH2(mCatalogFilePath, -1, columnNameToIndex, indexDbPathOut, false); // verbose = false
				removeTraceFile(indexDbPathOut);
			} catch (Exception e) {
				throw new BuildCatalogStepExecuteException("Could not generate index successfully for requested index column name: " + columnNameToIndex);
			}
		}
	}

	/** Remove the debugging trace file that is generated when writing an index file (ex: buildCatalog.allSteps.key.idx.trace.db) */
	private void removeTraceFile(String indexDbPathOut) {
		if( indexDbPathOut.endsWith("idx.h2.db") ) {
			File traceFile = new File(indexDbPathOut.replace(".idx.h2.db", ".idx.trace.db"));
			if( traceFile.exists() ) {
				traceFile.delete();
			}
		}
	}

	public String getLog() throws Exception {
		return null;
	}

	private void init(BuildInfo buildInfo) throws BuildCatalogStepInputException {
		
		mH2IndexList = buildInfo.getIndexes();
	
		mCatalogDirectoryPath = buildInfo.getTargetDirectory();
		mCatalogFilePath = mCatalogDirectoryPath + "/" + buildInfo.getCatalogPrefix() + CatalogMetadataConstant.CATALOG_FILE_SUFFIX;
		
		String catalogFilePrefix = Catalog.getCatalogPrefix(new File(mCatalogFilePath).getName());
		if (catalogFilePrefix == null) {
			throw new BuildCatalogStepInputException("IndexesStep.checkInput(): catalog file not valid suffix [" + mCatalogFilePath + "]. Please check catalog file name.");			
	
		}
		
		mColumnInfoFilePath = mCatalogDirectoryPath + "/" + catalogFilePrefix + CatalogMetadataConstant.COLUMN_INFO_SUFFIX;
		File columnsTsvFile = new File(mColumnInfoFilePath);
		if (columnsTsvFile != null && columnsTsvFile.canRead()) {
			try {
				ColumnMetaDataOperations colMetaOps = new ColumnMetaDataOperations(columnsTsvFile);
				mColumnInfoMap = colMetaOps.load();
			} catch (IOException io) {
				throw new BuildCatalogStepInputException("IndexesStep.checkInput(): Exception while reading in catalog columns.tsv file [" + mColumnInfoFilePath + "]. Please check columns.tsv file.");			
			}
		} else {
			throw new BuildCatalogStepInputException("IndexesStep.checkInput(): Catalog columns.tsv file is not a valid file [" + mColumnInfoFilePath + "]. Please check bior_build_catalog, create-property-files step results.");			
		}
	}

	private void checkInput() throws BuildCatalogStepInputException {
		init(mBuildInfo);
		
		if (mH2IndexList == null || mH2IndexList.size() <= 0) {
			//log("H2 Indexes defined for catalog: " + mCatalogFilePath);
			return;  // no indexes = nothing to do.
		}
		
		File catalogFile = new File(mCatalogFilePath);
		if (!CatalogFileValidator.catalogExists(catalogFile)) {
			throw new BuildCatalogStepInputException("IndexesStep.checkInput(): Catalog file is not a valid file. Please check bior_build_catalog, create-catalog step results.");
		}
		
		if (mColumnInfoMap == null || mColumnInfoMap.size() <= 0) {
			throw new BuildCatalogStepInputException("IndexesStep.checkInput(): Catalog columns.tsv file [" + mColumnInfoFilePath + "] was not read into HashMap correctly. Please check columns.tsv for expected five-column format.");								
		}
		
		for (String columnNameToIndex : mH2IndexList) {
			ColumnMetaData colInfo = mColumnInfoMap.get(columnNameToIndex);
			if (colInfo == null) {
				throw new BuildCatalogStepInputException("IndexesStep.checkInput(): Catalog column to be indexed [" + columnNameToIndex + "] is not found in the catalog's columns.tsv file [" + mColumnInfoFilePath + "]. Please check columns.tsv for expected five-column format.");											
			}
		}
	}
}
