package edu.mayo.bior.util;

import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertTrue;
import static org.junit.Assert.fail;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;

import org.junit.Test;

import edu.mayo.bior.cli.func.BaseFunctionalTest;
import edu.mayo.cli.InvalidDataException;
import edu.mayo.pipes.history.History;

/**
 * JUnit tests for {@linkplain ColumnResolver}
 *
 * <p>@author </p>
 */
public class ColumnResolverTest extends BaseFunctionalTest {

	
	/** 
	 * Get the absolute position of a column (0-based) (no negatives) 
	 * @throws Exception 
	 */
	@Test
	public void getPosIdx() throws Exception {
		History h = new History(new String[] { "A", "B", "C", "D", "E" });
		assertEquals(4 + "",  ColumnResolver.numberToZeroBasedIndex(h, -1, 1) + "");
		assertEquals(4 + "",  ColumnResolver.numberToZeroBasedIndex(h, -1, 1) + "");
		assertEquals(0 + "",  ColumnResolver.numberToZeroBasedIndex(h, -5, 1) + "");
		assertEquals(0 + "",  ColumnResolver.numberToZeroBasedIndex(h, 1, 1) + "");
		assertEquals(4 + "",  ColumnResolver.numberToZeroBasedIndex(h, 5, 1) + "");

		// Errors:
		String errMsg = "Error on line %s.  Specified column index is beyond the range of the columns in the data row.  Specified column:  %s.  Size of data row: %s";
		// -6 is too far negative, 0 is not allowed, 6 is beyond the end of the array
		int[] badCols = new int[] {-6, 0, 6};
		for( int badCol : badCols ) {
			try {
				ColumnResolver.numberToZeroBasedIndex(h, badCol, 1);
				fail("Should throw exception before this!");
			} catch(Exception e) {
				assertTrue(e instanceof IndexOutOfBoundsException);
				assertEquals(String.format(errMsg, 1, badCol, 5), e.getMessage());
			}
		}
		System.out.println("getPosIdx() - DONE.");
	}
	
	
	/**
	 * Test invalid ranges against the header
	 */
	@Test
	public void invalidRanges() {
		String header = concat("A", "B", "C", "D", "E", "F", "G", "H", "I");
		
		//================================================================================
		// Errors
		//================================================================================
		// cols: "1"    	(but header is null) - (ERROR: invalid column - no header present)
		assertException(null, "1", new InvalidDataException("Error: Invalid column - no header found"));

		// cols: "1"    	(but header is "") - (ERROR: invalid column - no header present)
		assertException("", "1", new InvalidDataException("Error: Invalid column - no header found"));

		// cols: "1"    	(but header is "  ") - (ERROR: invalid column - no header present)
		assertException("  ", "1", new InvalidDataException("Error: Invalid column - no header found"));

		// cols: "0"    	(ERROR: invalid column - header contains only columns 1-9)
		assertException(header, "0", new InvalidDataException("Error: Invalid column or range [0] - 0 is not a valid column number.  Column numbers start at 1"));

		//  cols: 10  		(ERROR - invalid column - header contains only columns 1-9)
		assertException(header, "10", new InvalidDataException("Error: Invalid column or range [10] - header only contains 9 columns"));
				
		//	cols: 8..14		(OK if # columns >= 14.  ERROR if < 14)
		assertException(header, "8..14", new InvalidDataException("Error: Invalid column or range [8..14] - header only contains 9 columns"));

		//	cols: 8-14		(ERROR - invalid specification.  Use 8..14)
		assertException(header, "8-14", new InvalidDataException("Error: Invalid column or range [8-14] - range could not be parsed"));

		//	cols: 2-
		assertException(header, "2-", new InvalidDataException("Error: Invalid column or range [2-] - range could not be parsed"));
	}
	
	
	/** 
	 * Verify ranges against columns that are in the header 
	 * @throws Exception 
	 */
	@Test
	public void validRanges() throws Exception {
		String header = concat("A", "B", "C", "D", "E", "F", "G", "H", "I");

		//================================================================================
		// Valid ranges
		//================================================================================
		//	cols: null 		(No columns specified - none will be added to INFO col)
		assertRanges(header, null, 		new ArrayList<Integer>());
		//	cols: "" 		(No columns specified - none will be added to INFO col)
		assertRanges(header, "", 		new ArrayList<Integer>());
	    //	cols: 1	(only the first column)
		assertRanges(header, "1", 		Arrays.asList(1));
		//  cols: -1  (only the last column)
		assertRanges(header, "-1", 		Arrays.asList(9));
		//  cols: 9   (only the last column - as positive integer)
		assertRanges(header, "9", 		Arrays.asList(9));
		assertRanges(header, "1..8", 	Arrays.asList(1,2,3,4,5,6,7,8));
		assertRanges(header, "1..1", 	Arrays.asList(1));
		assertRanges(header, "-1..-1", 	Arrays.asList(9));
		assertRanges(header, "3..1", 	Arrays.asList(1,2,3));
		assertRanges(header, "-1..-3", 	Arrays.asList(7,8,9));
		assertRanges(header, "-3..-1", 	Arrays.asList(7,8,9));
		assertRanges(header, "-2..2",  	Arrays.asList(2,3,4,5,6,7,8));
		assertRanges(header, "..2", 	Arrays.asList(1,2));
		assertRanges(header, "3..", 	Arrays.asList(3,4,5,6,7,8,9));
		assertRanges(header, "..-1", 	Arrays.asList(1,2,3,4,5,6,7,8,9));
		assertRanges(header, "-2..", 	Arrays.asList(8,9));
		assertRanges(header, "4,5,6..9", Arrays.asList(4,5,6,7,8,9));
		//  Add columns 1,2,3,4,-3,-2,-1,6,8..end.  Redundant columns are eliminated)
		assertRanges(header, "..3,-3..-4,1..2,6,8..", 		Arrays.asList(1,2,3,6,7,8,9));
		assertRanges(header, "1,2,3,3,2,1,2,3,1,4,3,1,2", 	Arrays.asList(1,2,3,4));
		assertRanges(header, ",1",  	Arrays.asList(1));
		assertRanges(header, "-2,", 	Arrays.asList(8));
	}
	
	
	// ========================================================================
	
	private void assertException(String header, String ranges,	InvalidDataException invalidDataException) {
		try {
			History history = createHistoryWithHeader(header);
			ColumnResolver.rangesToZeroBasedIndexes(history, ranges);
			fail("An exception was expected!");
		} catch(InvalidDataException e) {
			assertEquals("Exception message did not match!", invalidDataException.getMessage(), e.getMessage());
		} catch(Exception e) {
			fail("Exception did not match!  Expected: " + InvalidDataException.class + ",  got: " + e.getClass());
		}
	}
	
	private void assertRanges(String header, String ranges, List<Integer> colsExpectedOneBased) throws Exception {
		History history = createHistoryWithHeader(header);
		List<Integer> colsExpectedZeroBased = ColumnResolver.rangesToZeroBasedIndexes(history, ranges);
		assertEquals(colsExpectedZeroBased.size(), colsExpectedOneBased.size());
		for(int i=0; i < colsExpectedZeroBased.size(); i++) {
			assertEquals(new Integer(colsExpectedOneBased.get(i)-1), colsExpectedZeroBased.get(i));
		}
	}
}
