package edu.mayo.bior.pipeline.createCatalogProps;

import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertTrue;
import static org.junit.Assert.fail;

import java.io.ByteArrayOutputStream;
import java.io.File;
import java.io.IOException;
import java.io.PrintStream;
import java.util.List;

import org.apache.commons.io.FileUtils;
import org.junit.Before;
import org.junit.Rule;
import org.junit.Test;
import org.junit.rules.TemporaryFolder;

import edu.mayo.bior.cli.func.BaseFunctionalTest;
import edu.mayo.cli.InvalidDataException;
import edu.mayo.pipes.history.ColumnMetaData;

public class ColumnMetaFromCatalogCrawlingTest {

	@Rule
	public TemporaryFolder mTempFolder = new TemporaryFolder();
	
	private File mTempFile;

//	private ByteArrayOutputStream stdoutTemp;
//
//	private ByteArrayOutputStream stderrTemp;
//
//	private PrintStream stdoutOriginal;
//
//	private PrintStream stderrOriginal;
	
	@Before
	public void beforeEach() throws IOException {
		mTempFolder.create();
		
		// Create a small catalog file (tsv) from JSON
		mTempFile = mTempFolder.newFile();
		// Test: dot, null, String, Integer, Float, 
		// Also test possible change in type (such as int to float, or float to int) which are valid; as well as null to string, or dot to string, etc
		// JsonObj - null to JSON obj
		// StringA - null to String
		// StringB - dot to quoted int (string)
		// StringC - quoted int (string) to not present
		// StringArrayA - multi-value array to single value array
		// StringArrayB - empty array to single value array
		// IntA - int to int
		// IntB - negative int to positive int
		// IntC - null to int
		// IntArrayA - multi-value array to single value array with 0
		// IntArrayB - empty array to single value array with 0
		// FloatA - float to 0 (still a float)
		// FloatB - 0 (still a float) to negative float
		// Boolean - false to true
		String json = "{'JsonObj':null,'StringA':null,'StringB':'.','StringC':'1','StringArrayA':['A','B','C'],'StringArrayB':[],'IntA':0,'IntB':-128,'IntC':null,'IntArrayA':[-5,0,5],'IntArrayB':[],'FloatA':0.33,'FloatB':0,'Boolean':false}" + "\n"
					+ "{'JsonObj':{'A':1},'StringA':'jjj','StringB':'0','StringArrayA':[],'StringArrayB':['A'],'IntA':111,'IntB':99999,'IntC':30,'IntArrayA':[],'IntArrayB':[0],'FloatA':0,'FloatB':-999.9,'Boolean':true}";
		FileUtils.write(mTempFile, json);
	}
	
	@Test
	public void testTypeAndCount() throws IOException, InvalidDataException {
		ColumnMetaFromCatalogCrawling crawler = new ColumnMetaFromCatalogCrawling();
		List<ColumnMetaData> colMetaList = crawler.getColumnMetadata(mTempFile.getCanonicalPath(), /*NumProgressHandler=*/null);
		assertEquals(14, colMetaList.size());
		int i=0;
		//          metaObject            key            	Type       Count
		assertMeta(colMetaList.get(i++),  "StringA", 		"String",  "1");
		assertMeta(colMetaList.get(i++),  "StringB", 		"String",  "1");
		assertMeta(colMetaList.get(i++),  "StringC", 		"String",  "1");
		assertMeta(colMetaList.get(i++),  "StringArrayA", 	"String",  ".");
		assertMeta(colMetaList.get(i++),  "StringArrayB", 	"String",  ".");
		assertMeta(colMetaList.get(i++),  "IntA",    		"Integer", "1");
		assertMeta(colMetaList.get(i++),  "IntB", 			"Integer", "1");
		assertMeta(colMetaList.get(i++),  "IntC", 			"Integer", "1");
		assertMeta(colMetaList.get(i++),  "IntArrayA", 		"Integer", ".");
		assertMeta(colMetaList.get(i++),  "IntArrayB", 		"Integer", ".");
		assertMeta(colMetaList.get(i++),  "FloatA", 		"Float",   "1");
		assertMeta(colMetaList.get(i++),  "FloatB", 		"Float",   "1");
		assertMeta(colMetaList.get(i++),  "Boolean", 		"Boolean", "0");
		assertMeta(colMetaList.get(i++),  "JsonObj.A",		"Integer", "1");
	}
	

	@Test
	/** With a mix of floats and ints, the type should be float */
	public void testTypeWhenMixOfIntAndFloat() throws IOException, InvalidDataException {
		String json = "{'AN':1}\n"
					+ "{'AN':2.1}\n"
					+ "{'AN':3}";
		FileUtils.write(mTempFile, json);
		ColumnMetaFromCatalogCrawling crawler = new ColumnMetaFromCatalogCrawling();
		List<ColumnMetaData> colMetaList = crawler.getColumnMetadata(mTempFile.getCanonicalPath(), /*NumProgressHandler=*/null);
		assertEquals(1, colMetaList.size());
		int i=0;
		//          metaObject            key            	Type       Count
		assertMeta(colMetaList.get(i++),  "AN", 			"Float",   "1");
	}

	@Test
	/** For some reason, when there was a mix of int and quoted-int values, it chose Integer as type instead of string.
	 *  It should instead throw an InvalidDataException due to incompatible types */
	public void testTypeWhenMixOfStringAndInt() throws IOException, InvalidDataException {
		String json = "{'AN':1}\n"
					+ "{'AN':'2'}\n"
					+ "{'AN':3}";
		FileUtils.write(mTempFile, json);
		ColumnMetaFromCatalogCrawling crawler = new ColumnMetaFromCatalogCrawling();
		try {
			List<ColumnMetaData> colMetaList = crawler.getColumnMetadata(mTempFile.getCanonicalPath(), /*NumProgressHandler=*/null);
			fail("ERROR: This should have thrown an InvalidDataException");
		} catch(Exception e) {
			assertEquals("ERROR: Incompatible type change for key [AN] from [Integer] to [String] on data line 2", e.getMessage());
			assertTrue(e instanceof InvalidDataException);
		}
	}

	@Test
	/** With a mix of floats and ints and strings, this should throw an exception for bad type conversion */
	public void testTypeWhenMixOfIntAndFloatAndString() throws IOException, InvalidDataException {
		String json = "{'AN':1}\n"
					+ "{'AN':2.0}\n"
					+ "{'AN':'3'}";
		FileUtils.write(mTempFile, json);
		ColumnMetaFromCatalogCrawling crawler = new ColumnMetaFromCatalogCrawling();
		try {
			List<ColumnMetaData> colMetaList = crawler.getColumnMetadata(mTempFile.getCanonicalPath(), /*NumProgressHandler=*/null);
			fail("ERROR: This should have thrown an InvalidDataException");
		} catch(Exception e) {
			assertEquals("ERROR: Incompatible type change for key [AN] from [Float] to [String] on data line 3", e.getMessage());
			assertTrue(e instanceof InvalidDataException);
		}
	}

	@Test
	/** ERROR when going from string to array */
	public void testError_count_stringToArray() throws IOException, InvalidDataException {
		String json = "{'AN':'A'}\n"
					+ "{'AN':['A']}\n"
					+ "{'AN':['A','B','C']}";
		FileUtils.write(mTempFile, json);
		ColumnMetaFromCatalogCrawling crawler = new ColumnMetaFromCatalogCrawling();
		try {
			List<ColumnMetaData> colMetaList = crawler.getColumnMetadata(mTempFile.getCanonicalPath(), /*NumProgressHandler=*/null);
			fail("ERROR: This should have thrown an InvalidDataException");
		} catch(Exception e) {
			assertEquals("ERROR: Incompatible count change for key [AN] from [1] to [.] on data line 2", e.getMessage());
			assertTrue(e instanceof InvalidDataException);
		}
	}

	@Test
	/** ERROR when going from array to string */
	public void testError_count_arrayToString() throws IOException, InvalidDataException {
		String json = "{'AN':['A']}\n"
					+ "{'AN':'A'}";
		FileUtils.write(mTempFile, json);
		ColumnMetaFromCatalogCrawling crawler = new ColumnMetaFromCatalogCrawling();
		try {
			List<ColumnMetaData> colMetaList = crawler.getColumnMetadata(mTempFile.getCanonicalPath(), /*NumProgressHandler=*/null);
			fail("ERROR: This should have thrown an InvalidDataException");
		} catch(Exception e) {
			assertEquals("ERROR: Incompatible count change for key [AN] from [.] to [1] on data line 2", e.getMessage());
			assertTrue(e instanceof InvalidDataException);
		}
	}
	
	@Test
	/** ERROR when going from float to array */
	public void testError_count_floatToArray() throws IOException, InvalidDataException {
		String json = "{'AN':2}\n"
					+ "{'AN':3.11}\n"
					+ "{'AN':[2,3.8,4.9]}";
		FileUtils.write(mTempFile, json);
		ColumnMetaFromCatalogCrawling crawler = new ColumnMetaFromCatalogCrawling();
		try {
			List<ColumnMetaData> colMetaList = crawler.getColumnMetadata(mTempFile.getCanonicalPath(), /*NumProgressHandler=*/null);
			fail("ERROR: This should have thrown an InvalidDataException");
		} catch(Exception e) {
			assertEquals("ERROR: Incompatible count change for key [AN] from [1] to [.] on data line 3", e.getMessage());
			assertTrue(e instanceof InvalidDataException);
		}
	}
	
	
	@Test
	/** With a mix of JSON objects and nulls and empty json arrays - this should be allowed */
	public void testOk_TypeWhenMixOfJsonAndNulls() throws IOException, InvalidDataException {
		//            Test mix of JSON and null
		String json = "{'AN':null}\n"
					+ "{'AN':{'a':1}}\n"
					+ "{'AN':{'a':2}}\n"
					+ "{'AN':{'a':null}}\n"
					+ "{'AN':{'a':4.9}}\n"
					+ "{'AN':null}\n"
					// Test mix of arrays containing JSON and nulls and empty array - starting with a JsonObject then to null
					+ "{'aaa':[{'b':2}]}\n"
					+ "{'aaa':[]}\n"
					+ "{'aaa':[null]}\n"
					+ "{'aaa':[null,{'b':3}]}\n"
					+ "{'aaa':[{'b':3},null]}\n"
					+ "{'aaa':[null,null,{'b':3},null]}\n"
					// Test null to empty-array to JsonObject
					+ "{'bbb':[null]}\n"
					+ "{'bbb':[]}\n"
					+ "{'bbb':[{'c':{'d':'e'}}]}\n"
					;
		FileUtils.write(mTempFile, json);
		ColumnMetaFromCatalogCrawling crawler = new ColumnMetaFromCatalogCrawling();
		List<ColumnMetaData> colMetaList = crawler.getColumnMetadata(mTempFile.getCanonicalPath(), /*NumProgressHandler=*/null);
		assertEquals(3, colMetaList.size());
		int i=0;
		//          metaObject            key            	Type       Count
		// "AN" by itself is null, so ignored
		assertMeta(colMetaList.get(i++),  "AN.a", 			"Float",   "1");
		assertMeta(colMetaList.get(i++),  "aaa", 			"JSON",    ".");
		assertMeta(colMetaList.get(i++),  "bbb", 			"JSON",    ".");
	}


	@Test
	/** Json array is null for first several elements, and then Float */
	public void testOk_TypeNullFirstThenFloat() throws IOException, InvalidDataException {
		String json = "{'AN':[null,null,null,3.14159]}\n";
		FileUtils.write(mTempFile, json);
		ColumnMetaFromCatalogCrawling crawler = new ColumnMetaFromCatalogCrawling();
		List<ColumnMetaData> colMetaList = crawler.getColumnMetadata(mTempFile.getCanonicalPath(), /*NumProgressHandler=*/null);
		assertEquals(1, colMetaList.size());
		//          metaObject            key            	Type       Count
		assertMeta(colMetaList.get(0),    "AN", 			"Float",    ".");
	}

//	private void startStderrCapture() {
//		this.stdoutTemp = new ByteArrayOutputStream();
//		this.stderrTemp = new ByteArrayOutputStream();
//		this.stdoutOriginal = System.out;
//		this.stderrOriginal = System.err;
//		System.setOut(new PrintStream(this.stdoutTemp));
//		System.setErr(new PrintStream(this.stderrTemp));
//	}
//
//	private void stopStderrCapture() {
//		System.setOut(this.stdoutOriginal);
//		System.setErr(this.stderrOriginal);
//	}


	private void assertMeta(ColumnMetaData columnMetaData, String expectedKey, String expectedType, String expectedCount) {
		assertEquals(expectedKey,   columnMetaData.getColumnName());
		assertEquals(expectedType,  columnMetaData.getType().toString());
		assertEquals(expectedCount, columnMetaData.getCount());
	}
}
