package edu.mayo.bior.pipeline;

import static org.junit.Assert.assertEquals;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.HashMap;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.TreeSet;

import org.junit.Test;

import com.google.gson.JsonArray;
import com.google.gson.JsonObject;
import com.google.gson.JsonParser;

import edu.mayo.bior.cli.func.BaseFunctionalTest;
import edu.mayo.bior.pipeline.InfoMetaObject.KeyValType;
import edu.mayo.pipes.history.ColumnMetaData;
import edu.mayo.pipes.history.ColumnMetaData.Type;
import edu.mayo.pipes.history.History;
import edu.mayo.pipes.history.HistoryMetaData;
import edu.mayo.pipes.util.metadata.AddMetadataLines.BiorMetaControlledVocabulary;
import edu.mayo.pipes.util.test.PipeTestUtils;


/** Inserts data into the INFO column of the VCF */
public class VcfInfoColumnBuilderTest {
	
	enum MikeType { THIS, THAT, OTHER };

	
	@Test
	/** Test a single value within a JSON object */
	public void testJsonOneValToAdd() {
		List<String> columnNames = Arrays.asList("CHROM", "POS", "ID", "REF", "ALT", "QUAL", "FILTER", "INFO", "bior.dbSnp");
		History lineIn = new History( Arrays.asList("1",  "100", "rs1","A",   "C",   "0.1",  "PASS",   "AF=12","{'AC':3}" ) );
		String expectedInfo = "bior.dbSnp.AC=3";
		assertInfoBuilder(columnNames, lineIn, Arrays.asList(8), expectedInfo);
	}
	
	@Test
	public void testBooleanNotInJson() {
		List<String> columnNames = Arrays.asList("CHROM", "POS", "ID", "REF", "ALT", "QUAL", "FILTER", "INFO", "bior.isInDbSnp");
		History lineIn = new History( Arrays.asList("1",  "100", "rs1", "A",  "C",   "0.1",  "PASS",   "AF=12", "true" ) );
		String expectedInfo = "bior.isInDbSnp=true";
		assertInfoBuilder(columnNames, lineIn, Arrays.asList(8), expectedInfo);
	}

	@Test
	public void testInfoJsonToKeyValPairs_noVals() {
		List<String> columnNames = Arrays.asList("CHROM", "POS", "ID", "REF", "ALT", "QUAL", "FILTER", "INFO", "bior.dbSnp");
		History lineIn = new History( Arrays.asList("1", "100", "rs1", "A", "C", "0.1", "PASS", "AF=12;AN=1.43", "{}" ) );
		String expectedInfo = "";
		assertInfoBuilder(columnNames, lineIn, Arrays.asList(8), expectedInfo);
	}
	
	
	@Test
	/** Test the creation of INFO column data from a single line that contains several different types of values (some within JSON)
	 *  Also tests JsonArray as well as a dot within a column	 */
	public void testColsToInfo() {
		List<String> columnNames = Arrays.asList("CHROM", "Min", "Max", "BiorJson", "INFO.Name", "INFO.Count");
		History lineIn = new History(Arrays.asList("1", "10109", "10109", "{'_landmark':'1','_minBP':10109,'_maxBP':10109,'_refAllele':'A','_altAlleles':['C','T'],'_id':'rs123','QUAL':139.23}".replaceAll("'", "\""), ".", "2791"));
		String expectedInfo = "BiorJson._altAlleles=C,T;BiorJson._id=rs123;BiorJson._landmark=1;BiorJson._maxBP=10109;BiorJson._minBP=10109;BiorJson._refAllele=A;BiorJson.QUAL=139.23;CHROM=1;INFO.Count=2791;Max=10109;Min=10109";
		assertInfoBuilder(columnNames, lineIn, /*zeroBasedColsToAddToInfo=*/Arrays.asList(0,1,2,3,4,5), expectedInfo);
	}

	
	@Test
	public void testJsonObj() {
		List<String> columnNames = Arrays.asList("CHROM", "POS", "ID", "REF", "ALT", "QUAL", "FILTER", "INFO", "bior.dbSnp");
		History lineIn = new History( Arrays.asList("1", "100", "rs1", "A", "C", "0.1", "PASS", ".", 
				"{'AC':3,'MAF':0.05,'NA':[2.0,3.0,4.0],'Gene':'BRCA1','isInDbsnp':true,'isIn1000Genomes':false,'nested':{'key1':'val1','key2':'val2'}}") );
		String expectedInfo = "bior.dbSnp.AC=3;bior.dbSnp.Gene=BRCA1;bior.dbSnp.isInDbsnp;bior.dbSnp.MAF=0.05;bior.dbSnp.NA=2.0,3.0,4.0;bior.dbSnp.nested.key1=val1;bior.dbSnp.nested.key2=val2";
		assertInfoBuilder(columnNames, lineIn, Arrays.asList(8), expectedInfo);
	}

	@Test
	public void testJsonObj_headerIsEmpty() {
		List<String> columnNames = Arrays.asList("CHROM", "POS", "ID", "REF", "ALT", "QUAL", "FILTER", "INFO", "");
		History lineIn = new History( Arrays.asList("1", "100", "rs1", "A", "C", "0.1", "PASS", ".", 
				"{'AC':3,'MAF':0.05,'NA':[2.0,3.0,4.0],'Gene':'BRCA1','isInDbsnp':true,'isIn1000Genomes':false,'nested':{'key1':'val1','key2':'val2'}}") );
		String expectedInfo = "AC=3;Gene=BRCA1;isInDbsnp;MAF=0.05;NA=2.0,3.0,4.0;nested.key1=val1;nested.key2=val2";
		assertInfoBuilder(columnNames, lineIn, Arrays.asList(8), expectedInfo);
	}

	@Test
	/** Test a single-string value (ex: columnHeader:"isInDbsnp" with value "true") that has a BioR header that is of type "Flag" or "Boolean" */
	public void testBooleanSingleStringButFlagInBiorHeader() {
		HistoryMetaData meta = new HistoryMetaData(Arrays.asList("##BIOR=<ID=isInDbSnp,Type=String>"));
		meta.getColumns().add(new ColumnMetaData("isInDbSnp", Type.Boolean, "1", "(no description)"));
		History lineIn = new History("true");
		lineIn.setMetaData(meta);
		
		VcfInfoColumnBuilder infoBuilder = new VcfInfoColumnBuilder();
		List<InfoMetaObject> infoMetaList = new ArrayList<InfoMetaObject>();
		infoMetaList.addAll(infoBuilder.getColumnAsInfoMetaObjects(lineIn, 0));
		String infoActual = infoBuilder.getInfoMetaObjsAsString(infoMetaList);

		assertEquals("isInDbSnp", infoActual);
	}
	
	/** Build the INFO column from JSON or string, merged with the original INFO column data
	 * @param expected  Ex: AC=3;MAF=0.05
	 * @param columnIn  Ex: {"AC":3,"MAF":0.05}
	 * @param infoOriginal  Ex: NA=2;Gene=BRCA1
	 */
	private void assertInfoBuilder(List<String> columnNames, History line, List<Integer> zeroBasedColumnsToAddToInfoList, String expectedInfoCol) {
		HistoryMetaData historyMetaData = new HistoryMetaData(columnNames);
		List<ColumnMetaData> colMeta = historyMetaData.getColumns();
		line.setMetaData(historyMetaData);
		for(String colName : columnNames) {
			colMeta.add(new ColumnMetaData(colName, Type.String, "1", "Default description"));
		}
		
		VcfInfoColumnBuilder infoBuilder = new VcfInfoColumnBuilder();
		TreeSet<Integer> zeroBasedColsToAddToInfo_Set = createTreeSetReverse(zeroBasedColumnsToAddToInfoList.toArray(new Integer[0]));
		List<String> infoMeta = infoBuilder.createInfoMetadata(Arrays.asList(line), zeroBasedColsToAddToInfo_Set);
		String infoData = infoBuilder.createInfoData(line, zeroBasedColsToAddToInfo_Set);
		assertEquals(expectedInfoCol, infoData);
	}
	
	@Test
	public void testKeyInMeta() {
		String meta = "##BIOR=<ID=\"bior.dbSNP_142_GRCh37p13\",Operation=\"bior_lookup\",Nothing=\"\",DataType=\"Integer\">";
		VcfInfoColumnBuilder info = new VcfInfoColumnBuilder();
		Map<String, LinkedHashMap<String,String>>  biorHeaders = info.getBiorHeaderIdMap(new HistoryMetaData(Arrays.asList(meta)));
		final String ID = "bior.dbSNP_142_GRCh37p13";
		assertEquals("bior.dbSNP_142_GRCh37p13", biorHeaders.get(ID).get("ID"));
		assertEquals("bior_lookup", biorHeaders.get(ID).get("Operation"));
		assertEquals("", biorHeaders.get(ID).get("Nothing"));
		assertEquals(null, biorHeaders.get(ID).get("NotFound"));
	}


	@Test
	public void testType() {
		VcfInfoColumnBuilder info = new VcfInfoColumnBuilder();
		String json = (
					   "{'str':'abcd','flag':true,'int':123,'int2':123.0,'int3':'123','int4':'123.0','float':123.8,'float2':'123.9',"
					 + "'arrayNum':[1,2,3],'arrayFloat1':[1.0,1.2],'arrayFloat2':['1.0','1.2'],'arrayStr':['a','b','c'],'arrayMix':['a','10','10.0','11.3'],"
					 + "'obj':{'name':'bob'}}"
					  ).replaceAll("'", "\"");
		JsonObject jsonObj = (JsonObject)(new JsonParser().parse(json));

		
		assertEquals(KeyValType.String, 	info.getType(jsonObj.get("str")));
		assertEquals(KeyValType.Flag, 		info.getType(jsonObj.get("flag")));
		assertEquals(KeyValType.Integer, 	info.getType(jsonObj.get("int")));
		assertEquals(KeyValType.Integer, 	info.getType(jsonObj.get("int2")));
		assertEquals(KeyValType.String, 	info.getType(jsonObj.get("int3")));
		assertEquals(KeyValType.String, 	info.getType(jsonObj.get("int4")));
		assertEquals(KeyValType.Float, 		info.getType(jsonObj.get("float")));
		assertEquals(KeyValType.String,	 	info.getType(jsonObj.get("float2")));
		
		// All arrays and objects will be treated as Strings by this method
		// The code has to look at the type and determine what method to call
		assertEquals(KeyValType.String, 	info.getType(jsonObj.get("arrayNum")));
		assertEquals(KeyValType.String, 	info.getType(jsonObj.get("arrayFloat1")));
		assertEquals(KeyValType.String,	 	info.getType(jsonObj.get("arrayFloat2")));
		assertEquals(KeyValType.String, 	info.getType(jsonObj.get("arrayStr")));
		assertEquals(KeyValType.String, 	info.getType(jsonObj.get("arrayMix")));
		assertEquals(KeyValType.String, 	info.getType(jsonObj.get("obj")));
		
		assertEquals(KeyValType.Integer, 	info.getTypeArray((JsonArray)(jsonObj.get("arrayNum"))));
		assertEquals(KeyValType.Float,	 	info.getTypeArray((JsonArray)(jsonObj.get("arrayFloat1"))));
		assertEquals(KeyValType.String,	 	info.getTypeArray((JsonArray)(jsonObj.get("arrayFloat2"))));
		assertEquals(KeyValType.String, 	info.getTypeArray((JsonArray)(jsonObj.get("arrayStr"))));
		assertEquals(KeyValType.String, 	info.getTypeArray((JsonArray)(jsonObj.get("arrayMix"))));
	}
	

	private List<String> getMetaFromLine(History dataLine) {
		List<String> metaList = new ArrayList<String>();
		VcfInfoColumnBuilder infoBuilder = new VcfInfoColumnBuilder();
		for(int i=0; i < dataLine.size(); i++) {
			List<InfoMetaObject> meta = infoBuilder.getColumnAsInfoMetaObjects(dataLine, i);
			for(int j=0; j < meta.size(); j++) {
				metaList.add(meta.get(j).toString());
			}
		}
		return metaList;
	}

	@Test
	/** Test the InfoMetaObjects created by each column.  
	 *  Test the data that would go into the INFO column.
	 *  Also test whether column descriptions are loaded from the catalog's xx.columns.tsv file */ 
	public void testMetaAndData() {
		List<String> headers = new ArrayList<String>(Arrays.asList(
				"##fileFormat=4.2.0",
				"##BIOR=<ID='bior.dbSNP_142_GRCh37p13',Operation='bior_lookup',Nothing='',DataType='JSON',Path='./src/test/resources/metadata/00-All_GRCh37.tsv.bgz'>".replaceAll("'", "\""),
				concat("#chrom", "min", "max", "bior.dbSNP_142_GRCh37p13")
				));
		List<History> dataLines = Arrays.asList(
				new History(Arrays.asList("1", "10108", "10108", "{'CHROM':'1','POS':10108,'QUAL':0.3,'INFO':{'ASP':true,'RS':62651026,'GENEINFO':'NOC2L:26155'},'_id':'rs62651026','_altAlleles':['A','G']}".replaceAll("'", "\""))),
				new History(Arrays.asList("1", "10109", "10109", "{'CHROM':'1','POS':10109,'QUAL':'.','INFO':{'ASP':false,'RS':376007522,'RSPOS':10109,'Mike':123},'_id':'rs376007522','_altAlleles':['A','G','T']}".replaceAll("'",  "\"")))
				);
				
		// Set the headers in the dataLines History objects
		HistoryMetaData meta = new HistoryMetaData(headers);
		dataLines.get(0).setMetaData(meta);
		//dataLines.get(1).setMetaData(meta);
		
		// Set the column header meta
		List<ColumnMetaData> colMeta = new ArrayList<ColumnMetaData>( Arrays.asList(
				new ColumnMetaData("chrom", Type.String,  "1", "Chromosome"),
				new ColumnMetaData("min",   Type.Integer, "1", "Minimum base pair"),
				new ColumnMetaData("max",   Type.Integer, "1", "Maximum base pair"),
				new ColumnMetaData("bior.dbSNP_142_GRCh37p13", Type.JSON, "1", "dbSNP 142 GRCh37.p13 build - from BioR dbSNP catalog")
				) );
		dataLines.get(0).getMetaData().getColumns().addAll(colMeta);
		//dataLines.get(1).getMetaData().getColumns().addAll(colMeta);
		
				
		List<String> expectedHeader = Arrays.asList(
				"##INFO=<ID=bior.dbSNP_142_GRCh37p13._altAlleles,Number=.,Type=String,Description=\"One or more alternate alleles (non-reference) in a JSON array (basically a comma-separated list) (BioR field)\">",
				// "_id" not specified in test columns.tsv
				"##INFO=<ID=bior.dbSNP_142_GRCh37p13._id,Number=1,Type=String,Description=\"\">",
				"##INFO=<ID=bior.dbSNP_142_GRCh37p13.CHROM,Number=1,Type=String,Description=\"Chromosome\">",
				"##INFO=<ID=bior.dbSNP_142_GRCh37p13.INFO.ASP,Number=1,Type=Flag,Description=\"Is Assembly specific. This is set if the variant only maps to one assembly\">",
				"##INFO=<ID=bior.dbSNP_142_GRCh37p13.INFO.GENEINFO,Number=1,Type=String,Description=\"Pairs each of gene symbol:gene id.  The gene symbol and id are delimited by a colon (:) and each pair is delimited by a vertical bar (|)\">",
				// "Mike" should have NO entry in the catalog, so it should have no description
				"##INFO=<ID=bior.dbSNP_142_GRCh37p13.INFO.Mike,Number=1,Type=Integer,Description=\"\">",
				// "INFO.RS" is not in the test columns.tsv
				"##INFO=<ID=bior.dbSNP_142_GRCh37p13.INFO.RS,Number=1,Type=Integer,Description=\"\">",
				"##INFO=<ID=bior.dbSNP_142_GRCh37p13.INFO.RSPOS,Number=1,Type=Integer,Description=\"Chromosome position reported in dbSNP\">",
				"##INFO=<ID=bior.dbSNP_142_GRCh37p13.POS,Number=1,Type=Integer,Description=\"Starting position on the chromosome (1-based)\">",
				"##INFO=<ID=bior.dbSNP_142_GRCh37p13.QUAL,Number=1,Type=Integer,Description=\"Phred-scaled quality score\">",
				"##INFO=<ID=chrom,Number=1,Type=String,Description=\"Chromosome\">",
				"##INFO=<ID=max,Number=1,Type=Integer,Description=\"Maximum base pair\">",
				"##INFO=<ID=min,Number=1,Type=Integer,Description=\"Minimum base pair\">"
				);
		
		// Verify metadata
		VcfInfoColumnBuilder info = new VcfInfoColumnBuilder();
		TreeSet<Integer> colsSet = createTreeSetReverse(0, 1, 2, 3);
		List<String> infoMetaHeaders = info.createInfoMetadata(dataLines, colsSet);
		assertEquals(expectedHeader.size(), infoMetaHeaders.size());
		for(int i=0; i < expectedHeader.size(); i++) {
			System.out.println("----------------------------");
			System.out.println("Exp: " + expectedHeader.get(i));
			System.out.println("Act: " + infoMetaHeaders.get(i));
			assertEquals(i + ") ", expectedHeader.get(i), infoMetaHeaders.get(i));
		}
		
		
		// Verify data
		String infoExpected1 = 
				  "bior.dbSNP_142_GRCh37p13._altAlleles=A,G;"
				+ "bior.dbSNP_142_GRCh37p13._id=rs62651026;"
				+ "bior.dbSNP_142_GRCh37p13.CHROM=1;"
				+ "bior.dbSNP_142_GRCh37p13.INFO.ASP;"
				+ "bior.dbSNP_142_GRCh37p13.INFO.GENEINFO=NOC2L:26155;"
				+ "bior.dbSNP_142_GRCh37p13.INFO.RS=62651026;"
				+ "bior.dbSNP_142_GRCh37p13.POS=10108;"
				+ "bior.dbSNP_142_GRCh37p13.QUAL=0.3;"
				+ "chrom=1;max=10108;min=10108";

		String infoExpected2 = 
				  "bior.dbSNP_142_GRCh37p13._altAlleles=A,G,T;"
				+ "bior.dbSNP_142_GRCh37p13._id=rs376007522;"
				+ "bior.dbSNP_142_GRCh37p13.CHROM=1;"
				+ "bior.dbSNP_142_GRCh37p13.INFO.Mike=123;"
				+ "bior.dbSNP_142_GRCh37p13.INFO.RS=376007522;"
				+ "bior.dbSNP_142_GRCh37p13.INFO.RSPOS=10109;"
				+ "bior.dbSNP_142_GRCh37p13.POS=10109;"
				+ "bior.dbSNP_142_GRCh37p13.QUAL=.;"
				+ "chrom=1;max=10109;min=10109";
		
		String infoActual1 = info.createInfoData(dataLines.get(0), colsSet);
		String infoActual2 = info.createInfoData(dataLines.get(1), colsSet);
		assertEquals(infoExpected1, infoActual1);
		assertEquals(infoExpected2, infoActual2);
	}
	
	/** Return a TreeSet of integers in reverse order */
	private TreeSet<Integer> createTreeSetReverse(Integer... ints) {
		TreeSet<Integer> set = new TreeSet<Integer>(Collections.reverseOrder());
		set.addAll(Arrays.asList(ints));
		return set;
	}
	
	@Test
	/** Test the different instances of updating the InfoMetaObject based on ##BIOR headers:
	 *  If ##BIOR line not found, then do nothing and just return (description should be "")
	 *     If ##BIOR line found, but no catalog path, then update description to that of ##BIOR line along with key at end
	 *     If ##BIOR line found AND has catalog path, then lookup key in columns.tsv:
	 *     		If key is found there, then update its type, number, description from columns.tsv row
	 *     		(If key not there, then do nothing - description should stay as that of desc from ##BIOR line with key at end) */
	public void testUpdateInfoMetaObjFromBiorHeaderLine() {
		VcfInfoColumnBuilder infoBuilder = new VcfInfoColumnBuilder();
		
		String biorHeaderLine = "##BIOR=<ID='bior.dbSNP_142_GRCh37p13',Operation='bior_overlap',DataType='JSON',ShortUniqueName='dbSNP_142_GRCh37p13',Source='dbSNP',Description='NCBI dbSNP Variant Database',Version='142',Build='GRCh37.p13',Path='./src/test/resources/metadata/00-All_GRCh37.tsv.bgz'>".replaceAll("'", "\"");
		Map<String,LinkedHashMap<String,String>> biorHeaderMapFull   = infoBuilder.getBiorHeaderIdMap(new HistoryMetaData(Arrays.asList(biorHeaderLine)));
		Map<String,LinkedHashMap<String,String>> biorHeaderMapBadPathKey = infoBuilder.getBiorHeaderIdMap(new HistoryMetaData(Arrays.asList(biorHeaderLine.replace("Path=", "NoPath="))));
		Map<String,LinkedHashMap<String,String>> biorHeaderMapBadPathKeyNoDesc = infoBuilder.getBiorHeaderIdMap(new HistoryMetaData(Arrays.asList(biorHeaderLine.replace("Path=", "NoPath=").replace("Description","BadDescription"))));
		Map<String,LinkedHashMap<String,String>> biorHeaderMapPathNotFound = infoBuilder.getBiorHeaderIdMap(new HistoryMetaData(Arrays.asList(biorHeaderLine.replace("Path=\"./src", "Path=\"./badDir"))));
		Map<String,LinkedHashMap<String,String>> biorHeaderMapNoColumnsTsv   = infoBuilder.getBiorHeaderIdMap(new HistoryMetaData(Arrays.asList(biorHeaderLine.replace("00-All_GRCh37.tsv.bgz", "00.tsv.bgz"))));

		String biorHeaderLineKey = biorHeaderLine.replace("bior.dbSNP_142_GRCh37p13", "bior.dbSNP_142_GRCh37p13.INFO.dbSNPBuildID").replace("bior_overlap", "bior_drill").replace(",DataType", ",Field=\"INFO.dbSNPBuildID\",DataType");
		Map<String,LinkedHashMap<String,String>> biorHeaderMapFullPreDrilled   = infoBuilder.getBiorHeaderIdMap(new HistoryMetaData(Arrays.asList(biorHeaderLine)));

		
		// This one has NO tie to ##BIOR headers, and is a column with no JSON
		InfoMetaObject infoMetaObj = new InfoMetaObject("", Arrays.asList(""), "key1", KeyValType.String, "1", "");
		infoBuilder.updateInfoMetaObjFromColumnsTsvInBiorHeaderLine(infoMetaObj, biorHeaderMapFull);
		assertEquals("##INFO=<ID=key1,Number=1,Type=String,Description=\"\">", infoMetaObj.toString());

		// This one has a ##BIOR header, but no catalog path, and no description
		infoMetaObj.jsonKey = "key1";
		infoMetaObj.parentColumnName = "bior.dbSNP_142_GRCh37p13";
		infoBuilder.updateInfoMetaObjFromColumnsTsvInBiorHeaderLine(infoMetaObj, biorHeaderMapBadPathKeyNoDesc);
		assertEquals("##INFO=<ID=bior.dbSNP_142_GRCh37p13.key1,Number=1,Type=String,Description=\"\">", infoMetaObj.toString());

		// This one has a ##BIOR header, but no catalog path, but has description
		infoMetaObj.jsonKey = "key1";
		infoMetaObj.parentColumnName = "bior.dbSNP_142_GRCh37p13";
		infoBuilder.updateInfoMetaObjFromColumnsTsvInBiorHeaderLine(infoMetaObj, biorHeaderMapBadPathKey);
		assertEquals("##INFO=<ID=bior.dbSNP_142_GRCh37p13.key1,Number=1,Type=String,Description=\"\">", infoMetaObj.toString());

		// This one has a ##BIOR header, and the PATH defined to the catalog, but the columns.tsv file is not found
		infoMetaObj.jsonKey = "key1";
		infoMetaObj.parentColumnName = "bior.dbSNP_142_GRCh37p13";
		infoBuilder.updateInfoMetaObjFromColumnsTsvInBiorHeaderLine(infoMetaObj, biorHeaderMapPathNotFound);
		assertEquals("##INFO=<ID=bior.dbSNP_142_GRCh37p13.key1,Number=1,Type=String,Description=\"\">", infoMetaObj.toString());

		// This one has a ##BIOR header, and the PATH defined to the catalog, and the file is found, and the columns.tsv is found, but the key is not in columns.tsv
		infoMetaObj.jsonKey = "key1";
		infoMetaObj.parentColumnName = "bior.dbSNP_142_GRCh37p13";
		infoBuilder.updateInfoMetaObjFromColumnsTsvInBiorHeaderLine(infoMetaObj, biorHeaderMapFull);
		assertEquals("##INFO=<ID=bior.dbSNP_142_GRCh37p13.key1,Number=1,Type=String,Description=\"\">", infoMetaObj.toString());

		// This one has a ##BIOR header, and the PATH defined to the catalog, and the file is found, and the columns.tsv is found, and the key is in columns.tsv
		// Line from columns.tsv:
		//		INFO.dbSNPBuildID	Integer	1	First dbSNP build for RS
		infoMetaObj.jsonKey = "INFO.dbSNPBuildID";
		infoMetaObj.parentColumnName = "bior.dbSNP_142_GRCh37p13";
		infoBuilder.updateInfoMetaObjFromColumnsTsvInBiorHeaderLine(infoMetaObj, biorHeaderMapFull);
		assertEquals("##INFO=<ID=bior.dbSNP_142_GRCh37p13.INFO.dbSNPBuildID,Number=1,Type=Integer,Description=\"First dbSNP build for RS\">", infoMetaObj.toString());

		// This one has a ##BIOR header, and the PATH defined to the catalog, and the file is found, and the columns.tsv is found, and the key is in columns.tsv
		// NOTE: This has only the column header, but no JSON within the column, so it should still match to a ##BIOR header
		// For example, if this column had previously been drilled from a BioR catalog column (JSON), then it should still have a catalog path
		// Line from columns.tsv:
		//		INFO.dbSNPBuildID	Integer	1	First dbSNP build for RS
		infoMetaObj.jsonKey = "";
		infoMetaObj.parentColumnName = "bior.dbSNP_142_GRCh37p13.INFO.dbSNPBuildID";
		infoBuilder.updateInfoMetaObjFromColumnsTsvInBiorHeaderLine(infoMetaObj, biorHeaderMapFullPreDrilled);
		assertEquals("##INFO=<ID=bior.dbSNP_142_GRCh37p13.INFO.dbSNPBuildID,Number=1,Type=Integer,Description=\"First dbSNP build for RS\">", infoMetaObj.toString());

	}
	
	
    @Test
    /**
     *    :  =  %3A
	 *    ;  =  %3B
	 *    =  =  %3D
	 *    %  =  %25
	 *    ,  =  %2C
	 *    CR =  %0D
	 *    LF =  %0A
	 *    TAB=  %09
     */
    public void testInfoBadCharacterSubstitutions(){
		VcfInfoColumnBuilder infoBuilder = new VcfInfoColumnBuilder();
         
        //			EXPECTED							INPUT
        //          ------------------				-----------------------
        assertEquals("bar", 		infoBuilder.replaceBadCharsAndMetaDelimsInValue("foo", "bar"));
        assertEquals("a%2Cb%2Cc",	infoBuilder.replaceBadCharsAndMetaDelimsInValue("foo", "a,b,c"));
        assertEquals("a b  c",		infoBuilder.replaceBadCharsAndMetaDelimsInValue("foo", "a b  c"));
        assertEquals("a%3Bb%3Bc",	infoBuilder.replaceBadCharsAndMetaDelimsInValue("foo", "a;b;c"));
        assertEquals("a:b%3Dc",	infoBuilder.replaceBadCharsAndMetaDelimsInValue("foo", "a:b=c"));
        assertEquals("a -_b%3D%2Cc%3Bd",infoBuilder.replaceBadCharsAndMetaDelimsInValue("foo", "a -_b=,c;d"));
        assertEquals("a.b.c.d.",	infoBuilder.replaceBadCharsAndMetaDelimsInValue("foo", "a.b.c.d."));
    }

    
    @Test
    public void testInfoBadCharacterSubstitutions_oldWay(){
		VcfInfoColumnBuilder infoBuilder = new VcfInfoColumnBuilder();
		infoBuilder.setCharacterEncodingMethod(VcfInfoColumnBuilder.CharacterEncodingMethod.SIMPLE);
         
        //			EXPECTED							INPUT
        //          ------------------				-----------------------
        assertEquals("bar", 		infoBuilder.replaceBadCharsAndMetaDelimsInValue("foo", "bar"));
        assertEquals("a|b|c",		infoBuilder.replaceBadCharsAndMetaDelimsInValue("foo", "a,b,c"));
        assertEquals("a_b__c",		infoBuilder.replaceBadCharsAndMetaDelimsInValue("foo", "a b  c"));
        assertEquals("a|b|c",		infoBuilder.replaceBadCharsAndMetaDelimsInValue("foo", "a;b;c"));
        assertEquals("a:b:c",		infoBuilder.replaceBadCharsAndMetaDelimsInValue("foo", "a:b=c"));
        assertEquals("a_-_b:|c|d",	infoBuilder.replaceBadCharsAndMetaDelimsInValue("foo", "a -_b=,c;d"));
        assertEquals("a.b.c.d.",	infoBuilder.replaceBadCharsAndMetaDelimsInValue("foo", "a.b.c.d."));
    }


    @Test
    public void testDelimiterInMetadata() {
		VcfInfoColumnBuilder infoBuilder = new VcfInfoColumnBuilder();

		// With NO delimiter set
		//			EXPECTED				INPUT
        //          ------------------		-----------------------
        assertEquals("bar|none|baby%2Cdoll",infoBuilder.replaceBadCharsAndMetaDelimsInValue("foo", "bar|none|baby,doll"));

		
		setDelimiterInMetadataForKey(infoBuilder, "foo", "|");
        //			EXPECTED				INPUT
        //          ------------------		-----------------------
        assertEquals("bar,none,baby%2Cdoll",infoBuilder.replaceBadCharsAndMetaDelimsInValue("foo", "bar|none|baby,doll"));

        // Try a 3-char delimiter in header metadata
		setDelimiterInMetadataForKey(infoBuilder, "foo", "|||");
        assertEquals("bar|none,baby%2Cdoll",infoBuilder.replaceBadCharsAndMetaDelimsInValue("foo", "bar|none|||baby,doll"));

        setDelimiterInMetadataForKey(infoBuilder, "foo", ":");
        assertEquals("bar|none|baby",		infoBuilder.replaceBadCharsAndMetaDelimsInValue("foo", "bar|none|baby"));
        
        setDelimiterInMetadataForKey(infoBuilder, "foo", ",");
        assertEquals("bar,none,baby",		infoBuilder.replaceBadCharsAndMetaDelimsInValue("foo", "bar,none,baby"));    	

        setDelimiterInMetadataForKey(infoBuilder, "foo", ";");
        assertEquals("bar,none,baby",		infoBuilder.replaceBadCharsAndMetaDelimsInValue("foo", "bar;none;baby"));
        
        setDelimiterInMetadataForKey(infoBuilder, "foo", "|");
        assertEquals("bar%3Bnone%3Bbaby",	infoBuilder.replaceBadCharsAndMetaDelimsInValue("foo", "bar;none;baby"));    	
    }

    
    
    private void setDelimiterInMetadataForKey(VcfInfoColumnBuilder infoBuilder, String key, String delim) {
    	infoBuilder.mBiorHeaderMap = new HashMap<String, LinkedHashMap<String,String>>();
    	LinkedHashMap<String,String> keyValMap = new LinkedHashMap<String,String>();
    	keyValMap.put(BiorMetaControlledVocabulary.DELIMITER.toString(), delim);
    	infoBuilder.mBiorHeaderMap.put(key, keyValMap);
	}

	@Test
    public void infoJsonObjectToInfoStr() {
    	String jsonStr = "{'MAF':0.35,'AF':[0.2,0.1,0.05],'AF2':{'EUR':0.2,'AFR':0.11},'isDbsnp':false,'isAdded':false,'isUcsc':true,'AString':'aValue','Dud':'.'}";
    	JsonObject json = (JsonObject) new JsonParser().parse(jsonStr);
    	String actual = new VcfInfoColumnBuilder().jsonObjectToInfoDataString("", json);
    	String expected = "MAF=0.35;AF=0.2,0.1,0.05;AF2.EUR=0.2;AF2.AFR=0.11;isUcsc;AString=aValue";
    	assertEquals(expected, actual);
    }
    

    @Test
    public void testAddInfoMeta() {

		String infoToAdd = "##INFO=<ID=MQ,Number=1,Type=Float,Description=\"RMS Mapping Quality\">";
		
		String fileDate     = "##fileDate=20160412";
		String bior         = "##BIOR=<ID=\"dbNSFP_v3a_GRCh37.CaddPhred\",Operation=\"drill\",Field=\"CaddPhred\",DataType=\"String\",Number=\"1\",FieldDescription=\"CADD phred-like score. This is phred-like rank score based on whole genome CADD raw scores\",ShortUniqueName=\"dbNSFP_v3a_GRCh37\",Source=\"dbNSFP\",Description=\"Database developed for functional prediction and annotation of all potential non-synonymous single-nucleotide variants (nsSNVs) in the human genome.\",Version=\"v3.0a\",Build=\"GRCh37\",Path=\"/data5/bsi/catalogs/v1/dbNSFP/3.0a_GRCh37/variants_nodups.v1/dbNSFPv3.0a.zip.tsv.bgz\">";
		String infoBefore   = "##INFO=<ID=AC,Number=1,Type=Float,Description=\"Allele Count\">";
		String infoAfter    = "##INFO=<ID=QC,Number=1,Type=Float,Description=\"Quality Control\">";
		String contig       = "##contig=<ID=chr10,length=135534747>";
		String columnHeader = "#CHROM\tPOS\tREF";
		
		// Add an info line to empty header
		assertInfoAdded(
				/*Expected:*/ Arrays.asList(infoToAdd),
				/*Input:*/    new ArrayList<String>(),
				infoToAdd
				);
		
		// Add to header with only the column header
		assertInfoAdded(
				/*Expected:*/ Arrays.asList(infoToAdd, columnHeader),
				/*Input:*/    Arrays.asList(columnHeader),
				infoToAdd
				);

		// Add after another ##INFO
		assertInfoAdded(
				/*Expected:*/ Arrays.asList(infoBefore, infoToAdd, columnHeader),
				/*Input:*/    Arrays.asList(infoBefore, columnHeader),
				infoToAdd
				);

		// Add before another ##INFO
		assertInfoAdded(
				/*Expected:*/ Arrays.asList(infoToAdd, infoAfter, columnHeader),
				/*Input:*/    Arrays.asList(infoAfter, columnHeader),
				infoToAdd
				);

		// Add between two ##INFO
		assertInfoAdded(
				/*Expected:*/ Arrays.asList(infoBefore, infoToAdd, infoAfter, columnHeader),
				/*Input:*/    Arrays.asList(infoBefore, infoAfter, columnHeader),
				infoToAdd
				);

		// Add among many header lines
		assertInfoAdded(
				/*Expected:*/ Arrays.asList(fileDate, bior, infoBefore, infoToAdd, infoAfter, contig, columnHeader),
				/*Input:*/    Arrays.asList(fileDate, bior, infoBefore, infoAfter, contig, columnHeader),
				infoToAdd
				);

		// Add as only ##INFO line among several header lines
		assertInfoAdded(
				/*Expected:*/ Arrays.asList(fileDate, bior, contig, infoToAdd, columnHeader),
				/*Input:*/    Arrays.asList(fileDate, bior, contig, columnHeader),
				infoToAdd
				);

    }
    
    
	private void assertInfoAdded(List<String> expectedHeaders, List<String> startingHeaders, String infoToAdd) {
		History history = new History(Arrays.asList("X", "100", "A"));
		history.setMetaData(new HistoryMetaData(new ArrayList<String>(startingHeaders)));
		VcfInfoColumnBuilder infoBuilder = new VcfInfoColumnBuilder();
		infoBuilder.addInfoLineToHeader(history, infoToAdd);
		PipeTestUtils.assertListsEqual(expectedHeaders, history.getMetaData().getOriginalHeader());
	}

	private String concat(String... cols) {
		return PipeTestUtils.combine(cols, "\t");
	}


}
