package edu.mayo.bior.cli.func;

import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertTrue;
import static org.junit.Assert.fail;

import java.io.File;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.io.UnsupportedEncodingException;
import java.util.Arrays;
import java.util.List;

import org.junit.BeforeClass;
import org.junit.Ignore;
import org.junit.Test;

import edu.mayo.bior.cli.cmd.DrillCommand;
import edu.mayo.cli.CommandLineApp;
import edu.mayo.cli.CommandPlugin;
import edu.mayo.pipes.util.test.PipeTestUtils;

public class DrillITCase extends BaseFunctionalTest {

	// header line added from running command prior to drill
	private final String prevCmdHeader = 
			"##BIOR=<ID='bior.JSON_COL',Operation='bior_same_variant',DataType='JSON',ShortUniqueName='JSON_COL',Path='/path/to/catalog'>".replaceAll("'", "\"");
	
	private final String stdin =
			prevCmdHeader + "\n" +			
			"#bior.JSON_COL" + "\n" + 
			"{" +
					"\"key1\":\"string_value1\"," +
					"\"key2\":true," +
					"\"key3\":1" +
			"}";	
	
	
	private final String[] headerMeta = {
	  prevCmdHeader,
      "##BIOR=<ID=\"bior.JSON_COL.key1\",Operation=\"bior_drill\",Field=\"key1\",DataType=\"String\",FieldDescription=\"\",ShortUniqueName=\"JSON_COL\",Path=\"/path/to/catalog\">",
	  "##BIOR=<ID=\"bior.JSON_COL.key2\",Operation=\"bior_drill\",Field=\"key2\",DataType=\"String\",FieldDescription=\"\",ShortUniqueName=\"JSON_COL\",Path=\"/path/to/catalog\">",
	  "##BIOR=<ID=\"bior.JSON_COL.key3\",Operation=\"bior_drill\",Field=\"key3\",DataType=\"String\",FieldDescription=\"\",ShortUniqueName=\"JSON_COL\",Path=\"/path/to/catalog\">",
	};
	
	@BeforeClass
	public static void beforeAll() throws FileNotFoundException {
		BaseFunctionalTest.setBiorToolkitCmdsRequired(true);
	}
	
	@Test
	public void testNormalPath() throws IOException, InterruptedException {
        System.out.println("DrillITCase.testNormalPath");
        CommandOutput out = executeScript("bior_drill", stdin, "-p", "key1", "-p", "key2", "-p", "key3");

		assertEquals(out.stderr, 0, out.exit);
		assertEquals("", out.stderr);

		List<String> actual = Arrays.asList(out.stdout.split("\n"));
		List<String> expected = Arrays.asList(
				headerMeta[0], headerMeta[1], headerMeta[2], headerMeta[3], 
				"#bior.JSON_COL.key1\tbior.JSON_COL.key2\tbior.JSON_COL.key3",
				"string_value1\ttrue\t1"
				);
		PipeTestUtils.assertListsEqual(expected, actual);
	}	

	@Test
	public void testKeepJson() throws IOException, InterruptedException {
        System.out.println("DrillITCase.testKeepJson");
        CommandOutput out = executeScript("bior_drill", stdin, "-k", "-p", "key3");

		assertEquals(out.stderr, 0, out.exit);
		assertEquals("", out.stderr);

		List<String> actual = Arrays.asList(out.stdout.split("\n"));
		List<String> expected = Arrays.asList(
				headerMeta[0], headerMeta[3], 
				"#bior.JSON_COL.key3\tbior.JSON_COL",
				"1\t{\"key1\":\"string_value1\",\"key2\":true,\"key3\":1}"
				);
		PipeTestUtils.assertListsEqual(expected, actual);
	}
	
	@Test
	public void testCatalogWithProps() throws IOException, InterruptedException {
        System.out.println("DrillITCase.testCatalogWithProps");
        String catRelativePath  = "src/test/resources/metadata/00-All_GRCh37.tsv.bgz";
        String catCanonicalPath = (new File(catRelativePath)).getCanonicalPath();
		
		String prevCmdHeader = 
				String.format("##BIOR=<ID=\"bior.dbSNP137\",Operation=\"bior_same_variant\",DataType=\"JSON\",ShortUniqueName=\"dbSNP137\",Source=\"dbSNP\",Version=\"137\",Build=\"GRCh37.p10\",Path=\"%s\">", catCanonicalPath); 
		
		String stdin =
				prevCmdHeader + "\n" +			
				"#bior.dbSNP137" + "\n" + 
				"{" +
					"\"INFO\":{" +
						"\"RSPOS\":10145," +
						"\"dbSNPBuildID\":134" +
						"}" +
				"}";	
				
		CommandOutput out = executeScript("bior_drill", stdin, "-p", "INFO.RSPOS", "-p", "INFO.dbSNPBuildID");
		assertEquals(out.stderr, 0, out.exit);
		assertEquals("", out.stderr);

		List<String> actual = Arrays.asList(out.stdout.split("\n"));
		List<String> expected=Arrays.asList(
				prevCmdHeader,
				String.format("##BIOR=<ID=\"bior.dbSNP137.INFO.RSPOS\",Operation=\"bior_drill\",Field=\"INFO.RSPOS\",DataType=\"Integer\",Number=\"1\",FieldDescription=\"Chromosome position reported in dbSNP\",ShortUniqueName=\"dbSNP137\",Source=\"dbSNP\",Version=\"137\",Build=\"GRCh37.p10\",Path=\"%s\">", catCanonicalPath),
				String.format("##BIOR=<ID=\"bior.dbSNP137.INFO.dbSNPBuildID\",Operation=\"bior_drill\",Field=\"INFO.dbSNPBuildID\",DataType=\"Integer\",Number=\"1\",FieldDescription=\"First dbSNP build for RS\",ShortUniqueName=\"dbSNP137\",Source=\"dbSNP\",Version=\"137\",Build=\"GRCh37.p10\",Path=\"%s\">", catCanonicalPath),
				"#bior.dbSNP137.INFO.RSPOS\tbior.dbSNP137.INFO.dbSNPBuildID",
				"10145\t134"
				);
		PipeTestUtils.assertListsEqual(expected, actual);
	}	
	
	
	@Test
    /** Test multiple JSON paths drilled, positive column # for the JSON */
    public void testMultiPath_PosCol() throws IOException, InterruptedException {
        System.out.println("DrillITCase.testMultiPath_PosCol(): Test multiple JSON paths drilled, specifying a positive column number");
        
        
    	final String STDIN = prevCmdHeader + "\n"
        			+ "#CHROM\tPOS\tbior.JSON_COL\tINFO\n"
        			+ "1\t100\t{\"key1\":11,\"key2\":22,\"key3\":33}\tsomeInfo";	

        CommandOutput out = executeScript("bior_drill", STDIN, "-p", "key1", "-p", "key2", "-p", "key3", "-c", "3", "--log");

		assertEquals(out.stderr, 0, out.exit);
		assertEquals("", out.stderr);

		List<String> actual = Arrays.asList(out.stdout.split("\n"));
		List<String> expected = Arrays.asList(
				prevCmdHeader,
				"##BIOR=<ID=\"bior.JSON_COL.key1\",Operation=\"bior_drill\",Field=\"key1\",DataType=\"String\",FieldDescription=\"\",ShortUniqueName=\"JSON_COL\",Path=\"/path/to/catalog\">",
				"##BIOR=<ID=\"bior.JSON_COL.key2\",Operation=\"bior_drill\",Field=\"key2\",DataType=\"String\",FieldDescription=\"\",ShortUniqueName=\"JSON_COL\",Path=\"/path/to/catalog\">",
				"##BIOR=<ID=\"bior.JSON_COL.key3\",Operation=\"bior_drill\",Field=\"key3\",DataType=\"String\",FieldDescription=\"\",ShortUniqueName=\"JSON_COL\",Path=\"/path/to/catalog\">",
				"#CHROM\tPOS\tINFO\tbior.JSON_COL.key1\tbior.JSON_COL.key2\tbior.JSON_COL.key3",
				"1\t100\tsomeInfo\t11\t22\t33"
				);
		PipeTestUtils.assertListsEqual(expected, actual);
	}	

	@Test
    /** Test multiple JSON paths drilled, negative column # for the JSON */
    public void testMultiPath_NegCol() throws IOException, InterruptedException {
        System.out.println("DrillITCase.testMultiPath_NegCol(): Test multiple JSON paths drilled, specifying a negative column number");
        
        
    	final String STDIN = prevCmdHeader + "\n"
        			+ "#CHROM\tPOS\tbior.JSON_COL\tINFO\n"
        			+ "1\t100\t{\"key1\":11,\"key2\":22,\"key3\":33}\tsomeInfo";	

        CommandOutput out = executeScript("bior_drill", STDIN, "-p", "key1", "-p", "key2", "-p", "key3", "-c", "-2", "--log");

		assertEquals(out.stderr, 0, out.exit);
		assertEquals("", out.stderr);

		List<String> actual = Arrays.asList(out.stdout.split("\n"));
		List<String> expected = Arrays.asList(
				prevCmdHeader,
				"##BIOR=<ID=\"bior.JSON_COL.key1\",Operation=\"bior_drill\",Field=\"key1\",DataType=\"String\",FieldDescription=\"\",ShortUniqueName=\"JSON_COL\",Path=\"/path/to/catalog\">",
				"##BIOR=<ID=\"bior.JSON_COL.key2\",Operation=\"bior_drill\",Field=\"key2\",DataType=\"String\",FieldDescription=\"\",ShortUniqueName=\"JSON_COL\",Path=\"/path/to/catalog\">",
				"##BIOR=<ID=\"bior.JSON_COL.key3\",Operation=\"bior_drill\",Field=\"key3\",DataType=\"String\",FieldDescription=\"\",ShortUniqueName=\"JSON_COL\",Path=\"/path/to/catalog\">",
				"#CHROM\tPOS\tINFO\tbior.JSON_COL.key1\tbior.JSON_COL.key2\tbior.JSON_COL.key3",
				"1\t100\tsomeInfo\t11\t22\t33"
				);
		PipeTestUtils.assertListsEqual(expected, actual);

	}
	
	@Test
    /** Test multiple JSON paths drilled, negative column # for the JSON, and keep the JSON column (which will remove it from its current position and append it on the end) */
    public void testMultiPath_NegCol_keepJson() throws IOException, InterruptedException {
        System.out.println("DrillITCase.testMultiPath_NegCol_keepJson(): Test multiple JSON paths drilled, specifying a negative column number, and keeping the JSON column");
        
        
    	final String STDIN = prevCmdHeader + "\n"
        			+ "#CHROM\tPOS\tbior.JSON_COL\tINFO\n"
        			+ "1\t100\t{\"key1\":11,\"key2\":22,\"key3\":33}\tsomeInfo";	

        CommandOutput out = executeScript("bior_drill", STDIN, "-p", "key1", "-p", "key2", "-p", "key3", "-c", "-2", "-k", "--log");

		assertEquals(out.stderr, 0, out.exit);
		assertEquals("", out.stderr);

		List<String> actual = Arrays.asList(out.stdout.split("\n"));
		List<String> expected = Arrays.asList(
				prevCmdHeader,
				"##BIOR=<ID=\"bior.JSON_COL.key1\",Operation=\"bior_drill\",Field=\"key1\",DataType=\"String\",FieldDescription=\"\",ShortUniqueName=\"JSON_COL\",Path=\"/path/to/catalog\">",
				"##BIOR=<ID=\"bior.JSON_COL.key2\",Operation=\"bior_drill\",Field=\"key2\",DataType=\"String\",FieldDescription=\"\",ShortUniqueName=\"JSON_COL\",Path=\"/path/to/catalog\">",
				"##BIOR=<ID=\"bior.JSON_COL.key3\",Operation=\"bior_drill\",Field=\"key3\",DataType=\"String\",FieldDescription=\"\",ShortUniqueName=\"JSON_COL\",Path=\"/path/to/catalog\">",
				"#CHROM\tPOS\tINFO\tbior.JSON_COL.key1\tbior.JSON_COL.key2\tbior.JSON_COL.key3\tbior.JSON_COL",
				"1\t100\tsomeInfo\t11\t22\t33\t{\"key1\":11,\"key2\":22,\"key3\":33}"
				);
		PipeTestUtils.assertListsEqual(expected, actual);

	}
	
	@Ignore   // TODO: Add this back in for the v5.0.0 release to check that slashes are correctly handled
	@Test
    /** Test drilling JSON Arrays and JSON objects */
    public void testJsonArraysAndObjects() throws IOException, InterruptedException {
        System.out.println("DrillITCase.testJsonArraysAndObjects(): Test JSON Arrays and Objects");
        
        
    	final String STDIN =
    			(
    			prevCmdHeader + "\n"
        		+ concat("#CHROM",  "POS",  "bior.JSON_COL", 																"INFO\n")
        		+ concat("1",       "100",  "{'INFO':{'key':'val','_altAlleles':['A','CG','TAAT'],'ALTS':'A/CG/TAAT'}}", 	"someInfo\n")
        		).replaceAll("'", "\"");
        		

    	setStdin(STDIN);
        CommandOutput out = runCmdApp("-p", "INFO", "-p", "INFO.alts", "-p", "INFO.ALTS",  "-p", "INFO._altAlleles",  "-c", "-2");
        

		assertEquals(out.stderr, 0, out.exit);
		assertEquals("", out.stderr);

		List<String> actual = Arrays.asList(out.stdout.split("\n"));
		List<String> expected = Arrays.asList(
				prevCmdHeader,
				"##BIOR=<ID='bior.JSON_COL.INFO',Operation='bior_drill',Field='INFO',DataType='String',Number='.',FieldDescription='',ShortUniqueName='JSON_COL',Path='/path/to/catalog'>".replaceAll("'", "\""),
				"##BIOR=<ID='bior.JSON_COL.INFO.alts',Operation='bior_drill',Field='INFO.alts',DataType='String',Number='.',FieldDescription='',ShortUniqueName='JSON_COL',Path='/path/to/catalog'>".replaceAll("'", "\""),
				"##BIOR=<ID='bior.JSON_COL.INFO.ALTS',Operation='bior_drill',Field='INFO.ALTS',DataType='String',Number='.',FieldDescription='',ShortUniqueName='JSON_COL',Path='/path/to/catalog'>".replaceAll("'", "\""),
				concat("#CHROM",  "POS",  "INFO",     "bior.JSON_COL.INFO",  										"bior.JSON_COL.INFO.alts",  "bior_JSON_COL.INFO.ALTS",  "bior.JSON_COL.INFO._altAlleles"),
				concat("1",    	  "100",  "someInfo", "{'key':'val','_altAlleles':['A','CG','TAAT'],'ALTS':'A/CG/TAAT'}",	".",						"A/CG/TAAT",  				"[\"A\",\"CG\",\"TAAT\"]").replaceAll("'", "\"")
				);
		PipeTestUtils.assertListsEqual(expected, actual);

	}
	
	@Ignore   // TODO: Add this back in for the v5.0.0 release to verify special characters are escaped correctly
	@Test
    /** Test escape characters */
    public void testEscapeChars() throws IOException, InterruptedException {
        System.out.println("DrillITCase.testEscapeChars(): Test escape characters");
        
        												//		 ~!@#$%^&*()_-=+\|[]{};:'"<>,./?"
        String jsonWithSpecialChrs = "{\"INFO\":{\"ESC_CHARS\":\"~!@#$%^&*()_-=+\\|[]{};:\\\'\\\"<>,./?\"}}";
    	final String STDIN =
    			prevCmdHeader + "\n"
        		+ "#CHROM	POS	bior.JSON_COL\n"
        		+ "1	100	" + jsonWithSpecialChrs + "\n";
        		

    	setStdin(STDIN);
        CommandOutput out = runCmdApp("-p", "INFO", "-p", "INFO.alts", "-p", "INFO.ALTS", "-p", "INFO.ESC_CHARS", "-c", "-2");
        

		assertEquals(out.stderr, 0, out.exit);
		assertEquals("", out.stderr);

		List<String> actual = Arrays.asList(out.stdout.split("\n"));
		List<String> expected = Arrays.asList(
				prevCmdHeader,
				"##BIOR=<ID='bior.JSON_COL.INFO',Operation='bior_drill',Field='INFO',DataType='String',Number='.',FieldDescription='',ShortUniqueName='JSON_COL',Path='/path/to/catalog'>".replaceAll("'", "\""),
				"##BIOR=<ID='bior.JSON_COL.INFO.alts',Operation='bior_drill',Field='INFO.alts',DataType='String',Number='.',FieldDescription='',ShortUniqueName='JSON_COL',Path='/path/to/catalog'>".replaceAll("'", "\""),
				"#CHROM	POS	INFO	bior.JSON_COL.INFO	bior.JSON_COL.INFO.alts	bior.JSON_COL.INFO.ESC_CHARS	INFO".replaceAll("'", "\""),
				"1	100	someInfo	{'key':'val','alts':['A','CG','TAAT'],'ALTS':'A/CG/TAAT','ESC_CHARS':'~!@#$%^&*()_-=+\\|[]{};:\'\"<>,./?'}	['A','CG','TAAT']	~!@#$%^&*()_-=+\\|[]{};:\'\"<>,./?".replaceAll("'", "\"")
				);
		PipeTestUtils.assertListsEqual(expected, actual);

	}
	

	
	
	@Test
    /** Test default JSON array collapse behavior, which should use pipes ("|") */
    public void testJsonArray_defaultDelimiter_jsonNotLastCol() throws IOException, InterruptedException {
        System.out.println("DrillITCase.testJsonArray_defaultDelimiter(): Test default JSON Array separator");
        
    	final String STDIN = swapQuotes("##BIOR=<ID='Foo',Operation='bior_overlap',DataType='JSON',ShortUniqueName='',Path=''>") + "\n"
    					+    concat("#CHR", "POS",  "Foo",                                         "A",   "B") + "\n"
    					+    concat("1",    "100",  "{'gene':'ADT','key':['val1','val2','val3']}", "1.0", "1.1") + "\n";
        		
    	setStdin(STDIN);
        CommandOutput out = runCmdApp("-p", "key", "-p", "gene", "-c", "-3");
        
		assertEquals(out.stderr, 0, out.exit);
		assertEquals("", out.stderr);

		List<String> actual = Arrays.asList(out.stdout.split("\n"));
		List<String> expected = Arrays.asList(
				swapQuotes("##BIOR=<ID='Foo',Operation='bior_overlap',DataType='JSON',ShortUniqueName='',Path=''>"),
				swapQuotes("##BIOR=<ID='Foo.key',Operation='bior_drill',Field='key',DataType='String',FieldDescription='',ShortUniqueName='',Path='',Delimiter='|',Number='.'>"),
				swapQuotes("##BIOR=<ID='Foo.gene',Operation='bior_drill',Field='gene',DataType='String',FieldDescription='',ShortUniqueName='',Path=''>"),
				concat("#CHR", "POS",  "A",  "B",  "Foo.key",        "Foo.gene"),
				concat("1",    "100",  "1.0","1.1","val1|val2|val3", "ADT")
				);
		PipeTestUtils.assertListsEqual(expected, actual);
	}
	
	
	@Test
    /** Test default JSON array collapse behavior, which should use pipes ("|").  Keep the JSON column (and that column is not the last one) */
    public void testJsonArray_defaultDelimiter_keepJson_jsonNotLastCol() throws IOException, InterruptedException {
        System.out.println("DrillITCase.testJsonArray_defaultDelimiter_keepJson(): Test default JSON Array separator");
        
    	final String STDIN = concat("#CHR", "POS",  "Foo",                            "A",  "B") + "\n"
    					+    concat("1",    "100",  "{'key':['val1','val2','val3']}", "1.0","1.2") + "\n";
        		
    	setStdin(STDIN);
        CommandOutput out = runCmdApp("-p", "key", "-k", "-c", "3");
        
		assertEquals(out.stderr, 0, out.exit);
		assertEquals("", out.stderr);

		List<String> actual = Arrays.asList(out.stdout.split("\n"));
		List<String> expected = Arrays.asList(
				swapQuotes("##BIOR=<ID='bior.Foo.key',Operation='bior_drill',DataType='String',ShortUniqueName='',Path='',Delimiter='|',Number='.'>"),
				concat("#CHR", "POS",  "A",  "B",  "bior.Foo.key",   "Foo"),
				concat("1",    "100",  "1.0","1.2","val1|val2|val3", "{'key':['val1','val2','val3']}")
				);
		PipeTestUtils.assertListsEqual(expected, actual);
	}

	@Test
	public void testJsonArray_overrideDelimiter() throws UnsupportedEncodingException {
        System.out.println("DrillITCase.testJsonArrayDefaultDelimiter(): Test default JSON Array separator");
        
    	final String STDIN = "1	100	{'key':['val1','val2','val3']}" + "\n";
        		
    	setStdin(STDIN);
        CommandOutput out = runCmdApp("-p", "key", "-d", ";");
        
		assertEquals(out.stderr, 0, out.exit);
		assertEquals("", out.stderr);

		List<String> actual = Arrays.asList(out.stdout.split("\n"));
		List<String> expected = Arrays.asList(
				"##BIOR=<ID='bior.#UNKNOWN_3.key',Operation='bior_drill',DataType='String',ShortUniqueName='',Path='',Delimiter=';',Number='.'>".replaceAll("'", "\""),
				"#UNKNOWN_1" + "\t" + "#UNKNOWN_2" + "\t" + "bior.#UNKNOWN_3.key",
				"1	100	val1;val2;val3"
				);
		PipeTestUtils.assertListsEqual(expected, actual);
	}
	
	@Test
	public void testJsonArray_overrideDelimiterWithSame_skipNullsAndDots() throws UnsupportedEncodingException {
        System.out.println("DrillITCase.testJsonArrayDefaultDelimiter(): Test default JSON Array separator, and skip nulls and dots in JSON arrays");
        
    	final String STDIN = "1	100	{'key':['val1',null,'val2','val3','null','.','val4',null,'.']}" + "\n";
        		
    	setStdin(STDIN);
        CommandOutput out = runCmdApp("-p", "key", "-d", "|", "-s");
        
		assertEquals(out.stderr, 0, out.exit);
		assertEquals("", out.stderr);

		List<String> actual = Arrays.asList(out.stdout.split("\n"));
		List<String> expected = Arrays.asList(
				"##BIOR=<ID='bior.#UNKNOWN_3.key',Operation='bior_drill',DataType='String',ShortUniqueName='',Path='',Delimiter='|',Number='.'>".replaceAll("'", "\""),
				"#UNKNOWN_1" + "\t" + "#UNKNOWN_2" + "\t" + "bior.#UNKNOWN_3.key",
				// NOTE: The 'null' after 'val3' is a String and not the actual null value, so it SHOULD be included:
				"1	100	val1|val2|val3|null|val4"
				);
		PipeTestUtils.assertListsEqual(expected, actual);
	}
	
	
	@Test
	public void testJsonArray_exceptionFromDelimOccurringInArrayVals() throws UnsupportedEncodingException {
        System.out.println("DrillITCase.testJsonArray_exceptionFromDelimOccurringInArrayVals(): Test default JSON Array separator, and catch exception thrown by delim appearing in array value.");
        
    	final String STDIN = "1	100	{'key':['val1','val2|val3','val4']}" + "\n";
        		
    	setStdin(STDIN);
        CommandOutput out = runCmdApp("-p", "key");

        assertEquals("", out.stdout);
		assertEquals(out.stderr, 1, out.exit);
		assertTrue(out.stderr.contains("Error: the delimiter '|' was found within one of the array values that was drilled: 'val2|val3'"));
	}

	
    private CommandOutput runCmdApp(String... cmdArgs) throws UnsupportedEncodingException {
        CommandLineApp app = new CommandLineApp();
        app.captureSystemOutAndErrorToStrings();
        CommandPlugin mockPlugin;
        mockPlugin = new DrillCommand();
    	
        CommandOutput output = new CommandOutput();
        output.exit = app.runApplication(new DrillCommand().getClass().getName(), /*MOCK_SCRIPT_NAME=*/"bior_drill", cmdArgs);
        // Set SYSOUT and SYSERR back to their original output streams
        app.resetSystemOutAndError();

        output.stdout = app.getSystemOutMessages();
        output.stderr = app.getSystemErrorMessages();
        
        return output;
    }
}
