package edu.mayo.bior.cli.cmd;

import static org.junit.Assert.assertEquals;

import java.io.File;
import java.io.IOException;
import java.io.UnsupportedEncodingException;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;

import org.apache.commons.io.FileUtils;
import org.apache.commons.lang.CharSet;
import org.apache.commons.lang.StringUtils;
import org.junit.Before;
import org.junit.Test;

import edu.mayo.bior.cli.func.BaseFunctionalTest;
import edu.mayo.bior.cli.func.CommandOutput;
import edu.mayo.cli.CommandLineApp;
import edu.mayo.cli.CommandPlugin;
import edu.mayo.pipes.util.test.PipeTestUtils;

/** Tests for bior_vcf_to_tjson (VCF2VariantCommand) 
 *  @author  Michael Meiners 2017-03-03 */
public class VCF2VariantCommandTest  extends BaseFunctionalTest {

	private final String EOL = "\n";
	
	private List<String> STDIN1;
	
	@Before
	public void beforeEach() {
    	STDIN1 = Arrays.asList(
    		"##INFO=<ID=isFlag,Number=0,Type=Flag,Description=\"A flag\">",
    		"##INFO=<ID=aStr,Number=1,Type=String,Description=\"A string\">",
    		"##INFO=<ID=anIntArray,Number=.,Type=Integer,Description=\"An array of integers\">",
    		"##INFO=<ID=aFloat,Number=1,Type=Float,Description=\"A  single float value\">",
    		concat("#CHROM", "POS", "ID",   "REF", "ALT", "QUAL", "FILTER", "INFO"),
    		concat("1",      "100", ".",    ".",   ".",   ".",    ".",      "isFlag;aStr=ABC;anIntArray=1,2,3;aFloat=0.339"),
    		concat("2",      "200", "rs02", "C",   "G",   ".",    ".",      "aStr=XYZ;anIntArray=0;aFloat=0") // No flag on this one
    		);
	}
	
	
    @Test
    public void testVcfToTjson() throws IOException {
     	CommandOutput output = runCmdApp(StringUtils.join(STDIN1, EOL));
        
    	List<String> expected = new ArrayList<String>();
    	// Add the ## headers (indexes 0,1,2,3)
    	expected.addAll(STDIN1.subList(0, 4));
    	// And ##BIOR line
    	expected.add(swapQuotes("##BIOR=<ID='bior.ToTJson',Operation='bior_vcf_to_tjson',DataType='JSON',ShortUniqueName='ToTJson'>"));
    	// And the column header line with bior.ToTJson appended to end
    	expected.add(STDIN1.get(4) + "\t" + "bior.ToTJson");
    	expected.add(swapQuotes(concat("1",      "100", ".",    ".",   ".",   ".",    ".",      "isFlag;aStr=ABC;anIntArray=1,2,3;aFloat=0.339",  "{'CHROM':'1','POS':100,'INFO':{'isFlag':true,'aStr':'ABC','anIntArray':[1,2,3],'aFloat':0.339},'_landmark':'1','_minBP':100,'_maxBP':100}")));
    	expected.add(swapQuotes(concat("2",      "200", "rs02", "C",   "G",   ".",    ".",      "aStr=XYZ;anIntArray=0;aFloat=0",                 "{'CHROM':'2','POS':200,'ID':'rs02','REF':'C','ALT':'G','INFO':{'aStr':'XYZ','anIntArray':[0],'aFloat':0},'_id':'rs02','_type':'variant','_landmark':'2','_refAllele':'C','_altAlleles':['G'],'_minBP':200,'_maxBP':200}")));
    	
		assertOk(output, expected);
    }

    @Test
    public void testLargeInsDelDupToTjson() throws IOException {
    	String STDIN = FileUtils.readFileToString(new File("src/test/resources/vcfToTjson/largeInsDelDup.vcf"), 	 "UTF-8");
    	String EXP   = FileUtils.readFileToString(new File("src/test/resources/vcfToTjson/largeInsDelDup.expected"), "UTF-8");
    	//List<String> EXP_LINES 	= FileUtils.readLines(		 new File("src/test/resources/vcfToTjson/largeInsDelDup.expected"), "UTF-8");
     	CommandOutput output = runCmdApp(STDIN);
    	assertEquals(EXP, output.stdout);
		//assertOk(output, EXPECTED);
    }

 	@Test
    public void testVcfToTjson_addFlagFalses() throws IOException {
     	CommandOutput output = runCmdApp(StringUtils.join(STDIN1, EOL), "-f");
        
    	List<String> expected = new ArrayList<String>();
    	// Add the ## headers (indexes 0,1,2,3)
    	expected.addAll(STDIN1.subList(0, 4));
    	// And ##BIOR line
    	expected.add(swapQuotes("##BIOR=<ID='bior.ToTJson',Operation='bior_vcf_to_tjson',DataType='JSON',ShortUniqueName='ToTJson'>"));
    	// And the column header line with bior.ToTJson appended to end
    	expected.add(STDIN1.get(4) + "\t" + "bior.ToTJson");
    	expected.add(swapQuotes(concat("1",      "100", ".",    ".",   ".",   ".",    ".",      "isFlag;aStr=ABC;anIntArray=1,2,3;aFloat=0.339",  "{'CHROM':'1','POS':100,'INFO':{'isFlag':true,'aStr':'ABC','anIntArray':[1,2,3],'aFloat':0.339},'_landmark':'1','_minBP':100,'_maxBP':100}")));
    	expected.add(swapQuotes(concat("2",      "200", "rs02", "C",   "G",   ".",    ".",      "aStr=XYZ;anIntArray=0;aFloat=0",                 "{'CHROM':'2','POS':200,'ID':'rs02','REF':'C','ALT':'G','INFO':{'aStr':'XYZ','anIntArray':[0],'aFloat':0,'isFlag':false},'_id':'rs02','_type':'variant','_landmark':'2','_refAllele':'C','_altAlleles':['G'],'_minBP':200,'_maxBP':200}")));
    	
		assertOk(output, expected);
}
    
    
    //--------------------------------------------------------------------------------------------
	private void assertOk(CommandOutput output, List<String> expected) {
		assertEquals(output.stderr, "", output.stderr);
		List<String> actual = Arrays.asList(output.stdout.split("\n"));
		PipeTestUtils.assertListsEqual(expected, actual);
        assertEquals(output.exit,   0);
	}
    

 	private CommandOutput runCmdApp(String stdin, String... cmdArgs) throws UnsupportedEncodingException {
        String MOCK_SCRIPT_NAME = "bior_vcf_to_variant";
        CommandLineApp app = new CommandLineApp();
        app.captureSystemOutAndErrorToStrings();
        CommandPlugin mockPlugin;
        mockPlugin = new VCF2VariantCommand();
    	
        CommandOutput output = new CommandOutput();
        app.setStdIn(stdin);
        output.exit = app.runApplication(mockPlugin.getClass().getName(), MOCK_SCRIPT_NAME, cmdArgs);
        // Set SYSOUT and SYSERR back to their original output streams
        app.resetSystemOutAndError();

        output.stdout = app.getSystemOutMessages();
        output.stderr = app.getSystemErrorMessages();
	        
        return output;
   }
}
