package edu.mayo.bior.cli.cmd;

import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertTrue;

import java.io.ByteArrayInputStream;
import java.io.File;
import java.io.IOException;
import java.io.InputStream;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;

import org.apache.commons.lang.ArrayUtils;
import org.apache.commons.lang.StringUtils;
import org.junit.Before;
import org.junit.Rule;
import org.junit.Test;
import org.junit.rules.TemporaryFolder;

import com.tinkerpop.pipes.util.Pipeline;

import edu.mayo.bior.cli.func.CommandOutput;
import edu.mayo.cli.CommandLineApp;
import edu.mayo.pipes.JSON.tabix.BgzipWriter;
import edu.mayo.pipes.UNIX.CatGZPipe;
import edu.mayo.pipes.history.HistoryInPipe;
import edu.mayo.pipes.util.test.PipeTestUtils;

/** Test the bior_catalog_remove_duplicates command */
public class CatalogRemoveDuplicatesCommandTest {

	private List<String> mInput1    = new ArrayList<String>();
	private File mInputCatalog1;

	private List<String> mInput2    = new ArrayList<String>();
	private File mInputCatalog2;

	@Rule
	public TemporaryFolder mTempFolder;
	private File mTempDir;
	
	@Before
	public void before() throws IOException {
		mTempFolder = new TemporaryFolder();
		mTempFolder.create();
		mTempDir = mTempFolder.newFolder();
		mInput1.addAll( Arrays.asList(
				"1	100	100	{'_refAllele':'A'}",
				"1	100	100	{'_refAllele':'A'}", // remove
				"1	100	100	{'_refAllele':'T'}",
				"1	100	100	{'_refAllele':'A','_altAlleles':['C','G']}", // remove
				"1	200	200	{'_refAllele':'A'}",
				"M	10	10	{'_refAllele':'C'}"
				) );
		replaceAllSingleWithDoubleQuotes(mInput1);
		mInputCatalog1 = buildCatalogFrom(mInput1, "inputCatalog1.tsv.bgz");

	
		mInput2.addAll( Arrays.asList(			// Array match:    Any:		Exact:
				"1	100	100	{'_altAlleles':['A']}",
				"1	100	100	{'_altAlleles':['C','A','T']}",		// remove
				"1	100	100	{'_altAlleles':['T']}",				
				"1	100	100	{'_altAlleles':['A']}",				// remove	remove
				"1	200	200	{'_altAlleles':['C','G','T']}",
				"1	200	200	{'_altAlleles':['C','G']}",			// remove
				"1	200	200	{'_altAlleles':['C','G']}",			// remove	remove
				"1	200	200	{'_altAlleles':['C','G','T']}",		// remove	remove
				"M	10	10	{'_altAlleles':['C']}"
				) );
		replaceAllSingleWithDoubleQuotes(mInput2);
		mInputCatalog2 = buildCatalogFrom(mInput2, "inputCatalog2.tsv.bgz");
	}
	

	//=============================================================================

	@Test
	public void testSkipRegion() throws IOException {
		try {
		List<String> actual = removeDuplicates_Files(mInputCatalog1, "-p", "_refAllele", "-s", "M:1-100");
		List<String> expected = getInputMinusRemovedRows(mInput1, 1, 3);
		System.out.println("Actual:");
		PipeTestUtils.printLines(actual);
		System.out.println("Expected:");
		PipeTestUtils.printLines(expected);
		PipeTestUtils.assertListsEqual(expected, actual);
		}catch(Exception e) {
			e.printStackTrace();
		}
	}

	@Test
	public void testSkipRegionStdinStdout() throws IOException {
		try {
			String stdin = StringUtils.join(mInput1, "\n");
			CommandOutput out = removeDuplicates_StdinStdout(stdin, "-p", "_refAllele", "-s", "M:1-100");
			
			List<String> expected = getInputMinusRemovedRows(mInput1, 1, 3);
			List<String> actual = Arrays.asList(out.stdout.split("\n"));
			//PipeTestUtils.printLines(actual);
			//PipeTestUtils.printLines(expected);
			PipeTestUtils.assertListsEqual(expected, actual);
			
			// NOTE: DEPRECATION warning has been added for this cmd in BioR v4.4.0
			final String DEPRECATION_WARNING = "WARNING:  THIS COMMAND WILL BE REMOVED IN BIOR v5.0.0!" + "\n"
											+  "You can continue to use older versions of BioR to run this command" + "\n";
			assertEquals(DEPRECATION_WARNING, out.stderr);
		}catch(Exception e) {
			e.printStackTrace();
		}
	}

	@Test
	public void testSkipRegionOnlyChrom() throws IOException {
		List<String> actual = removeDuplicates_Files(mInputCatalog1, "-p", "_refAllele", "-s", "M");
		List<String> expected = getInputMinusRemovedRows(mInput1, 1, 3);
		PipeTestUtils.assertListsEqual(expected, actual);
		// Verify that the catalog exists and is non-zero size
		File ctg = new File(mTempDir, "outputCatalog.tsv.bgz");
		File tbi = new File(mTempDir, "outputCatalog.tsv.bgz.tbi");
		assertTrue(ctg.exists() && ctg.length() > 0);
		// Verify that the tabix index exists and is non-zero size
		assertTrue(tbi.exists() && ctg.length() > 0);
	}

	@Test
	public void testJsonPathNotFound() throws IOException {
		List<String> actual = removeDuplicates_Files(mInputCatalog1, "-p", "_altAlleles", "-s", "M:1-100");
		List<String> expected = getInputMinusRemovedRows(mInput1, 1, 2);
		PipeTestUtils.assertListsEqual(expected, actual);
	}
	
	

	@Test
	public void testAllElementsInArray() throws IOException {
		List<String> actual = removeDuplicates_Files(mInputCatalog2, "-p", "_altAlleles");
		List<String> expected = getInputMinusRemovedRows(mInput2, 3, 6, 7);
		PipeTestUtils.assertListsEqual(expected, actual);
	}

	@Test
	public void testAnyElementInArray() throws IOException {
		List<String> actual = removeDuplicates_Files(mInputCatalog2, "-p", "_altAlleles", "-a");
		List<String> expected = getInputMinusRemovedRows(mInput2, 1, 3, 5, 6, 7);
		PipeTestUtils.assertListsEqual(expected, actual);
	}
	
//=============================================================================
	
	

	private void replaceAllSingleWithDoubleQuotes(List<String> list) {
		for(int i=0; i < list.size(); i++) {
			list.set(i, list.get(i).replaceAll("'", "\""));
		}
	}



	private File buildCatalogFrom(List<String> inputLines, String catalogFilename) throws IOException {
		File catalogFile = new File(mTempDir, catalogFilename);
		Pipeline pipeline = new Pipeline(
				new HistoryInPipe(),
				new BgzipWriter(catalogFile.getCanonicalPath())
				);
		pipeline.setStarts(inputLines);
		while(pipeline.hasNext()) 
			pipeline.next();
		return catalogFile;
	}

	/** Return the input catalog rows after removing the specified rows */
	private List<String> getInputMinusRemovedRows(List<String> inputList,  int... rowsToRemove) {
		ArrayList<String> list = new ArrayList<String>(inputList);
		Arrays.sort(rowsToRemove);
		for(int i=rowsToRemove.length-1; i >=0; i--) {
			list.remove(rowsToRemove[i]);
		}
		return list;
	}


	private List<String> removeDuplicates_Files(File inputCatalog, String... cmdArgs) throws IOException {
		File outputCatalog = new File(mTempDir, "outputCatalog.tsv.bgz");
		CommandLineApp cmdLineApp = new CommandLineApp();
		
		cmdLineApp.runApplication(
				"edu.mayo.bior.cli.cmd.CatalogRemoveDuplicatesCommand",
				"bior_catalog_remove_duplicates",
				(String[]) ArrayUtils.addAll(
						new String[] {
								"-i",  inputCatalog.getCanonicalPath(),
								"-o",  outputCatalog.getCanonicalPath(),
								},
						cmdArgs
						)
				);
		
		return getBgzipContents(outputCatalog.getCanonicalPath());
	}

	private CommandOutput removeDuplicates_StdinStdout(String stdIn, String... args) throws IOException {
		CommandLineApp cmdLineApp = new CommandLineApp();
		
		setStdin(stdIn);
		
		cmdLineApp.captureSystemOutAndErrorToStrings();
		CommandOutput out = new CommandOutput();
		out.exit = cmdLineApp.runApplication(
				"edu.mayo.bior.cli.cmd.CatalogRemoveDuplicatesCommand",
				"bior_catalog_remove_duplicates",
				args
				);
		
		out.stderr = cmdLineApp.getSystemErrorMessages();
		out.stdout = cmdLineApp.getSystemOutMessages();
		
		cmdLineApp.resetSystemOutAndError();
		
		return out;
	}
	
	protected void setStdin(String s) {
		InputStream fakeIn = new ByteArrayInputStream(s.getBytes());
		System.setIn(fakeIn);
	}

	
	private List<String> getBgzipContents(String bgzipPath) {
		Pipeline<String, String> pipeline = new Pipeline(
				new CatGZPipe("gzip")
				);
		pipeline.setStarts(Arrays.asList(bgzipPath));
		return PipeTestUtils.getResults(pipeline);
	}
}
