package edu.mayo.bior.catalog.verification;

import static junit.framework.Assert.assertNotNull;
import static junit.framework.Assert.assertNull;
import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertFalse;
import static org.junit.Assert.assertTrue;

import org.junit.Before;
import org.junit.Rule;
import org.junit.Test;
import org.junit.rules.ExpectedException;
import org.junit.rules.TemporaryFolder;

import java.io.*;
import java.util.ArrayList;
import java.util.List;

/**
 * Created by m026366 on 8/3/16.
 * Testing MessageLogger
 */
public class MessageLoggerTest
{
   @Rule
   public TemporaryFolder temporaryFolder = new TemporaryFolder();

   @Rule
   public ExpectedException expectedException = ExpectedException.none();

   private static final String NL = System.getProperty("line.separator");

   private MessageLogger logger = null;
   private File logfile = null;
   private static final String MSG = "msg";

   @Before
   public void setUp() throws IOException
   {
      logfile = temporaryFolder.newFile();
      logger = new MessageLogger(logfile.getPath());
   }

   @Test
   public void testBasic() throws IOException
   {
      assertEquals(0, logger.numErrors());
      assertEquals(0, logger.numWarnings());
      assertEquals(0, logger.numInfos());
      assertEquals(0, logger.numMessages());
      assertFalse(logger.hasErrors());

      logger.logWarning(MSG);
      logger.logError(MSG, 1);
      logger.logError(MSG, 1);
      logger.logError(MSG, 2);
      logger.logWarning(MSG);
      logger.logInfo(MSG);
      assertEquals(3, logger.numErrors());
      assertEquals(2, logger.numErrorsForCode(1));
      assertEquals(1, logger.numErrorsForCode(2));
      assertEquals(0, logger.numErrorsForCode(3));
      assertEquals(0, logger.numErrorsForCode(0));
      assertEquals(2, logger.numWarnings());
      assertEquals(1, logger.numInfos());
      assertEquals(6, logger.numMessages());
      assertTrue(logger.hasErrors());
      assertNotNull(logger.getLogFile());

      MessageLogger loggerFromOutput = MessageLogger.loggerFromOutput(new FileReader(logfile));
      assertEquals(3, loggerFromOutput.numErrors());
      assertEquals(2, loggerFromOutput.numErrorsForCode(1));
      assertEquals(1, loggerFromOutput.numErrorsForCode(2));
      assertEquals(0, loggerFromOutput.numErrorsForCode(3));
      assertEquals(0, loggerFromOutput.numErrorsForCode(0));
      assertEquals(2, loggerFromOutput.numWarnings());
      assertEquals(1, loggerFromOutput.numInfos());
      assertEquals(6, loggerFromOutput.numMessages());
      assertTrue(loggerFromOutput.hasErrors());
      assertNull(loggerFromOutput.getLogFile());
      loggerFromOutput.logWarning("msg that won't go anywhere but should work");


      StringWriter msgWriter = new StringWriter();
      MessageLogger stringLogger = new MessageLogger(new BufferedWriter(msgWriter));
      stringLogger.logError(MSG, 1);
      stringLogger.logWarning(MSG);
      stringLogger.logInfo(MSG);
      assertEquals(1, stringLogger.numErrors());
      assertEquals(1, stringLogger.numErrorsForCode(1));
      assertEquals(0, stringLogger.numErrorsForCode(2));
      assertEquals(1, stringLogger.numWarnings());
      assertEquals(1, stringLogger.numInfos());
      assertEquals(3, stringLogger.numMessages());
      assertTrue(msgWriter.toString().contains(MSG));
      assertTrue(msgWriter.toString().contains(LogLevel.ERROR.getLabel() + " 1:"));
      assertTrue(msgWriter.toString().contains(NL));
      assertNull(stringLogger.getLogFile());

      loggerFromOutput = MessageLogger.loggerFromOutput(new StringReader(msgWriter.toString()));
      assertEquals(1, loggerFromOutput.numErrors());
      assertEquals(1, loggerFromOutput.numErrorsForCode(1));
      assertEquals(0, loggerFromOutput.numErrorsForCode(2));
      assertEquals(1, loggerFromOutput.numWarnings());
      assertEquals(1, loggerFromOutput.numInfos());
      assertEquals(3, loggerFromOutput.numMessages());

      // Now add to existing logger and check if it works
      List<Reader> readerList = new ArrayList<Reader>();
      readerList.add(new FileReader(logfile));
      readerList.add(new StringReader(msgWriter.toString()));
      loggerFromOutput = MessageLogger.loggerFromOutputList(readerList);
      assertEquals(4, loggerFromOutput.numErrors());
      assertEquals(3, loggerFromOutput.numErrorsForCode(1));
      assertEquals(1, loggerFromOutput.numErrorsForCode(2));
      assertEquals(0, loggerFromOutput.numErrorsForCode(3));
      assertEquals(0, loggerFromOutput.numErrorsForCode(0));
      assertEquals(3, loggerFromOutput.numWarnings());
      assertEquals(2, loggerFromOutput.numInfos());
      assertEquals(9, loggerFromOutput.numMessages());

      // Do it again and add the logfile twice
      readerList = new ArrayList<Reader>();
      readerList.add(new FileReader(logfile));
      readerList.add(new StringReader(msgWriter.toString()));
      readerList.add(new FileReader(logfile));
      loggerFromOutput = MessageLogger.loggerFromOutputList(readerList);
      assertEquals(7, loggerFromOutput.numErrors());
      assertEquals(5, loggerFromOutput.numErrorsForCode(1));
      assertEquals(2, loggerFromOutput.numErrorsForCode(2));
      assertEquals(0, loggerFromOutput.numErrorsForCode(3));
      assertEquals(0, loggerFromOutput.numErrorsForCode(0));
      assertEquals(5, loggerFromOutput.numWarnings());
      assertEquals(3, loggerFromOutput.numInfos());
      assertEquals(15, loggerFromOutput.numMessages());
   }

   @Test
   public void testErrorMixedCodeFromText() throws Exception
   {
      String error = "ERROR";
      int code = 1;
      String msg = "msg";
      String message3Lines = String.format("%s: %s%n%s %d: %s%n%s: %s%n", error, msg, error, code, msg, error, msg);
      MessageLogger loggerFromOutput = MessageLogger.loggerFromOutput(new StringReader(message3Lines));
      assertEquals(3, loggerFromOutput.numErrors());
      assertEquals(2, loggerFromOutput.numErrorsForCode(-99));
      assertEquals(1, loggerFromOutput.numErrorsForCode(code));
   }

   /*
   This only works if Message class in MessageLogger is public but that's not usually
   how it is and then you have to import static ...MessageLogger.Message
   If you have problems with messageFromString method in Message, this would be good to test
   @Test
   public void testMessage()
   {
      MessageLogger.Message inMessage = new MessageLogger.Message("msga:b: c", LogLevel.WARNING, null);
      String inMessageStr = inMessage.toString();
      MessageLogger.Message fromMessage = MessageLogger.Message.messageFromString(inMessageStr);
      assertEquals(inMessage.getMsg(), fromMessage.getMsg());
      assertEquals(inMessage.getLevel(), fromMessage.getLevel());
      assertEquals(inMessage.getCode(), fromMessage.getCode());

      inMessage = new MessageLogger.Message("msga:b: c", LogLevel.INFO, null);
      inMessageStr = inMessage.toString();
      fromMessage = MessageLogger.Message.messageFromString(inMessageStr);
      assertEquals(inMessage.getMsg(), fromMessage.getMsg());
      assertEquals(inMessage.getLevel(), fromMessage.getLevel());
      assertEquals(inMessage.getCode(), fromMessage.getCode());

      inMessage = new MessageLogger.Message("msga:b: c", LogLevel.ERROR, 1);
      inMessageStr = inMessage.toString();
      fromMessage = MessageLogger.Message.messageFromString(inMessageStr);
      assertEquals(inMessage.getMsg(), fromMessage.getMsg());
      assertEquals(inMessage.getLevel(), fromMessage.getLevel());
      assertEquals(inMessage.getCode(), fromMessage.getCode());

      assertNull(MessageLogger.Message.messageFromString("blah"));
      assertNull(MessageLogger.Message.messageFromString("ERROR notint: msg"));
      assertNull(MessageLogger.Message.messageFromString("warning: msg"));
      assertNull(MessageLogger.Message.messageFromString("WARNING 1: msg"));

      String loggerMsg = "ERROR: msg";
      MessageLogger.Message message = MessageLogger.Message.messageFromString(loggerMsg);
      assertTrue(message.isError());
      assertEquals(-99, (long)message.getCode());
      assertEquals("msg", message.getMsg());
      assertEquals(loggerMsg, message.toString());
   }
   */
}
