package edu.mayo.bior.catalog.list;

import edu.mayo.bior.catalog.CatalogFormatException;
import edu.mayo.bior.catalog.list.CatalogInfo.CatalogStatus;

import org.junit.Rule;
import org.junit.Test;
import org.junit.rules.ExpectedException;
import org.junit.rules.TemporaryFolder;

import java.io.File;
import java.io.IOException;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Iterator;
import java.util.List;

import static junit.framework.Assert.assertEquals;
import static junit.framework.Assert.assertTrue;

/**
 * Test CatalogList class
 */
public class CatalogListTest
{
   @Rule
   public ExpectedException expectedException = ExpectedException.none();

   @Rule
   public TemporaryFolder temporaryFolder = new TemporaryFolder();

   @Test
   public void testBasic() throws CatalogFormatException
   {
	   String catalogListPath = "src/main/resources/CATALOG_LIST.txt";
	   List<CatalogInfo> actual = new CatalogList(catalogListPath).infoList;
	   
	   // There are 54 lines, but we'll just take the first  (1000_genomes/20110521)
	   final CatalogInfo expected1 = new CatalogInfo();
	   expected1.status      = CatalogStatus.Active;
	   expected1.path        = "/research/bsi/data/catalogs/bior/v1/1000_genomes/20110521/ALL.wgs.phase1_release_v3.20101123.snps_indels_sv.sites_GRCh37.tsv.bgz";
	   expected1.lastUpdated = "2014-01-08 14:16:11";
	   expected1.shortName   = "1kG_3";
	   expected1.indexes     = new ArrayList();
	   expected1.dataSource  = "1000 Genomes";
	   expected1.version     = "3";
	   expected1.build       = "GRCh37";
	   expected1.buildBase   = "GRCh37";
	   expected1.dataset     = "Variants";

	   assertTrue(actual.size() > 30);
	   assertSame(expected1, actual.get(0));
   }

   private void assertSame(CatalogInfo expected, CatalogInfo actual) {
	   assertEquals(expected.status, actual.status);
	   assertEquals(expected.path, actual.path);
	   assertEquals(expected.lastUpdated, actual.lastUpdated);
	   assertEquals(expected.shortName, actual.shortName);
	   assertEquals(expected.indexes, actual.indexes);
	   assertEquals(expected.dataSource, actual.dataSource);
	   assertEquals(expected.version, actual.version);
	   assertEquals(expected.build, actual.build);
	   assertEquals(expected.buildBase, actual.buildBase);
	   assertEquals(expected.dataset, actual.dataset);
   }

   @Test
   public void testCatalogListWith10CatalogsAndMoreLines() throws CatalogFormatException
   {
      CatalogList list = new CatalogList("src/test/resources/testData/catalog_list_10_catalogs_with_more_lines.txt");
      assertEquals(10, list.size());
      Iterator<CatalogInfo> iterator = list.catalogInfoIterator();
      CatalogInfo info = iterator.next();
      assertTrue(info.isActive());
      assertEquals("1000 Genomes", info.getDataSource());
   }
   
   @Test
   public void testEmptyFile() throws CatalogFormatException, IOException
   {
      File file = temporaryFolder.newFile();
      CatalogList list = new CatalogList(file.getPath());
      assertEquals(0, list.size());
   }

   @Test
   public void testCatalogListDoesntExist() throws CatalogFormatException
   {
      expectedException.expect(CatalogFormatException.class);
      expectedException.expectMessage("does not exist");
      new CatalogList("catalogfilethatcouldntpossiblyexist");
   }

   @Test
   public void testCatalogListNotAFile() throws CatalogFormatException, IOException
   {
      File dir = temporaryFolder.newFolder();
      expectedException.expect(CatalogFormatException.class);
      expectedException.expectMessage("not a file");
      new CatalogList(dir.getPath());
   }

   @Test
   public void testCatalogListNotReadable() throws CatalogFormatException, IOException
   {
      File file = temporaryFolder.newFile();
      if (!file.setReadable(false))
      {
         throw new IOException(String.format("Couldn't set '%s' in test to not readable", file.getPath()));
      }
      expectedException.expect(CatalogFormatException.class);
      expectedException.expectMessage("is not readable");
      new CatalogList(file.getPath());
   }
}
