/**
 * bior_pipeline
 *
 * <p>@author Gregory Dougherty</p>
 * Copyright Mayo Clinic, 2014
 *
 */
package edu.mayo.bior.util;

import java.util.HashMap;
import java.util.Map;

/**
 * Get a {@linkplain Map} of Strings to Strings, the first a String to look for, the second a string 
 * to replace the first with when found.  If contents of text field do not match one of the strings, 
 * return the initial contents of the text field
 *
 * <p>@author Gregory Dougherty</p>
 */
public class TextLookupProcessor extends FieldProcessorBase
{
	private Map<String, String>	lookupTable;
	
	
	/**
	 * Constructor that takes two arrays of Strings, and puts them in the lookup table
	 * 
	 * @param find		Table of Strings to look for
	 * @param replace	Table of Strings to replace the {@code find} strings with
	 */
	@SuppressWarnings ("null")	// Compiler can't do math
	public TextLookupProcessor (String[] find, String[] replace)
	{
		int	findSize = (find == null) ? 0 : find.length;
		int	replaceSize = (replace == null) ? 0 : replace.length;
		int	size = Math.min (findSize, replaceSize);
		
		lookupTable = new HashMap<String, String> (size);
		
		for (int i = 0; i < size; ++i)
			lookupTable.put (find[i], replace[i]);
	}
	
	
	/**
	 * Constructor that takes a {@linkplain Map} of Strings to find, and replace, and stores 
	 * the data
	 * 
	 * @param lookupTable	{@linkplain Map} of Strings to find to Strings to replace the find strings with
	 */
	public TextLookupProcessor (Map<String, String>	lookupTable)
	{
		if (lookupTable == null)
		{
			this.lookupTable = new HashMap<String, String> ();
			return;
		}
		
		this.lookupTable = new HashMap<String, String> (lookupTable);
	}
	
	
	/**
	 * Add a String to look for, and what to replace it with
	 * 
	 * @param find		String to look for, ignored if null or empty
	 * @param replace	String to replace {@code find}.  If null effect is to stop looking for {@code find}
	 * @return	Previous value of {@code replace} for {@code find}, null if none
	 */
	public String addPair (String find, String replace)
	{
		if ((find == null) || find.isEmpty ())
			return null;
		
		if (replace == null)
			return lookupTable.remove (find);
		
		return lookupTable.put (find, replace);
	}


	/* (non-Javadoc)
	 * @see edu.mayo.bior.util.FieldProcessor#process(java.lang.String)
	 */
	public String process (String value)
	{
		String	result = lookupTable.get (value);
		
		if (result == null)
		{
			if (super.isBlank (value))
				return gCurBlankValue;
			
			return value;
		}
		
		return result;
	}
	
}
