package edu.mayo.bior.cli.cmd;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.Properties;

import org.apache.commons.cli.CommandLine;
import org.apache.commons.cli.Options;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;


import com.tinkerpop.pipes.Pipe;
import com.tinkerpop.pipes.util.Pipeline;

import edu.mayo.bior.pipeline.UnixStreamPipeline;
import edu.mayo.bior.util.validation.GoldenValidator;
import edu.mayo.cli.CommandPlugin;
import edu.mayo.cli.InvalidDataException;
import edu.mayo.pipes.SplitPipe;
import edu.mayo.pipes.JSON.InjectIntoJsonPipe;
import edu.mayo.pipes.JSON.inject.ColumnArrayInjector;
import edu.mayo.pipes.JSON.inject.ColumnInjector;
import edu.mayo.pipes.JSON.inject.Injector;
import edu.mayo.pipes.JSON.inject.LiteralInjector;
import edu.mayo.pipes.JSON.modify.JsonTypeExtended;
import edu.mayo.pipes.UNIX.CatPipe;
import edu.mayo.pipes.history.History;
import edu.mayo.pipes.history.HistoryInPipe;
import edu.mayo.pipes.history.HistoryOutPipe;
import edu.mayo.pipes.util.metadata.Metadata;

/**
 *
 * @author dquest
 */
public class Tab2JSONCommand implements CommandPlugin {
	
	private static Logger sLogger = LoggerFactory.getLogger(Tab2JSONCommand.class);
	
	private UnixStreamPipeline mPipeline = new UnixStreamPipeline();
	private String operation;
    
	public void init(Properties props) throws Exception {
		operation = props.getProperty("command.name");
	}
        
	public void execute(CommandLine line, Options opts) throws Exception {

        String config = "";
        if(line.hasOption('c')){
            config = line.getOptionValue('c');
        }
        
        parseAndInject(config);
	}
	

	protected void parseAndInject(String configFile) throws InvalidDataException {
        Injector[] injectors = parseConfigFile(configFile);

		Metadata metadata = new Metadata(operation);
		
		Pipe<String,  History>  preLogic  = new HistoryInPipe(metadata);
		Pipe<History, History>  logic     = new InjectIntoJsonPipe(true, injectors);
		Pipe<History, String>   postLogic = new HistoryOutPipe();

		mPipeline.execute(preLogic, logic, postLogic);        
	}



    /**
     * A config file takes the following tab-delimited form:
     * ColNum	Key JsonType InjectorType    Delimiter/Literal_Value   GoldenIdentifier
     * 
     * Here is what each of these values means:
     * 
     * 0) Column# (not specified) is the column number that we wish to get the data from.  
     * The parser will insert the keys in the order that they are found in the file
     * It will assume there is one key for every column in the tab delimited file
     * Followed by any literals that should be injected for all values in the set
     * 
     * 1) Key is the name of the identifier used to describe the value in the column
     * 
     * 2) JsonType is 	BOOLEAN, NUMBER, or STRING
     * This describes the types of values the column can take.
     * 
     * 3) An injector takes the value from the tab delimited file and puts it into the JSON column.
     * InjectorType is LITERAL, ARRAY, or COLUMN
     * LITERAL - every JSON will have the same value over the entire set
     * COLUMN - the data that appears in the COLUMN will be injected into the JSON (99% of the time this is what you want)
     * ARRAY - the data that appears in the column is actually a delimited array (e.g. values separated by a comma) and should be converted to a JSON array
     * 
     * 4) Delimiter/Literal_Value additional information to direct the injector
     * If the injector is a COLUMN this value is just a period (.)
     * If the injector is a LITERAL, this value is the value of the literal that should be injected.
     * If the injector is an ARRAY, this denotes the delimiter that should be used to parse the array.
     * 
     * 5) Golden Identifier (used to be called attribute)
     * If the column can be interpreted as a golden identifier (e.g. _landmark, _minBP) then place it here
     * else place a dot (.)
     * There can not be more than one golden attributed associated with a column, users will need to use tools
     * such as perl and awk to replicate the column before ingesting the data.
     * 
     * @param filename 
     */
	
	
    public Injector[] parseConfigFile(String filename) throws InvalidDataException {
        ArrayList<Injector> injectors = new ArrayList<Injector>();
        ArrayList<Injector> addQueue = new ArrayList<Injector>(); // golden attribute injectors that we need to add out of order
        Pipeline<String,ArrayList<String>> parse = new Pipeline<String,ArrayList<String>>(
                new CatPipe(),
                new SplitPipe("\t")
                );
        parse.setStarts(Arrays.asList(filename));
        for(int count = 0; parse.hasNext(); count++) {
            ArrayList<String> next = parse.next();
            if (next.size() < 6) {
            	sLogger.warn("WARNING: Tab2JSONCommand.parseConfigFile(): Invalid config file format; less that 6 columns. Ignoring line contents: " + next.toString());
            	continue;
            }
            Integer colNum = new Integer(next.get(0));
            String jsonKeyName = next.get(1);
            JsonTypeExtended jsonDataType = getJsonType(next.get(2), next.toString());
            String injectionType = next.get(3).toUpperCase();
            String delimiterOrLiteralVal = next.get(4);
            String goldenAttrUsage = next.get(5);
            
            if (!isValidInjectorType(injectionType)) {
            	throw new InvalidDataException("Tab2JSONCommand: JSON injector type specified is not valid: " + injectionType + " for config values: " + next.toString());
            }
            if (!goldenAttrUsage.equals(".")) { 
            	if (GoldenValidator.isValidGolden(goldenAttrUsage)) {
            		GoldenValidator.validateDataType(goldenAttrUsage, jsonDataType.toString());
            	} else {
                	throw new InvalidDataException("Tab2JSONCommand: JSON golden attribute specified is not valid: " + goldenAttrUsage + " for config values: " + next.toString());           		
            	}
    		} 
            
            Injector i = null;  Injector j = null;
            if(injectionType.equalsIgnoreCase(kInjectorTypes.LITERAL.name())){
                //e.g. new LiteralInjector(CoreAttributes._type.toString(), Type.VARIANT.toString(), JsonType.STRING),
                //String key, String value, JsonType type
                i = new LiteralInjector(jsonKeyName, delimiterOrLiteralVal, jsonDataType);
                if(!goldenAttrUsage.equalsIgnoreCase(".")){
                    j = new LiteralInjector(goldenAttrUsage, delimiterOrLiteralVal, jsonDataType);
                    addQueue.add(j);
                }
            } else if(injectionType.equalsIgnoreCase(kInjectorTypes.COLUMN.name())){
                //int column, String key, JsonType type
                i = new ColumnInjector(colNum, jsonKeyName, jsonDataType);
                if(!goldenAttrUsage.equalsIgnoreCase(".")){
                    j = new ColumnInjector(colNum, goldenAttrUsage, jsonDataType);
                    addQueue.add(j);
                }
            } else if(injectionType.equalsIgnoreCase(kInjectorTypes.ARRAY.name())){
                //int column, String key, JsonType type, String delimiterRegex, boolean stripWhitespace
                i= new ColumnArrayInjector(colNum, jsonKeyName, jsonDataType, delimiterOrLiteralVal, /*isStripWhitespace=*/true, /*isDelimiterRegex=*/false);
                if(!goldenAttrUsage.equalsIgnoreCase(".")){
                    j = new ColumnArrayInjector(colNum, goldenAttrUsage, jsonDataType, delimiterOrLiteralVal, /*isStripWhitespace=*/true, /*isDelimiterRegex=*/false);
                    addQueue.add(j);
                }
            } 
            
            injectors.add(i);
            
        }
        for(Injector i : addQueue){
            injectors.add(i);
        }
        Injector[] ret = new Injector[injectors.size()];
        ret = injectors.toArray(ret);

        return ret;
        
    }
    

    /** Allow all of the JsonTypeExtended values as well as "NUMBER" for backwards compatibility (this should convert to Float now) 
     * @param jsonConfigLine 
     * @throws InvalidDataException */
    private JsonTypeExtended getJsonType(String jsonTypeStr, String jsonConfigLine) throws InvalidDataException {
    	if( "NUMBER".equalsIgnoreCase(jsonTypeStr) ) {
    		return JsonTypeExtended.FLOAT;
    	}
    	
    	try {
    		return JsonTypeExtended.valueOf(jsonTypeStr.toUpperCase());
    	} catch(Exception e) {
        	throw new InvalidDataException("Tab2JSONCommand: JSON data type specified is not valid: " + jsonTypeStr + " for config values: " + jsonConfigLine);
        }
	}

	private static enum kInjectorTypes {COLUMN, LITERAL, ARRAY};
    private boolean isValidInjectorType(String inputInjector) {
    	for (kInjectorTypes each : kInjectorTypes.values()) {
    		if (each.name().equals(inputInjector)) {
    			return true;
    		}
    	}
    	return false;
    }

}
