package edu.mayo.bior.cli.cmd;

import java.io.BufferedWriter;
import java.io.File;
import java.io.FileWriter;
import java.io.IOException;
import java.net.URISyntaxException;
import java.util.Arrays;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Properties;

import edu.mayo.bior.util.progress.NumLineProgressHandler;
import org.apache.commons.cli.CommandLine;
import org.apache.commons.cli.Options;
import org.apache.commons.configuration.ConfigurationException;
import org.apache.commons.io.FileUtils;

import edu.mayo.bior.buildcatalog.BuildCatalog;
import edu.mayo.bior.buildcatalog.StepLogger;
import edu.mayo.bior.catalog.CatalogMetadataConstant;
import edu.mayo.bior.catalog.GoldenAttribute;
import edu.mayo.bior.pipeline.createCatalogProps.ColumnMetaFromCatalogCrawling;
import edu.mayo.bior.pipeline.createCatalogProps.ColumnMetaFromVcf;
import edu.mayo.bior.pipeline.exception.CatalogMetadataInputException;
import edu.mayo.bior.util.CatalogUtils;
import edu.mayo.bior.util.ClasspathUtil;
import edu.mayo.cli.CommandPlugin;
import edu.mayo.cli.InvalidDataException;
import edu.mayo.cli.InvalidOptionArgValueException;
import edu.mayo.pipes.history.ColumnMetaData;
import edu.mayo.pipes.history.ColumnMetaDataOperations;
import edu.mayo.pipes.util.metadata.AddMetadataLines;
import edu.mayo.pipes.util.metadata.AddMetadataLines.BiorMetaControlledVocabulary;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/** Given a catalog path, construct the metadata files associated with that catalog:
 *  <catalog>.datasource.properties
 *  <catalog>.columns.properties
 *  
 * @author Surendra Konathala, Michael Meiners (m054457)
 * Date created: Aug 1, 2013
 */
public class CreateCatalogPropsCommand implements CommandPlugin
{
   private static final Logger sLogger = LoggerFactory.getLogger(CreateCatalogPropsCommand.class);
   
   /** Report status every X lines.   NOTE: This is public static so it can be used by tests to set the callback reporting and test large row counts (> Integer.MAX_VALUE) */
   public static long PROGRESS_HANDLER_NUM_LINES_CALLBACK_ON = 10000;

   /** Catalog path */
	public static final char OPTION_CATALOG = 'd';
	
	/** Catalog's original vcf source file (optional) */
	public static final char OPTION_ORIGINAL_VCF = 'v';
	
	/** Target directory (optional) - by default it is the same as the catalog */
	public static final char OPTION_TARGET_DIR = 't';
	
	/** BIOR vocabulary is defined here */
	private List<BiorMetaControlledVocabulary> biorMetaControlledVocabulary = Arrays.asList(BiorMetaControlledVocabulary.values());
	
	
	/** Should we print progress? */
	private boolean mIsVerbose = true;
	
	private StepLogger mStepLogger;
	

	public void init(Properties props) throws Exception {
	}
	
	public void setStepLogger(StepLogger stepLogger) {
		mStepLogger = stepLogger;
	}
	
	public void execute(CommandLine line, Options opts) throws InvalidDataException, IOException {	
		// Catalog path and key are required		
		String catalogBgzipPath = line.getOptionValue(OPTION_CATALOG);
		CatalogUtils.warnIfDeprecated(catalogBgzipPath);
		
		// Original VCF file that the catalog was constructed from (optional, and may not apply)
		// null if not applicable
		boolean isVcfSpecified = line.hasOption(OPTION_ORIGINAL_VCF);
		String catalogOriginalVcfPath = line.getOptionValue(OPTION_ORIGINAL_VCF);
		
		String targetDir = line.getOptionValue(OPTION_TARGET_DIR);
		
		try
		{
			execNoCmd(catalogBgzipPath, catalogOriginalVcfPath, targetDir, isVcfSpecified, new ProgressHandler());
		}
		catch (IOException ioe)
		{
			throw new InvalidDataException("Error accessing one of the files: " + ioe.getMessage());
		} catch (URISyntaxException e) {
			throw new InvalidDataException("Could not find the columns defaults properties file: " + e.getMessage());
		} catch (ConfigurationException e) {
			throw new InvalidDataException("Error modifying the properties file: " + e.getMessage());
		}
	}
	
	public void execNoCmd(String catalogBgzipPath, String catalogOriginalVcfPath, String targetDirPath,
			boolean isVcfSpecified, NumLineProgressHandler progressHandler)
			throws InvalidDataException, IOException, URISyntaxException, ConfigurationException
	{
		File catalogFile = new File(catalogBgzipPath);
		
		// Throw exception if catalog does not exist
		if( ! catalogFile.exists() )
			throw new IOException("Catalog file could not be found: " + catalogBgzipPath);

		// If the target dir is not specified, then use the same dir as the catalog
		if( targetDirPath == null || targetDirPath.length() == 0 )
			targetDirPath = new File(catalogBgzipPath).getCanonicalFile().getParent();
		
		// Throw exception if target directory does not exist
		File targetDir = new File(targetDirPath).getCanonicalFile();
		if( ! targetDir.exists() )
			throw new IOException("Target directory does not exist: " + targetDir);
		
		// Throw exception if target dir is NOT a directory
		if( ! targetDir.isDirectory() )
			throw new IOException("Target directory is not a directory: " + targetDir);

		// Throw exception if target dir is NOT writable
		if( ! targetDir.canWrite() )
			throw new IOException("Target directory is not writable: " + targetDir);
		
		String catalogFullFilename = catalogFile.getName();
		String catalogFilenamePrefix = catalogFullFilename;
		if( catalogFullFilename.endsWith(".tsv") )
			catalogFilenamePrefix = catalogFullFilename.substring(0, catalogFullFilename.lastIndexOf(".tsv"));
		else if( catalogFullFilename.endsWith(".tsv.bgz") )
			catalogFilenamePrefix = catalogFullFilename.substring(0, catalogFullFilename.lastIndexOf(".tsv.bgz"));
		
		File datasrcPropsFile = new File(targetDir.getCanonicalPath() + File.separator + catalogFilenamePrefix + CatalogMetadataConstant.DATASRC_PROPS_SUFFIX);
		File columnsPropsFile = new File(targetDir.getCanonicalPath() + File.separator + catalogFilenamePrefix + CatalogMetadataConstant.COLUMN_INFO_SUFFIX);
		
		File blacklistFile = new File(targetDir.getCanonicalPath() + File.separator + catalogFilenamePrefix + CatalogMetadataConstant.BLACKLIST_SUFFIX);
		File blacklistBiorwebFile = new File(targetDir.getCanonicalPath() + File.separator + catalogFilenamePrefix + CatalogMetadataConstant.BLACKLIST_BIORWEB_SUFFIX);
		
		// Throw exception if datasource or columns properties file is an existing directory
		if( datasrcPropsFile.isDirectory() )
			throw new IOException("Datasource properties file is an existing directory.  It should be a file");
		if( columnsPropsFile.isDirectory() )
			throw new IOException("Columns properties file is an existing directory.  It should be a file");
		
		// If original vcf was specified, but it does not exist, then throw exception
		File originalVcfFile  = isVcfSpecified ? new File(catalogOriginalVcfPath) : null;
		if( isVcfSpecified  &&  (originalVcfFile == null || ! originalVcfFile.exists()) )
			throw new IOException("Orginal VCF source file that the catalog was built from does not exist!: " + catalogOriginalVcfPath);
		
		
		// If crawling the catalog (NOT using the original VCF file to parse the info), then check size of catalog and warn user if it is big
		boolean isCrawlCatalog = ! isVcfSpecified;
		warnIfLargeCatalog(catalogFile, isCrawlCatalog);

		
		// Create the datasource.props file - NOTE: This will ONLY create the default datasource.properties file (not filled in from build_info.txt, which will be done later in the merge step)
		createDatasourcePropsFile(datasrcPropsFile);
			
		// Create the columns.props file
		createColumnPropsFile(columnsPropsFile, catalogFile, originalVcfFile, isVcfSpecified, progressHandler);
		
		// Create the blacklist file based on the columns.props file
		try {
			File NewColumnsTsvFileObj = new File(columnsPropsFile.getCanonicalPath());
			createBlackListFiles(NewColumnsTsvFileObj, blacklistFile, blacklistBiorwebFile);
			if (!blacklistFile.exists() || !blacklistBiorwebFile.exists()) {
				throw new IOException("Unable to create the blacklist or blacklist biorweb file successfully.");
			} 
		} catch (CatalogMetadataInputException i) {
			throw new InvalidDataException(i.getMessage());
		} catch (IOException io) {
			throw io;
		}
		
		sLogger.info("Done.");
	}

	

	private void warnIfLargeCatalog(File catalogFile, boolean isCrawlCatalog) {
		if( isCrawlCatalog ) {
			final long MB = 1 * 1024 * 1024;
			boolean isZip = catalogFile.getName().toLowerCase().endsWith(".gz")  ||  catalogFile.getName().toLowerCase().endsWith(".bgz");
			long sizeMB = catalogFile.length() / MB;
			boolean isBig = isZip  ?  (sizeMB > 50)  :  (sizeMB > 200);
			if( isBig ) {
				String msg = "  WARNING: Catalog is large (" + (catalogFile.length() / MB) + "MB).  This may take a while to process...";
				System.err.println(msg);
				sLogger.warn(msg);
			}
		}
	}

	/**
	 * Create the datasource.properties file
	 * @param datasourcePropsFile
	 * @throws InvalidOptionArgValueException
	 * @throws InvalidDataException
	 * @throws IOException
	 * @throws ConfigurationException 
	 */
	protected void createDatasourcePropsFile(File datasourcePropsFile) throws InvalidDataException, IOException, ConfigurationException {

		String catalogNamePrefix = datasourcePropsFile.getName().replace(CatalogMetadataConstant.DATASRC_PROPS_SUFFIX, "");
        
		// jaj: 5/30/15: Path cannot be found in distribution. there is one here: conf/datasource.properties.default
        //               I believe this was the intended file to be preferenced here.
        //String datasourceProps = FileUtils.readFileToString(new File("src/main/resources/datasource.properties.default"))
        //                                                     .replace("_CATALOG_PREFIX_", catalogNamePrefix);
       
        //String deployedDistributionDefault = "conf/datasource.properties.default";  // deployed location
        // defaultDataSrcProps = new File(deployedDistributionDefault);
		
		String deployedDistributionDefault = "datasource.properties.default";   // $BIOR_LITE_HOME/conf/datasource.properties.default
		String uri = this.getClass().getClassLoader().getResource(deployedDistributionDefault).getPath();
		File defaultDataSrcProps = null;
		if (uri != null && uri.length() > 0) { 
			defaultDataSrcProps = new File(uri);
		} else {
            String testEnvDistributionDefault = "src/main/resources/datasource.properties.default"; // junit test location
            File testEnvDefaultDataSrcProps = new File(testEnvDistributionDefault);
            if (testEnvDefaultDataSrcProps.canRead()) {
                  defaultDataSrcProps = testEnvDefaultDataSrcProps;
            } else {
            	System.err.println("Unable to locate file:" + deployedDistributionDefault);
            	throw new IOException("Unable to locate file:" + deployedDistributionDefault);
            }
		}
		
        String datasourcePropsContents = FileUtils.readFileToString(defaultDataSrcProps).replace("_CATALOG_PREFIX_", catalogNamePrefix);
        
        // Write the .default file
        File dataSrcDefaultFile = new File(datasourcePropsFile.getParentFile().getCanonicalPath() + "/" + BuildCatalog.BUILD_SUBDIR, datasourcePropsFile.getName() + ".default");
        FileUtils.writeStringToFile(dataSrcDefaultFile, datasourcePropsContents);
        
        // Only write the current file if it doesn't already exist:
        // WARNING:  We don't want to overwrite the file if it already exists, as the user may have changed it
        if( ! datasourcePropsFile.exists() ) {
        	FileUtils.writeStringToFile(datasourcePropsFile, datasourcePropsContents);
        	logAll("Datasource properties file created at: " + datasourcePropsFile.getCanonicalPath());
        } else {
        	logAll("Datasource properties file already exists.  No change.");
        }

	}
	
	/** Log to summary, step log, and bior log file */
	private void logAll(String msg) {
		if( mStepLogger != null ) {
			mStepLogger.logAndSummary(msg);
		}
		sLogger.info(msg);
	}


	/**
	 * Create the columns.tsv file
	 * @param columnPropsFile  The columns.tsv file to write to
	 * @param originalVcfFile  Original VCF from which we can extract column info
	 * @param catalogFile  Location of catalog from which to base locations of other files
	 * @param isVcfSpecified  Determines whether the originalVcfFile was specified and should be used to extract info from 
	 * @param progressHandler  Handler to display progress to the user
	 * @throws InvalidOptionArgValueException
	 * @throws InvalidDataException
	 * @throws IOException
	 * @throws URISyntaxException 
	 */
	protected void createColumnPropsFile(File columnPropsFile, File catalogFile, File originalVcfFile,
			boolean isVcfSpecified, NumLineProgressHandler progressHandler)
 	 throws InvalidDataException, IOException, URISyntaxException
 	{
	    List<ColumnMetaData> colMetaList = mergeCrawlerWithVcf(catalogFile, originalVcfFile, isVcfSpecified, progressHandler);
	    colMetaList = mergeWithDefaults(colMetaList);
	    
	    StringBuilder content = getColumnContents(columnPropsFile, colMetaList);

	    // Write to the .default file
	    File defaultColumnsTsvFile = new File(columnPropsFile.getParentFile().getCanonicalPath() + "/" + BuildCatalog.BUILD_SUBDIR, columnPropsFile.getName() + ".default");
	    FileUtils.writeStringToFile(defaultColumnsTsvFile, content.toString());
	   
	    // Write the props file, but only if it doesn't already exist.
	    // WARNING: don't overwrite it if it already exists as the user may have changed it
		if( ! columnPropsFile.exists() ) {
			columnPropsFile.createNewFile();
			FileUtils.writeStringToFile(columnPropsFile, content.toString());
			logAll("Columns properties file created at: " + columnPropsFile.getCanonicalPath());
		} else {
			logAll("Columns properties file already exists.  No change.");
		}
	}

	/**
	 * Create columns.tsv.blacklist file which tells programs which columns should not be displayed to the user in applications that use BioR catalogs
	 * @param columnPropsFile  The columns.tsv file to scan for column names
	 * @param blacklistPropsFile  The columns.tsv.blacklist file to write.  These columns should not be displayed to the user
	 * @param blacklistPropsFile  The columns.tsv.blacklist.biorweb file to write.  These columns should not be displayed to the user in the BioRWeb application
	 * @throws CatalogMetadataInputException When problem accessing and loading columns.tsv file content.
	 * @throws IOException When problems writing the blacklist file.
	 */
	protected void createBlackListFiles(File columnPropsFile, File blacklistPropsFile, File blacklistBiorwebFile) throws CatalogMetadataInputException, IOException {
		String blacklistTopRow = "### These columns are to generally be ignored within UIs as they are duplicates or are normalized versions of the original columns.\n";
		String blacklistBiorwebTopRow = "### These fields will NOT be shown in BioRWeb.  All others will.\n";
		
		// Build up the blacklist content
		StringBuilder blackListContent = new StringBuilder(
				"### If this file is NOT present, then assume that all fields, except for the golden attributes (those beginning with _) should be shown.\n" +
				"### If this file is present, but contains no columns, then assume that all fields should be shown.\n"
				);


		HashMap<String,ColumnMetaData> columnKeyToMetaMap = null;
		try {
			columnKeyToMetaMap = new ColumnMetaDataOperations(columnPropsFile).load();
		} catch (IOException e) {
			throw new CatalogMetadataInputException("Could not create blacklist file. Could not get columns.tsv info from file: " + columnPropsFile.getCanonicalPath() + "\nError: " + e.getMessage());
		} 
		
		for (String colName : columnKeyToMetaMap.keySet()) {
			if( isGoldenAttribute(colName) )
				blackListContent.append(colName + "\n");
		}
		
		// Write the *default* files
		String buildDir = blacklistPropsFile.getParentFile().getCanonicalPath() + "/" + BuildCatalog.BUILD_SUBDIR;
		File blacklistPropsFileDefault = new File(buildDir,  blacklistPropsFile.getName() + ".default");
		FileUtils.writeStringToFile(blacklistPropsFileDefault, blacklistTopRow + blackListContent.toString());
		File blacklistBiorwebFileDefault = new File(buildDir,  blacklistBiorwebFile.getName() + ".default");
		FileUtils.writeStringToFile(blacklistBiorwebFileDefault, blacklistBiorwebTopRow + blackListContent.toString());
		
		// Write the *current* blacklist files (if they don't already exist)
		if( ! blacklistPropsFile.exists() ) {
			FileUtils.writeStringToFile(blacklistPropsFile, blacklistTopRow + blackListContent.toString());
			logAll("Blacklist properties file created at: " + blacklistPropsFile.getCanonicalPath());
		} else {
			logAll("Blacklist properties file already exists.  No change.");			
		}
			

		if( ! blacklistBiorwebFile.exists() ) {
			FileUtils.writeStringToFile(blacklistBiorwebFile,   blacklistBiorwebTopRow + blackListContent.toString());
			logAll("Blacklist (BioRWeb) properties file created at: " + blacklistBiorwebFile.getCanonicalPath());
		} else {
			logAll("Blacklist (BioRWeb) properties file already exists.  No change.");
		}
	}
	
	private boolean isGoldenAttribute(String colName) {
		GoldenAttribute[] goldenAttributes = GoldenAttribute.values();
		for (GoldenAttribute golden : goldenAttributes) {
			if (colName.equals(golden.name())) {
				return true;
			}
		}
		return false;
	}

	private List<ColumnMetaData> mergeCrawlerWithVcf(File catalogFile, File originalVcfFile,
		boolean isVcfSpecified, NumLineProgressHandler progressHandler) throws IOException, InvalidDataException
	{
		// First crawl the catalog to get all possible values (since this may have BioR or VCF keys that would not appear in the VCF INFO fields)
		ColumnMetaFromCatalogCrawling catalogCrawler = new ColumnMetaFromCatalogCrawling();
	    List<ColumnMetaData> colMetaList = catalogCrawler.getColumnMetadata(catalogFile.getCanonicalPath(), progressHandler);
	    
	    // If there IS a VCF file specified, then we want to use the values derived from that, 
	    // and add the VCF values to the list, replacing any values obtained from crawling
	    // (since those are only a best guess)
	    if( isVcfSpecified && originalVcfFile != null  &&  originalVcfFile.exists() ) {
	    	List<ColumnMetaData> fromVcfColMetaList = new ColumnMetaFromVcf().getColumnMetadata(originalVcfFile.getCanonicalPath());
	    	for(ColumnMetaData colMetaVcf : fromVcfColMetaList) {
    			// If the metadata is already in the list from the catalog crawler, 
    			// THEN remove it since the vcf one should be more accurate
    			// This should also add in any fields from BioR or the VCF first 7 columns
	    		for(int i=colMetaList.size()-1; i>=0; i--) {
	    			if( colMetaList.get(i).getColumnName().equals(colMetaVcf.getColumnName()) ) {
	    				colMetaList.remove(i);
	    			}
	    		}
	    		// Now, add the metadata from the vcf parser
	    		colMetaList.add(colMetaVcf);
	    	}
	    }
		return colMetaList;
	}
	
	/** Merge with default column meta data for BioR and VCF fields.  
	 *  The defaults take preference over the crawled and VCF values.
	 *  NOTE: The list is modified and returned.
	 * @param colMetaList
	 * @return
	 * @throws URISyntaxException 
	 * @throws IOException
	 */
	private List<ColumnMetaData> mergeWithDefaults(List<ColumnMetaData> colMetaList) throws URISyntaxException, IOException {
		// Load the defaults (known BioR and VCF columns)
	    File colDefaultsProps = ClasspathUtil.loadResource("/allCatalogs.columns.tsv");
	    HashMap<String, ColumnMetaData> defaultsColNameToMetaMap = new AddMetadataLines().parseColumnProperties(colDefaultsProps.getCanonicalPath());
	    for(int i=0; i < colMetaList.size(); i++) {
	    	String colName = colMetaList.get(i).getColumnName();
	    	
	    	// If the column name is present but matches one we have in the defaults file, then use the default
	    	if( defaultsColNameToMetaMap.containsKey(colName) )
	    		colMetaList.set(i, defaultsColNameToMetaMap.get(colName));
	    }
	    return colMetaList;
	}

	private StringBuilder getColumnContents(File columnPropsFile, List<ColumnMetaData> colMetaList)
	{
	    ColumnMetaDataOperations colMetaOps = new ColumnMetaDataOperations();
	    StringBuilder content = new StringBuilder( colMetaOps.getLinesAsString(colMetaOps.getDefaultColumnsTsvHeader()) );

	    // Sort the list
	    Collections.sort(colMetaList);
	    // Merge with defaults - adding to the content string
	    for(ColumnMetaData colMeta : colMetaList) {
	    	content.append(colMeta.toString() + "\n");
	    }
		return content;
	}

	
	public class ProgressHandler implements NumLineProgressHandler {
		public long NUM_LINES_CALLBACK_ON = CreateCatalogPropsCommand.PROGRESS_HANDLER_NUM_LINES_CALLBACK_ON;
		
		@Override
		public long getNumLinesToCallbackOn() {
			return NUM_LINES_CALLBACK_ON;
		}

		@Override
		public void startingRead() {
			// Default:	"Reading lines from catalog:  .=10k  o=100k  |=1M  X=10M"
			System.out.println("Reading lines from catalog:  .=" + NUM_LINES_CALLBACK_ON
														+ "  o=" + (NUM_LINES_CALLBACK_ON * 10)
														+ "  |=" + (NUM_LINES_CALLBACK_ON * 100)
														+ "  X=" + (NUM_LINES_CALLBACK_ON * 1000));
		}	

		@Override
		public void readNumLines(long numLines, long numLinesInFile) {
			if( numLines % (NUM_LINES_CALLBACK_ON * 1000) == 0 ) {
				System.out.println("X");
			} else if( numLines % (NUM_LINES_CALLBACK_ON * 100) == 0 ) {
				System.out.println("|");
			} else if( numLines % (NUM_LINES_CALLBACK_ON * 10) == 0 ) {
				System.out.print("o");
			} else if( numLines % NUM_LINES_CALLBACK_ON == 0 ) {
				System.out.print(".");
			}
			if (numLines % 100000 == 0) {
				sLogger.info(String.format("Read %d lines of catalog", numLines));
			}
		}

		@Override
		public void readAllLines(long lineNumber) {
			System.out.println("\n# catalog lines crawled: " + lineNumber);
			sLogger.info(String.format("Finished reading %d lines of catalog", lineNumber));
		}
	}

}
