/*
 * To change this template, choose Tools | Templates
 * and open the template in the editor.
 */
package edu.mayo.bior.cli.cmd;

import java.io.File;
import java.util.Properties;

import org.apache.commons.cli.CommandLine;
import org.apache.commons.cli.Options;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import edu.mayo.bior.buildcatalog.TempDirectory;
import edu.mayo.bior.catalog.HumanBuildAssembly;
import edu.mayo.bior.pipeline.createcatalog.TjsonToCatalog;
import edu.mayo.cli.CommandPlugin;


/**
 *
 * @author Michael Meiners (m054457), Daniel Quest (m102417)
 * 
 * Assumes that the user does not know the structure required for a catalog
 * e.g.
 * _landmark    _minBP  _maxBP  JSON
 * 
 * it will format a TJSON input file by cutting the JSON, 
 * drilling out the golden identifiers,
 * fixing any dots that result, to conform to the standard
 * removing any header
 * and prepping the file for bgzip followed by tabix
 * 
 */
public class CreateCatalogCommand implements CommandPlugin {

    private static final Logger sLogger = LoggerFactory.getLogger(CreateCatalogCommand.class);

    private static final char OPTION_INPUT_FILE  = 'i';
    private static final char OPTION_OUTPUT_FILE = 'o';
    private static final char OPTION_JSON_COLUMN = 'c';
    private static final char OPTION_NO_SORT     = 'n';
    private static final char OPTION_CHROM_SORT_FILE = 'f';
    private static final char OPTION_JSON_ONLY   = 'j';
    private static final char OPTION_TEMP_DIR    = 't';
    private static final char OPTION_KEEP_CHROM_UNCHANGED = 'k';
    private static final char OPTION_BUILD_ASSEMBLY = 'b';

    
    
    public void init(Properties props) throws Exception {

    }

    public void execute(CommandLine cl, Options options) throws Exception {
        String infile = "";
        String bgzipCatalogOutFile = "";
        if (cl.hasOption(OPTION_INPUT_FILE)) {
                infile = cl.getOptionValue(OPTION_INPUT_FILE);
        }
        
        if (cl.hasOption(OPTION_OUTPUT_FILE)) {
        	bgzipCatalogOutFile = cl.getOptionValue(OPTION_OUTPUT_FILE);
            if( ! bgzipCatalogOutFile.endsWith(".tsv.bgz") ) {
            	throw new IllegalArgumentException("Error: Catalog file should end with extension .tsv.bgz");
            }
            
            if( new File(bgzipCatalogOutFile).exists() ) {
            	throw new IllegalArgumentException("Error: Output file already exists: " + bgzipCatalogOutFile);
            }
        }
        
        // Get the JSON column that we will pull the chrom,min,max columns from and use as the main JSON column for the catalog
        int jsoncol = -1;
        if (cl.hasOption(OPTION_JSON_COLUMN)) {
            jsoncol = new Integer(cl.getOptionValue(OPTION_JSON_COLUMN));
        }

        
        String chromSortOrderFilePath = null;
        if( cl.hasOption(OPTION_CHROM_SORT_FILE) ) {
        	chromSortOrderFilePath = cl.getOptionValue(OPTION_CHROM_SORT_FILE);
        }

        // Is the catalog made up of only the json column?
        boolean isJsonOnly = cl.hasOption(OPTION_JSON_ONLY);

        // Sort by default
        boolean isSort = ! cl.hasOption(OPTION_NO_SORT)  &&  ! isJsonOnly;
        
        // Use Java temp directory by default, but user could specify
        String tempDir = TempDirectory.findBaseTempDir("bior_create_catalog_tmp");
        if( cl.hasOption(OPTION_TEMP_DIR) ) {
        	tempDir = cl.getOptionValue(OPTION_TEMP_DIR);
        }

        // Get the build assembly  ("GRCh37" vs "GRCh38")
        HumanBuildAssembly buildAssembly = getBuildAssembly(cl);
        
        boolean isModifyChrom = ! cl.hasOption(OPTION_KEEP_CHROM_UNCHANGED);
        
        boolean isInputFileAlreadyInFinalCtgFormatAndBgzipped = false;
        
        TjsonToCatalog tjsonToCatalog = new TjsonToCatalog();
        tjsonToCatalog.createCatalog(infile, bgzipCatalogOutFile, isSort, chromSortOrderFilePath, jsoncol, 
        		isJsonOnly, tempDir, buildAssembly, isModifyChrom, isInputFileAlreadyInFinalCtgFormatAndBgzipped);
        
    }

	private HumanBuildAssembly getBuildAssembly(CommandLine cmdLine) throws Exception {
        if( cmdLine.hasOption(OPTION_BUILD_ASSEMBLY) ) {
        	String buildOption = cmdLine.getOptionValue(OPTION_BUILD_ASSEMBLY);
        	String errMsg = "Build Assembly not recognized: " + cmdLine.getOptionValue(OPTION_BUILD_ASSEMBLY) + "\n"
        			+ "Valid build assemblies:\n";
        	for(HumanBuildAssembly build : HumanBuildAssembly.values()) {
        		if( buildOption.equalsIgnoreCase(build.toString()) )
        			return build;
        		errMsg += "  " + build.toString() + "\n";
        	}
        	// None found, so show error with list of possible values:
        	sLogger.error(errMsg);
        	throw new Exception(errMsg);
        }
		// Default: GRCh37
        else {
        	return HumanBuildAssembly.GRCh37;
        }
	}


    
}
