package edu.mayo.bior.catalog.verification;

import java.io.*;
import java.util.*;

import org.apache.commons.io.output.NullWriter;
import org.apache.commons.lang.StringUtils;



public class MessageLogger
{
   private Writer writer;
   private long warningCount = 0;
   private long infoCount = 0;
   private File logFile = null;
   private ErrorCount errorCounter = new ErrorCount();


   /**
    * @param readerList A list of Reader objects that are reading output created by using MessageLogger to log WARNING, INFO, ERROR
    * @return MessageLogger that you can query. It is reconstructed from the contents read line by line from the readers.
    *    You can call log* methods but it won't go anywhere as the messages are being logged to a NullWriter
    * @throws IOException if there is some issue reading the stuff
    */
   public static MessageLogger loggerFromOutputList(List<Reader> readerList) throws IOException
   {
      if (readerList.isEmpty())
      {
         return null;
      }

      Iterator<Reader> iter = readerList.iterator();
      MessageLogger messageLogger = loggerFromOutput(iter.next());
      while (iter.hasNext())
      {
         addToLoggerFromOutput(messageLogger, iter.next());
      }
      return messageLogger;
   }

   /**
    * @param reader A Reader object that is reading output created by using MessageLogger to log WARNING, INFO, ERROR
    * @return MessageLogger that you can query. It is reconstructed from the contents read line by line from reader.
    *    You can call log* methods but it won't go anywhere as the messages are being logged to a NullWriter
    * @throws IOException if there is some issue reading the stuff
    */
   public static MessageLogger loggerFromOutput(Reader reader) throws IOException
   {
      MessageLogger messageLogger = new MessageLogger(new NullWriter());
      addToLoggerFromOutput(messageLogger, reader);
      return messageLogger;
   }

   private static void addToLoggerFromOutput(MessageLogger messageLogger, Reader reader) throws IOException
   {
      BufferedReader bufferedReader = new BufferedReader(reader);
      String line;
      try
      {
         while ((line = bufferedReader.readLine()) != null)
         {
            Message message = Message.messageFromString(line);
            if (message != null)
            {
               if (message.isError())
               {
                  messageLogger.logError(message);
               }
               else
               {
                  messageLogger.logMessage(message);
               }
            }
         }
      }
      finally
      {
         bufferedReader.close();
      }
   }

   public MessageLogger(Writer writer)
   {
      this.writer = writer;
   }

   public MessageLogger(String path) throws IOException
   {
      logFile = new File(path);
      writer = new BufferedWriter(new FileWriter(logFile));
   }

   public File getLogFile()
   {
      return logFile;
   }

   public void logError(String msg, int code)
   {
      logError(new Message(msg, LogLevel.ERROR, code));
   }

   public void logWarning(String msg)
   {
      logMessage(new Message(msg, LogLevel.WARNING));
   }

   public void logInfo(String msg)
   {
      logMessage(new Message(msg, LogLevel.INFO));
   }

   private void logError(Message error)
   {
      errorCounter.add(error);
      writeMessage(error);
   }

   private void logMessage(Message message)
   {
      if (message.isWarning())
      {
         warningCount++;
      }
      else if (message.isInfo())
      {
         infoCount++;
      }
      writeMessage(message);
   }

   private void writeMessage(Message message)
   {
      try
      {
         writer.write(String.format("%s%n", message));
         writer.flush();
      }
      catch (IOException e)
      {
         // TODO - handle this a lot better
         throw new RuntimeException("Problem writing to verify output: " + e.getMessage());
      }
   }


   public long numErrors()
   {
      return errorCounter.numErrors();
   }

   public long numErrorsForCode(int code)
   {
      return errorCounter.numErrorsForCode(code);
   }

   public long numWarnings()
   {
      return warningCount;
   }

   public long numInfos()
   {
      return infoCount;
   }

   public long numMessages()
   {
      return numErrors() + numWarnings() + numInfos();
   }

   public boolean hasErrors()
   {
      return numErrors() > 0;
   }

   public boolean hasWarnings()
   {
      return numWarnings() > 0;
   }

   private class ErrorCount
   {
	  // Map<code,count>
      private Map<Integer, Long> counter = new HashMap<Integer, Long>();

      public void add(Message error)
      {
         int code = error.getCode();
         Long count = counter.get(code);
         if (count == null)
         {
            count = 0L;
         }
         count++;
         counter.put(code, count);
      }

      public long numErrors()
      {
         long totalCount = 0;
         for (int code: counter.keySet())
         {
            totalCount += numErrorsForCode(code);
         }
         return totalCount;
      }

      public long numErrorsForCode(int code)
      {
         Long count = counter.get(code);
         return (count == null) ? 0 : count;
      }
   }

   
   private static class Message
   {
      private static final int ERROR_NO_CODE = -99;
      private String msg;
      private LogLevel level;
      private Integer code;

      public static Message messageFromString(String str)
      {
         String splitOn = ": ";
         String[] fields = str.split(splitOn);
         if (fields.length < 2)
         {
            return null;
         }
         List<String> parts2ThruN = new ArrayList<String>();
         for (int i = 1; i < fields.length; i++)
         {
            parts2ThruN.add(fields[i]);
         }
         String msg = StringUtils.join(parts2ThruN, splitOn);

         String part1 = fields[0];
         String[] part1Fields = part1.split(" ");
         if (part1Fields.length == 0 || part1Fields.length > 2)
         {
            return null;
         }
         String levelLabel = part1Fields[0];
         LogLevel level = null;
         if (LogLevel.ERROR.getLabel().equals(levelLabel))
         {
            level = LogLevel.ERROR;
         }
         else if (LogLevel.WARNING.getLabel().equals(levelLabel))
         {
            level = LogLevel.WARNING;
         }
         else if (LogLevel.INFO.getLabel().equals(levelLabel))
         {
            level = LogLevel.INFO;
         }
         else
         {
            return null;
         }
         Integer code = null;
         if (part1Fields.length == 2)
         {
            String intString = part1Fields[1];
            try
            {
               code = Integer.parseInt(intString);
            }
            catch (NumberFormatException e)
            {
               return null;
            }
         }
         else if (LogLevel.ERROR.equals(level))
         {
            code = ERROR_NO_CODE;
         }

         if (LogLevel.ERROR.equals(level))
         {
            if (code == null)
            {
               return null;
            }
         }
         else if (code != null)
         {
            return null;
         }

         return new Message(msg, level, code);
      }

      public Message(String msg, LogLevel level)
      {
         this(msg, level, null);
      }

      public Message(String msg, LogLevel level, Integer code)
      {
         // Must assign before the checkArgument call
         this.level = level;
         this.msg = msg;
         this.code = code;
      }

      public String getMsg()
      {
         return msg;
      }

      public LogLevel getLevel()
      {
         return level;
      }

      public Integer getCode()
      {
         return code;
      }

      public boolean isError()
      {
         return LogLevel.ERROR == level;
      }

      public boolean isWarning()
      {
         return LogLevel.WARNING == level;
      }

      public boolean isInfo()
      {
         return LogLevel.INFO == level;
      }

      public String toString()
      {
         if (isError() && !(code.equals(ERROR_NO_CODE)))
         {
            return String.format("%s %d: %s", level.getLabel(), code, msg);
         }
         return String.format("%s: %s", level.getLabel(), msg);
      }
   }

}



enum LogLevel {
  INFO,
  WARNING,
  ERROR;

  public String getLabel() {
    return this.toString();
  }
}


