package edu.mayo.bior.catalog.verification;

import com.google.gson.*;
import com.google.gson.stream.JsonReader;
import com.google.gson.stream.JsonToken;
import com.google.gson.stream.MalformedJsonException;
import edu.mayo.pipes.history.ColumnMetaData;
import edu.mayo.pipes.history.ColumnMetaData.Type;

import org.apache.commons.io.IOUtils;
import org.apache.commons.lang.StringUtils;

import java.io.ByteArrayInputStream;
import java.io.IOException;
import java.io.InputStreamReader;
import java.io.UnsupportedEncodingException;
import java.math.BigDecimal;
import java.util.*;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

/**
 * Verify all json for a catalog and keep track of some statistics
 * Verify JSON types and counts against columns.tsv file
 */
public class CatalogJsonVerifier
{
   private Map<String, ColumnMetaData> mColumnInfo;
   private MessageLogger mLogger;
   private JsonParser mParser = new JsonParser();

   private List<Error> mJsonErrors = new ArrayList<Error>();
   private List<String> mJsonWarnings = new ArrayList<String>();

   private List<String> mCatalogColsNotFoundInSomeJsonRows = new ArrayList<String>();
   private Set<String> mJsonKeysNotFoundInColumnsTsv = new HashSet<String>();
   private Map<String, Long> mNumberOfTimesKeySeenInCatalog = new HashMap<String, Long>();
   // The max values seen for the key (Ex:  'A':1 == 1,  'A':[1,2,3,4] == 4
   private Map<String, Long> mMaxValuesForKey = new HashMap<String, Long>();

   public CatalogJsonVerifier(Map<String, ColumnMetaData> columnInfo, MessageLogger logger)
   {
      mColumnInfo = columnInfo;
      mLogger = logger;
      if (!haveColumnInfo())
      {
         logError("Unable to check json has valid keys because we have no column info from columns.tsv.", VerifyErrorCodes.JSON_KEYS_CANNOT_BE_VALIDATED_DUE_TO_MISSING_COLUMNS_TSV);
      }
   }

   /** Used for testing counts > Integer.MAX_VALUE */
   protected void _test_setInitial_numberOfTimesKeySeenInCatalog(String key, long num) {
	   this.mNumberOfTimesKeySeenInCatalog.put(key, num);
   }
   
   /** Used for testing counts > Integer.MAX_VALUE */
   protected void _test_setInitial_maxValuesForKey(String key, long num) {
	   this.mMaxValuesForKey.put(key, num);
   }
   
   public void verify(String json) throws JsonParseException
   {
      JsonObject catalogRowJsonObj;
      try {
         catalogRowJsonObj = mParser.parse(json).getAsJsonObject();
      }
      catch (Exception e) {
         logError(String.format("Trouble parsing '%s'. Msg: '%s'", json, e.getMessage()), VerifyErrorCodes.JSON_UNPARSEABLE);  // Error code 402?  Diff from order verifier?  But still same parsing logic
         return;
      }


      mJsonErrors.clear();
      mJsonWarnings.clear();
      boolean keysAndQuotingValid = traverseJsonForInvalidJson(json);
      addErrorToListIfQuotingNotValid(keysAndQuotingValid);

      if (haveColumnInfo()) {
         // TODO - what is the point of adding to mCatalogColsNotFoundInSomeJsonRows? It's never used.
         //        I think we should be calling traverseJsonByColumn to check it's reasonable
         List<String> columnsNotFoundInJson = getColumnTsvFieldsNotFoundInJson(catalogRowJsonObj);
         addColumnsNotFoundToKeyList(columnsNotFoundInJson);

         // TODO - what is the point of adding to mJsonKeysNotFoundInColumnsTsv? I don't see it used. I think it is
         //        necessary to call traverseJsonByKey
         Set<String> jsonKeysNotFound = getJsonKeysNotInColumnsTsv(catalogRowJsonObj);
         addJsonKeysNotFoundInColumnsTsv_toKeyListAndErrorList( jsonKeysNotFound );
      }

      logAnyJsonWarnings(mJsonWarnings, json);
      logAnyJsonErrors(mJsonErrors,     json);
   }

   private void addErrorToListIfQuotingNotValid(boolean keysAndQuotingValid) {
	   if ( ! keysAndQuotingValid) {
		   // only add this if you don't have any issues in the json already
		   String msg = "If there aren't other obvious problems, look for unquoted keys/alphanumeric values " +
				   "or single quoted keys/values";
		   mJsonErrors.add(new Error(msg, VerifyErrorCodes.JSON_UNQUOTED_OR_BAD_QUOTES_IN_KEYS_OR_VALUES));
	   }
   }

   private void addColumnsNotFoundToKeyList(List<String> columnsNotFoundInJson) {
       for (String jsonKey : columnsNotFoundInJson) {
          if (!mCatalogColsNotFoundInSomeJsonRows.contains(jsonKey)) {
             mCatalogColsNotFoundInSomeJsonRows.add(jsonKey);
          }
       }
   }

   private void addJsonKeysNotFoundInColumnsTsv_toKeyListAndErrorList(Set<String> jsonKeysNotFound) {
       mJsonKeysNotFoundInColumnsTsv.addAll(jsonKeysNotFound);
       if ( ! jsonKeysNotFound.isEmpty()) {
          List<String> sortedJsonKeys = new ArrayList<String>(jsonKeysNotFound);
          Collections.sort(sortedJsonKeys);
          String msg = String.format("Keys in json not found in columns.tsv: '%s'", StringUtils.join(sortedJsonKeys, ","));
          mJsonErrors.add(new Error(msg, VerifyErrorCodes.JSON_KEY_NOT_FOUND_IN_COLUMNS_TSV));
       }
   }

   private void logAnyJsonWarnings(List<String> jsonWarnings, String json) {
	   if ( ! jsonWarnings.isEmpty() ) {
		   logInfo(String.format("Found the following warnings parsing this json '%s'", json));
		   for (String warning : jsonWarnings) {
			   logWarning(warning);
		   }
	   }
   }


   private void logAnyJsonErrors(List<Error> jsonErrors, String json) {
	   if ( ! jsonErrors.isEmpty()) {
		   // Only print the first 1000 chars of the line if it is really long
		   int MAX = 1000;
		   if( json.length() > 1000 )
			   logInfo(String.format("Found the following issues parsing this json (1st %d chrs): '%s' ....", MAX, json.substring(0, MAX)));
		   else
			   logInfo(String.format("Found the following issues parsing this json '%s'", json));
		   for (Error error : jsonErrors) {
			   logError(error.getMsg(), error.getCode());
		   }
	   }
   }

   private boolean haveColumnInfo()
   {
      return mColumnInfo != null && mColumnInfo.size() > 0;
   }

   public Set<String> keysNotFoundInColumnInfo()
   {
      return mJsonKeysNotFoundInColumnsTsv;
   }

   public long getNumberOfTimesKeySeenInCatalog(String key)
   {
      Long numTimesSeen = mNumberOfTimesKeySeenInCatalog.get(key);
      if (numTimesSeen == null)
      {
         return 0;
      }
      return numTimesSeen;
   }

   /**
    * Get the maximum number of values seen in the catalog for this key (0 if not seen).
    * NOTE: The key is usually a JsonArray.
    */
   public long getMaxValuesForKey(String key)
   {
      Long maxValues = mMaxValuesForKey.get(key);
      if (maxValues == null)
      {
         return 0;
      }
      return maxValues;
   }

   private List<String> getColumnTsvFieldsNotFoundInJson(JsonObject catalogRowJsonObj)
   {
      List<String> columnsNotFoundInJson = new ArrayList<String>();

      // Check Json by Catalog Columns:
      for (String colName : mColumnInfo.keySet())
      {
         Object jsonObjFound = getJsonValueForCatalogKey(colName, catalogRowJsonObj, mColumnInfo.get(colName));
         if (jsonObjFound == null)
         {
            //logger.write("WARNING: no json obj found for column name: " + eachCatColNm + NL);
            columnsNotFoundInJson.add(colName);
         }
         else
         {
            incrementNumberOfTimesKeySeenInCatalog(mColumnInfo.get(colName).getColumnName());
         }
      }

      return columnsNotFoundInJson;
   }

   private Set<String> getJsonKeysNotInColumnsTsv(JsonObject catalogRowJsonObj)
   {
      return getJsonKeysNotInColumnsTsv(catalogRowJsonObj, null);
   }

   // TODO: Need to now compare number of elements to columns.tsv count value??
   private Set<String> getJsonKeysNotInColumnsTsv(JsonObject catalogRowJsonObj, String parentJsonKeyName)
   {
	  // Are we using a HashSet here in case there are multiple of the same keys in a JSON object???
      Set<String> jsonKeysNotInColumnInfo = new HashSet<String>();
      java.lang.Object currentJsonObj;
      for (Map.Entry<String, JsonElement> thisEntry : catalogRowJsonObj.entrySet())
      {
         String thisJsonKey = thisEntry.getKey();
         JsonElement thisJsonElement = thisEntry.getValue();

         currentJsonObj = getJavaObjForJSON(thisJsonKey, thisJsonElement, jsonKeysNotInColumnInfo);
         // its ok if a json object comes back null from this because it was already interrogated
         // by recursive call to this method from the getJavaObjForJSON() method.
         if (currentJsonObj == null) {
        	 addToWarningsIfNonJsonObject(thisJsonElement, thisJsonKey);
        	 // Continue to next element in for loop
        	 continue;
         }
         
         // OK, do we have a valid object from looking through JSON
         //     and creating a Java object based on the JSON data type?
         //     If we do have an object and the json key name(s) are found in the catalog column info, we are ok.
         //     If we do have an object but the json key isn't found in the catalog column info, we want to report that.
         String key = (parentJsonKeyName != null)  ?  (parentJsonKeyName + "." + thisJsonKey)  :  thisJsonKey;
         addWarningsIfNotFoundOrTypeMismatch(key, currentJsonObj, jsonKeysNotInColumnInfo);
      } // end for loop

      return jsonKeysNotInColumnInfo;
   }

   private void addWarningsIfNotFoundOrTypeMismatch(String jsonKey, java.lang.Object currentJsonObj, Set<String> jsonKeysNotInColumnInfo)
   {
	   ColumnMetaData colMeta = mColumnInfo.get(jsonKey);
	   if ( colMeta == null ) {
		   jsonKeysNotInColumnInfo.add(jsonKey);
		   mJsonWarnings.add(String.format("Catalog key '%s' not found in columns.tsv. Cannot check data type.", jsonKey));
	   } else {
		   // TODO - just call verifyDataTypesMatch and put lots of info in that method about context
		   if ( ! dataTypesMatch(currentJsonObj, colMeta)) {
			   ColumnMetaData.Type objType = getColumnDataTypeFromJsonObj(currentJsonObj);
			   String msg = "Data type does not match between json ["
					   + objType.name()
					   + "] and columns.tsv [" + colMeta.getType()
					   + "] for key: " + colMeta.getColumnName();
			   mJsonErrors.add(new Error(msg, VerifyErrorCodes.JSON_DATA_TYPE_MISMATCH));
		   }

		   // Verify the counts match (that if there is a dot in the columns.tsv that it matches a JsonArray
		   if ( ! countMatch(currentJsonObj, colMeta)) {
			   String msg = "Count does not match between json and columns.tsv for key: " + colMeta.getColumnName();
			   mJsonErrors.add(new Error(msg, VerifyErrorCodes.JSON_DATA_COUNT_MISMATCH));
		   }

	   }
   }

   // Get the JSON data type as a ColumnMetaData.Type
   // If the JsonObject is of type JsonArray, then look at the first element if it has one
   private ColumnMetaData.Type getColumnDataTypeFromJsonObj(Object o) {
	   if( o == null )
		   return null;

	   if( o instanceof Boolean )
		   return ColumnMetaData.Type.Boolean;
	   else if( o instanceof Integer  ||  o instanceof Long )
		   return ColumnMetaData.Type.Integer;
	   else if( o instanceof Float || o instanceof Double || o instanceof BigDecimal )
		   return ColumnMetaData.Type.Float;
	   else if( o instanceof JsonObject )
		   return ColumnMetaData.Type.JSON;
	   else if( o instanceof JsonArray ) {
		   // NOTE: JsonArray could be empty, have null values, contain a mixture of different types
		   //       (especially Integer,Long,Float,Double,BigDecimal if it is a number)
		   Set<ColumnMetaData.Type> typesSet = getJsonDataTypesInJsonArray((JsonArray)o);
		   return getLowestLevelJsonDataType(typesSet);
	   } else if( o instanceof String ) {
		   return ColumnMetaData.Type.String;
	   }

	   // Else: String?
	   return ColumnMetaData.Type.String;
   }

   private Set<ColumnMetaData.Type> getJsonDataTypesInJsonArray(JsonArray arr) {
	   Set<Type> types = new HashSet<ColumnMetaData.Type>();
	   for( int i=0; i < arr.size(); i++ ) {
		   types.add(getColumnDataTypeFromJsonObj(arr.get(i)));
	   }
	   return types;
   }
   
   // Order of precedence:  JSON, String, Float, Integer, Boolean, null
   private Type getLowestLevelJsonDataType(Set<Type> typesSet) {
	   if( typesSet.contains(ColumnMetaData.Type.JSON))
		   return ColumnMetaData.Type.JSON;
	   if( typesSet.contains(ColumnMetaData.Type.String) )
		   return ColumnMetaData.Type.String;
	   if( typesSet.contains(ColumnMetaData.Type.Float) )
		   return ColumnMetaData.Type.Float;
	   if( typesSet.contains(ColumnMetaData.Type.Integer) )
		   return ColumnMetaData.Type.Integer;
	   if( typesSet.contains(ColumnMetaData.Type.Boolean) )
		   return ColumnMetaData.Type.Boolean;
	   // Empty set or nulls
	   return null;
   }



   // Ensure that if the columns.tsv says it is a dot, then the json object is a JsonArray
   // Or vice-versa if columns.tsv says it is NOT a dot, that the json object is not a JsonArray
   private boolean countMatch(Object currentJsonObj, ColumnMetaData colMeta) {
	   return (  isArrayChar(colMeta.getCount())  &&     currentJsonObj instanceof JsonArray)
		   || (! isArrayChar(colMeta.getCount())  &&  ! (currentJsonObj instanceof JsonArray));
   }
   
   // Is the "Count" string from the columns.tsv an array character?
   // We will allow "." or "A" from the VCF spec.
   // From VCF 4.3 spec:  https://samtools.github.io/hts-specs/VCFv4.3.pdf
   //   A: The field has one value per alternate allele. The values must be in the same order as listed in the ALT column
   //   R: The field has one value for each possible allele, including the reference. The order of the values must be the
   //		reference allele first, then the alternate alleles as listed in the ALT column.
   //	G: The field has one value for each possible genotype. The values must be in the same order as prescribed in section 1.6.2 (see Genotype Ordering)
   private boolean isArrayChar(String s) {
	   return ".".equals(s) || "A".equals(s);
   }

   private void addToWarningsIfNonJsonObject(JsonElement thisJsonElement, String thisJsonKey) {
       // TODO - Why a warning? Or is it ok that it's not an object
       if (!thisJsonElement.isJsonObject())
       {
          String msg = String.format("Java Object could not be created for Json key '%s' and value '%s'.",
             thisJsonKey, thisJsonElement.toString());
          mJsonWarnings.add(msg);
       }
   }

   private Object getJavaObjForJSON(String jsonKey, JsonElement jsonElement, Set<String> jsonKeysNotInColumnInfoMap)
   {
      Object currentJsonObj = null;
      if (jsonElement.isJsonNull())
      {
         // This is handled elsewhere and logged for verify so no need to log it again
         return null;
      }
      else if (jsonElement.isJsonObject())
      {
         // Embedded JSON:
         Set<String> jsonKeysNotFoundInCols = getJsonKeysNotInColumnsTsv(jsonElement.getAsJsonObject(), jsonKey);
         jsonKeysNotInColumnInfoMap.addAll(jsonKeysNotFoundInCols);
         currentJsonObj = null; // set the current object to null because we already traversed it here if it was a JSON object
      }
      else if (jsonElement.isJsonArray())
      {
         currentJsonObj = jsonElement.getAsJsonArray();
      }
      else if (jsonElement.isJsonPrimitive())
      {
         JsonPrimitive jsonPrimVal = jsonElement.getAsJsonPrimitive();
         if (jsonPrimVal.isBoolean())
         {
            Boolean boolVal = jsonPrimVal.getAsBoolean();
            currentJsonObj = boolVal;
         }
         else if (jsonPrimVal.isNumber())
         {
            currentJsonObj = jsonNumberToJavaNumber(jsonPrimVal);
         }
         else if (jsonPrimVal.isString())
         {
            String strVal = jsonPrimVal.getAsString();
            if (strVal != null)
            {
               currentJsonObj = strVal;
            }
         }
      }
      else
      {
         mJsonErrors.add(new Error("traverseJsonByJsonKeyset(): haven't found a json datatype for json key: " + jsonKey, VerifyErrorCodes.JSON_NO_DATATYPE_FOUND));
         currentJsonObj = null;
      }


      // TODO make a check here that you don't have somethng and put out 1 message and skip message above
//    if (currentJsonObj == null and !jsonElement.isJsonObject())
//    {
//
//    }

      return currentJsonObj;
   }


   /**
    * traverseJsonForInvalidJson: This method traverses the BioR Catalog JSON string looking for the following
    * JSON structural issues:
    * 1.) Any json property name not surrounded by double quotes will fail to
    * be read, and will throw an exception. The issue is logged and false is returned.
    * 2.) Any json property name that is surrounded by double quotes, and read
    * successfully is checked to ensure there are no embedded '.' (period)
    * characters within the key. This confuses as this character is reserved
    * to represent embedded JSON objects.
    * 3.) Property values that are alphanumeric that are not surrounded by
    * double quotes will also cause an exception. (This does not include a number
    * that has been surrounded by double quotes; this is a valid JSON value even
    * though may be more desirable to have this be a JSON number and not surrounded
    * by quotes so that JSON can load it and identify its data type automatically.)
    *
    * @param rdr
    * @return false if any issues found, true if no issues found
    */
   private boolean traverseJsonForInvalidJson(String json)
   {
	   JsonReader jsonRdrFromTabix;
	   try {
		   jsonRdrFromTabix = new JsonReader(new InputStreamReader(new ByteArrayInputStream(json.getBytes("UTF-8"))));
	   }
	   // TODO - Should this be a Runtime Exception instead?
	   catch (UnsupportedEncodingException e) {
		   logError("Couldn't get JsonReader to verify json " + json, VerifyErrorCodes.JSON_UNPARSEABLE_JSON_READER);
		   return true;
	   }

	   boolean isAllJsonValid = traverseJsonForInvalidJson(jsonRdrFromTabix);
	   
	   IOUtils.closeQuietly(jsonRdrFromTabix);

	   return isAllJsonValid;
   }
   
   private boolean traverseJsonForInvalidJson(JsonReader jsonRdrFromTabix) {
	   boolean isAllJsonValid = true;  // default to no problem, switch to false if find a problem.
	   String currentPropName = null;
	   try
	   {
		   if (jsonRdrFromTabix.isLenient())
		   {
			   String msg = "Json Reader is set to lenient in traverseJsonToCheckForInvalidJson(). " +
					   "Verification needs the reader set to strict checking for this method to perform checks adequately.";
			   jsonRdrFromTabix.close();
			   throw new RuntimeException(msg);
		   }

		   jsonRdrFromTabix.beginObject();
		   while (jsonRdrFromTabix.hasNext())
		   {
			   // IMPORTANT NOTE: Because are using the com.google.gson.stream.JsonReader, nextName() will
			   //   fail when strict checking is enabled if the key isn't quoted, which is one important
			   //   thing we are checking for.
			   currentPropName = jsonRdrFromTabix.nextName();

			   if (currentPropName.contains(".")) {
				   // We don't want embedded periods within a key name.
				   String msg = String.format("Not allowed to have '.' in json key '%s'", currentPropName);
				   mJsonErrors.add(new Error(msg, VerifyErrorCodes.JSON_KEY_MUST_NOT_CONTAIN_DOT));
				   isAllJsonValid = false;
			   }
			   if (jsonRdrFromTabix.peek() != JsonToken.NAME) {
				   Object value = readNextJsonTokenValue(jsonRdrFromTabix, currentPropName);
				   if ( value == null) {
					   mJsonErrors.add(new Error("Did not return a json value for json property name: " + currentPropName,  VerifyErrorCodes.JSON_NO_JSON_VALUE_FOUND));
					   isAllJsonValid = false;
				   } else {
					   // Check if the value is an array and contains a pipe
					   if( isValueArrayAndContainsPipe(value) ) {
						   mJsonErrors.add(new Error("Detected a pipe '|' character in a JSON Array value.  "
								   + "This is dangerous since pipe characters are used for the delimiter when drilling out array values.  "
								   + "This will cause an error when drilling out this key.  " + currentPropName + ":" + value,  VerifyErrorCodes.JSON_PIPE_CHARACTER_FOUND_IN_JSON_ARRAY));
					   }
				   }
			   }
		   }
		   jsonRdrFromTabix.endObject();
	   }
	   catch (MalformedJsonException mal)
	   {
		   logJsonParsingErrorMessage(currentPropName, mal);
		   isAllJsonValid = false;
	   }
	   catch (IOException io)
	   {
		   logJsonParsingErrorMessage(currentPropName, io);
		   isAllJsonValid = false;
	   } 
	   return isAllJsonValid;
   }


   private boolean isValueArrayAndContainsPipe(Object value) {
	   if( ! isNonEmptyStringArrayList(value) )
		   return false;

	   ArrayList<String> valueList = (ArrayList<String>)value;
	   for(String valStr : valueList) {
		   if( valStr.contains("|") )
			   return true;
	   }
	   return false;
   }

   private boolean isNonEmptyStringArrayList(Object value) {
	   return value != null
		 && (value instanceof ArrayList)
		 && ((ArrayList)value).size() > 0
		 && ((ArrayList)value).get(0) instanceof String;
   }

   private void logJsonParsingErrorMessage(String propertyName, Exception e)
   {
      String msg;
      if (propertyName != null)
      {
         msg = String.format("Problem parsing json property '%s'. Msg: '%s'", propertyName, e.getMessage());
      }
      else
      {
         msg = String.format("Problem parsing json. Msg: '%s'", e.getMessage());
      }
      mJsonErrors.add(new Error(msg, VerifyErrorCodes.JSON_PROBLEM_PARSING_JSON_PROPERTY));
   }

   private Object getJsonValueForCatalogKey(String columnName, JsonObject jsonObj, ColumnMetaData catalogCol)
   {
      String[] columnNameElems;
      if (!columnName.contains("."))
      {
         columnNameElems = new String[]{columnName};
      }
      else
      {
         columnNameElems = columnName.split("\\.");  // escape special char '.' so splits on period ('.').
         if (columnNameElems.length < 1)
         {
            String msg = String.format("Bad column name format. Less than one element: '%s'", columnName);
            mJsonErrors.add(new Error(msg, VerifyErrorCodes.JSON_BAD_COLUMN_NAME_FORMAT_LESS_THAN_ONE_NAME));
            return null;
         }
      }

      java.lang.Object currentJsonObj = jsonObj;
      java.util.Iterator<String> columnNameElemItr = Arrays.asList(columnNameElems).iterator();
      while (columnNameElemItr.hasNext())
      {
         String colElem = columnNameElemItr.next();

         // Make sure you can get the json element before start trying to verify it:
         JsonElement jsonElem = null;
         if (currentJsonObj instanceof com.google.gson.JsonObject)
         {
            com.google.gson.JsonObject currentJson = (com.google.gson.JsonObject) currentJsonObj;
            jsonElem = currentJson.get(colElem);
         }
         if (jsonElem == null)
         {
            //logger.write("ERROR: getJsonForCatalogColumn(): Column element [" + colElem + "] is not found in the json object. json object: " + jsonObj.getAsJsonObject().toString());
            return null;
         }

         // Ok, got the json element. Now verify it:
         if (jsonElem.isJsonObject())
         {
            com.google.gson.JsonObject embeddedJsonObj = jsonElem.getAsJsonObject();
            if (embeddedJsonObj == null)
            {
               // TODO - why is this a warning?
               mJsonWarnings.add("Json object retrieval was null for column key '" + columnName +
                  "' and column value '" + colElem + "'.");
               return null;
            }
            else
            {
               // Now get the JSON object for the current column name element
               if (!columnNameElemItr.hasNext())
               {
                  return embeddedJsonObj;  // we are done, no more drilling needed to get to final json obj
               }
               else
               {
                  String nextColNameElement = columnNameElemItr.next();
                  currentJsonObj = getJsonValueForCatalogKey(nextColNameElement, embeddedJsonObj, catalogCol);
               }
            }
         }
         else if (jsonElem.isJsonArray())
         {
            JsonArray jsonArr = jsonElem.getAsJsonArray();
            currentJsonObj = jsonArr;
            setMaxValuesForKeyIfGreater(catalogCol.getColumnName(), jsonArr.size());
         }
         else if (jsonElem.isJsonPrimitive())
         {
            JsonPrimitive jsonPrim = jsonElem.getAsJsonPrimitive();
            if (jsonPrim.isJsonNull())
            {
               //System.out.println("WARNING: getJsonForCatalogColumn(): column element name: " + colElem + " Embedded JsonObject is Json primitive (json null): " +  jsonPrim );
               currentJsonObj = jsonPrim;
            }
            else if (jsonPrim.isBoolean())
            {
               //System.out.println("column element name: " + colElem + " Json primitive: [" +  jsonPrim + "] boolean value: " + jsonPrim.getAsBoolean());
               currentJsonObj = jsonPrim.getAsBoolean();
            }
            else if (jsonPrim.isNumber())
            {

               String numStr = jsonPrim.getAsString();
               if (!isValidInteger(numStr, catalogCol.getType()))
               {
                  //System.out.println("column element name: " + colElem + "Json primitive Number: [" +  jsonPrim + "] Make it a double: " + jsonPrim.getAsDouble());
                  currentJsonObj = jsonPrim.getAsDouble();
               }
               else
               {
                  //System.out.println("column element name: " + colElem + "Json primitive Number: [" +  jsonPrim + "] Make it an int: " + jsonPrim.getAsInt());
                  currentJsonObj = jsonPrim.getAsInt();
               }

            }
            else if (jsonPrim.isString())
            {
               //System.out.println("column element name: " + colElem + "Json primitive: [" +  jsonPrim + "] Is a String: " + jsonPrim.getAsString());
               currentJsonObj = jsonPrim.getAsString();
               String strVal = (String) currentJsonObj;
               if (!isQuotedString(strVal))
               {
                  mJsonErrors.add(new Error("Invalid json formatting: json string value '" + strVal + "' is not quoted.", VerifyErrorCodes.JSON_STRING_VALUE_NOT_QUOTED));
               }
            }
            else if (jsonPrim.isJsonObject())
            {
               // I don't think is a valid condition in our code, but put in case we hit it.
               // TODO - ERROR and WARNING?
               throw new RuntimeException("ERROR: getJsonForCatalogColumn(): column element name: " + colElem +
                  " WARNING: Embedded JsonObject is Json primitive but says it is a json object too. Not expecting this.");
            }
            else if (jsonPrim.isJsonArray())
            {
               // I don't think is a valid condition in our code, but put in case we hit it.
               // TODO - ERROR and WARNING?
               throw new RuntimeException("ERROR: getJsonForCatalogColumn(): column element name: " + colElem + " WARNING: Embedded JsonObject is Json primitive but says it is a json array too. Not expecting this.");
            }
         } // end if Json Types

         if (currentJsonObj != null && columnNameElemItr.hasNext())
         {
            continue; // Keep navigating to the final column name element to get that element's object/value within the json
         }
         else
         {
            return currentJsonObj;
         }
      } // end while iterate over column name elements


      return null; // If we end up down here, it is unexpected, and we probably didn't find a valid json
   }

   private Object readNextJsonTokenValue(JsonReader rdr, String currentPropName) throws IOException
   {
      Object jsonValue = null;
      JsonToken jsonToken = rdr.peek();  // when there is an invalid quoted string in a value, the exception will happen

      if (jsonToken == JsonToken.NULL)
      {
         // Even though this is a valid way to specify JSON null, it causes problems for us
         // See http://stackoverflow.com/questions/21120999/representing-null-in-json for JSON discussion
         String msg = String.format("JSON null for key '%s'. This causes issues with pre 4.2 commands. Please leave " +
               "'%s' out when the value is null.",
            currentPropName, currentPropName);
         mJsonErrors.add(new Error(msg, VerifyErrorCodes.JSON_NULL_VALUE));
         rdr.skipValue();
      }
      else if (jsonToken == JsonToken.BEGIN_ARRAY)
      {
         List<Object> arrayElements = new ArrayList<Object>();
         int idx = 0;
         rdr.beginArray();
         while (rdr.hasNext())
         {
            arrayElements.add(readNextJsonTokenValue(rdr, currentPropName + "_" + idx));
            idx++;
         }
         rdr.endArray();
         jsonValue = arrayElements;
      }
      else if (jsonToken == JsonToken.BEGIN_OBJECT)
      {
         jsonValue = true;
         if (!traverseJsonForInvalidJson(rdr))
         {
            jsonValue = null;
         }
      }
      else if (jsonToken == JsonToken.NUMBER)
      {
         String numStr = rdr.nextString(); // read it as string first
         try
         {
            // shortcut to see if this is a double
            if (numStr.contains("."))
            {
               jsonValue = new Double(numStr);
            }
            // now pay the price to see if it's an int. Hopefully optimizer optimizes this
            // NOTE: Changed the type from Integer here to Long, because integers over 2.1G would throw exception
            //       Hopefully this change doesn't impact anything downstream
            else if (isLongInt(numStr))
            {
               jsonValue = new Long(numStr);
            }
            // last ditch effort. Try to make it a double
            else
            {
               jsonValue = new Double(numStr);
            }
         }
         catch (NumberFormatException e)
         {
            throw new IOException(String.format("Couldn't turn number %s into a float or int", numStr));
         }

      }
      else if (jsonToken == JsonToken.STRING)
      {
         // Once the String is read in by the .nextString() method, the quotes are stripped off.
         // If it does contain alphabetic and not quoted, this method will throw exception when
         //   strict json parsing is enabled, as is in current code implementation.
         String s = rdr.nextString();
         //System.out.println(currentPropName + " -- String value in field is: " + s);
         jsonValue = s;
      }
      else if (jsonToken == JsonToken.BOOLEAN)
      {
         jsonValue = rdr.nextBoolean();
         //System.out.println(currentPropName + " -- Boolean value in field is: " + jsonValue.toString());
      }


      return jsonValue;
   }

   // is Long if is all digits, length is < 18, or can typecast to Long
   // Process the checks in this order to save computational time
   // Long range: 9,223,372,036,854,775,807
   private boolean isLongInt(String numStr) {
	   if( ! numStr.matches("\\d+") )
		   return false;
	   if( numStr.length() < 18 )
		   return true;
	   try {
		   Long.parseLong(numStr);
	   } catch(Exception e) {
		   return false;
	   }
	   return true;
   }
	   
   private void incrementNumberOfTimesKeySeenInCatalog(String key)
   {
      Long numTimesSeen = mNumberOfTimesKeySeenInCatalog.get(key);
      if (numTimesSeen == null)
         mNumberOfTimesKeySeenInCatalog.put(key, 1L);
      else
         mNumberOfTimesKeySeenInCatalog.put(key, numTimesSeen + 1);
   }

   // TODO - change this to be verifyDataTypesMatch and give some more info if they don't and don't put a message
   //        out where this is called
   private boolean dataTypesMatch(Object jsonObj, ColumnMetaData columnDefinition)
   {

      ColumnMetaData.Type definedDataType = columnDefinition.getType();
      String definedCount = columnDefinition.getCount();

      if (jsonObj instanceof Boolean && definedDataType == ColumnMetaData.Type.Boolean)
      {
         return true;
      }
      else if ((jsonObj instanceof Integer || jsonObj instanceof Long) && definedDataType == ColumnMetaData.Type.Integer)
      {
         return true;
      }
      else if ((jsonObj instanceof Float || jsonObj instanceof Double ||
         jsonObj instanceof BigDecimal || jsonObj instanceof Integer) &&
         definedDataType.equals(ColumnMetaData.Type.Float))
      {
         return true;
      }
      else if (jsonObj instanceof String && definedDataType == ColumnMetaData.Type.String)
      {
         return true;
         // TODO - is this right? Yes it is
//         if (definedCount.equals("1"))
//         {
//            return true;
//         }
      }
      else if (jsonObj instanceof com.google.gson.JsonArray)
      {
    	  for(JsonElement o : (JsonArray)jsonObj) {
    		  Object javaObj = getJavaObjForJSON(columnDefinition.getColumnName(), o, null);
    		  boolean isMatch = dataTypesMatch(javaObj, columnDefinition);
    		  if( ! isMatch )
    			  return false;
    	  }
    	  // Made it through all items in the array and none were a mismatch, so return true
    	  return true;
      }

      return false;
   }

   private Object jsonNumberToJavaNumber(com.google.gson.JsonPrimitive jsonNumber)
   {
      Number num = stringToJavaNumber(jsonNumber.getAsString());
      if (num instanceof Integer || num instanceof Long ||
         num instanceof Float || num instanceof BigDecimal || num instanceof Double)
      {
         return num;
      }

      return null;
   }

   private void setMaxValuesForKeyIfGreater(String key, long numValues)
   {
      Long maxValues = mMaxValuesForKey.get(key);
      if (maxValues == null)
         mMaxValuesForKey.put(key, numValues);
      else
      {
         if (numValues > maxValues)
            mMaxValuesForKey.put(key, numValues);
      }
   }

   // TODO - check if Julie had done some unit tests previously for this and bring them back in
   private boolean isValidInteger(String intStrValue, ColumnMetaData.Type type)
   {
      if (type.equals(ColumnMetaData.Type.Integer))
      {
         try
         {
            Integer intVal = new Integer(intStrValue);
            // TODO - I assume this would fail if intStrValue is not an integer and you don't need to do this
            if (intVal % 1 != 0)
            {
               return false;
            }
            if (intVal >= java.lang.Integer.MIN_VALUE && intVal <= java.lang.Integer.MAX_VALUE)
            {
               return true;
            }
         }
         catch (NumberFormatException e)
         {
            return false;
         }
      }

      return false;
   }

   private boolean isQuotedString(String str)
   {
      if (str == null)
      {
         return false;
      }
      else if (!(str.startsWith("\"") && str.endsWith("\"")))
      {
         return true;
      }
      else
      {
         return false;
      }
   }

   private Number stringToJavaNumber(String value)
   {
      if (!value.contains("."))
      {
         try
         {
            long valueAsLong = Long.parseLong(value);
            // if value fits into an int return it as an int
            if (valueAsLong >= Integer.MIN_VALUE && valueAsLong <= Integer.MAX_VALUE)
            {
               return Integer.parseInt(value);
            }
            else
            {
               return valueAsLong;
            }
         }
         catch (NumberFormatException ignored)
         {
            // Ignore exception so that it will try to create a double below.
         }
      }

      try
      {
         Double valueAsDouble = Double.parseDouble(value);
         ;
         if (valueAsDouble >= Float.MIN_VALUE && valueAsDouble <= Float.MAX_VALUE)
         {
            return Float.parseFloat(value);
         }
         else
         {
            return valueAsDouble;
         }
      }
      catch (NumberFormatException nf)
      {
         return new BigDecimal(value);
      }
   }

   private void logInfo(String msg)
   {
      mLogger.logInfo(msg);
   }

   private void logWarning(String msg)
   {
      mLogger.logWarning(msg);
   }

   private void logError(String msg, int code)
   {
      mLogger.logError(msg, code);
   }

   private class Error
   {
      private String msg;
      private int code;

      public Error(String msg, int code)
      {
         this.msg = msg;
         this.code = code;
      }

      public String getMsg()
      {
         return msg;
      }

      public int getCode()
      {
         return code;
      }
   }
}
