package edu.mayo.bior.catalog.verification;

import com.google.gson.*;
import edu.mayo.bior.catalog.CatalogFormatException;
import edu.mayo.bior.catalog.GoldenAttribute;
import org.apache.commons.lang.StringUtils;

import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;

import org.apache.commons.lang.ObjectUtils;

public class CatalogEntryGoldenJson
{
   private static final String NL = VerifyUtils.NL;

   private String landmarkChr = null;
   private Long minBP = null;
   private Long maxBP = null;
   private String refAllele = null;
   private List<String> altAlleles = null;
   private String strand = null;
   private List<String> errors = new ArrayList<String>();

   public CatalogEntryGoldenJson(String json) throws CatalogFormatException, JsonParseException
   {
      this(VerifyUtils.getJsonObject(json));
   }

   public CatalogEntryGoldenJson(JsonObject catalogRowJson) throws CatalogFormatException
   {
      landmarkChr = getStringFromJson(catalogRowJson, GoldenAttribute._landmark.toString());
      minBP = getLongFromJson(catalogRowJson, GoldenAttribute._minBP.toString());
      maxBP = getLongFromJson(catalogRowJson, GoldenAttribute._maxBP.toString());

      refAllele = getStringFromJson(catalogRowJson, GoldenAttribute._refAllele.toString());

      strand = getStringFromJson(catalogRowJson, GoldenAttribute._strand.toString());

      altAlleles = getStringArrayFromJson(catalogRowJson, GoldenAttribute._altAlleles.toString());
      if (errors.size() > 0)
      {
         throw new CatalogFormatException(StringUtils.join(errors, NL));
      }
   }

   public boolean isVariant()
   {
      return getRefAllele() != null;
   }

   private String getStringFromJson(JsonObject catalogRowJson, String key)
   {
      if (catalogRowJson.get(key) == null)
      {
         return null;
      }
      JsonPrimitive jsonPrimitiveString = catalogRowJson.getAsJsonPrimitive(key);
      if (jsonPrimitiveString == null)
      {
         errors.add(String.format("Couldn't get key '%s' from json '%s' as a JsonPrimitive. It should be a string",
                                  key, catalogRowJson.toString()));
         return null;
      }
      if (!jsonPrimitiveString.isString())
      {
         errors.add(String.format("key %s value '%s' from json '%s' is not a JSON string: ",
            key, jsonPrimitiveString.getAsString(), catalogRowJson.toString()));
      }
      return jsonPrimitiveString.getAsString();
   }

   private Long getLongFromJson(JsonObject catalogRowJson, String key)
   {
      if (catalogRowJson.get(key) == null)
      {
         return null;
      }
      JsonPrimitive jsonPrimitiveInt = catalogRowJson.getAsJsonPrimitive(key);
      if (jsonPrimitiveInt == null)
      {
         errors.add(String.format("Couldn't get key '%s' from json '%s' as a JsonPrimitive. It should be an integer",
            key, catalogRowJson.toString()));
         return null;
      }
      if (!jsonPrimitiveInt.isNumber())
      {
         errors.add(String.format("key %s value '%s' from json '%s' is not a JSON number: ",
            key, jsonPrimitiveInt.getAsString(), catalogRowJson.toString()));
         return null;
      }
      try
      {
         return new Long(jsonPrimitiveInt.getAsString());
      }
      catch (NumberFormatException e)
      {
         errors.add(String.format("key %s value '%s' from json '%s' is not an integer: ",
            key, jsonPrimitiveInt.getAsString(), catalogRowJson.toString()));
         return null;
      }
   }

   private List<String> getStringArrayFromJson(JsonObject catalogRowJson, String key)
   {
      if (catalogRowJson.get(key) == null)
      {
         return null;
      }

      JsonArray jsonArray;
      try
      {
         jsonArray = catalogRowJson.getAsJsonArray(key);
         if (jsonArray == null)
         {
            errors.add(String.format("Couldn't get key '%s' from json '%s' as a JSON Array.",
               key, catalogRowJson.toString()));
            return null;
         }
      }
      catch (Exception e)
      {
         errors.add(String.format("Couldn't get key '%s' from json '%s' as a JSON Array.",
            key, catalogRowJson.toString()));
         return null;
      }

      Iterator<JsonElement> iterator = jsonArray.iterator();
      List<String> list = new ArrayList<String>();
      while (iterator.hasNext())
      {
         JsonElement jsonElement = iterator.next();
         JsonPrimitive jsonPrimitive = jsonElement.getAsJsonPrimitive();
         if (jsonPrimitive == null)
         {
            errors.add(String.format("Couldn't get an item from array for '%s' from json '%s' as a JsonPrimitive. " +
               "It should be a string", key, catalogRowJson.toString()));
            return null;
         }
         if (!jsonPrimitive.isString())
         {
            errors.add(String.format("Item from array for key %s value '%s' from json '%s' is not a JSON string: ",
               key, jsonPrimitive.getAsString(), catalogRowJson.toString()));
            return null;
         }
         list.add(jsonPrimitive.getAsString());
      }
      return list;
   }

   public String getChr()
   {
      return landmarkChr;
   }

   public Long getMinBP()
   {
      return minBP;
   }

   public Long getMaxBP()
   {
      return maxBP;
   }

   public String getRefAllele()
   {
      return refAllele;
   }

   public List<String> getAltAlleles()
   {
      return altAlleles;
   }

   public String getStrand()
   {
      return strand;
   }

   public boolean equals(CatalogEntryGoldenJson otherVariant)
   {
      if (ObjectUtils.notEqual(this.getChr(), otherVariant.getChr()))
      {
         return false;
      }
      if (ObjectUtils.notEqual(this.getMinBP(), otherVariant.getMinBP()))
      {
         return false;
      }
      if (ObjectUtils.notEqual(this.getMaxBP(), otherVariant.getMaxBP()))
      {
         return false;
      }

      if (ObjectUtils.notEqual(this.getRefAllele(), otherVariant.getRefAllele()))
      {
         return false;
      }

      if (ObjectUtils.notEqual(this.getStrand(), otherVariant.getStrand()))
      {
         return false;
      }

      // test in non-order dependent way:
      List<String> theseAlts = this.getAltAlleles();
      List<String> otherAlts = otherVariant.getAltAlleles();
      if (theseAlts == null && otherAlts != null)
      {
         return false;
      }
      if (theseAlts != null && otherAlts == null)
      {
         return false;
      }
      if (theseAlts != null)
      {
         if (theseAlts.size() != otherAlts.size())
         {
            return false;
         }
         for (String eachAlt : theseAlts)
         {
            if (!otherAlts.contains(eachAlt))
            {
               return false;
            }
         }
      }


      return true;
   }
}
