package edu.mayo.bior.buildcatalog;

import java.io.File;
import java.io.IOException;
import java.util.ArrayList;
import java.util.List;

import org.apache.commons.io.FileUtils;
import org.apache.commons.lang.StringUtils;

import edu.mayo.bior.pipeline.exception.CatalogMetadataInputException;

/** Performs a possibly 4-way merge for each of the files:
 *    - [catalog].columns.tsv
 *    - [catalog].datasource.properties
 *    - [catalog].blacklist
 *  Between (in this order of most important to least, taking into account that if any value is blank it looks to the next level)
 *    - the current possibly-manually-modified file  (ex:  $currentWorkingDir/[catalog].columns.tsv )
 *    - the previously modified file                 (ex:  $currentWorkingDir/[catalog].columns.tsv.03 - choosing the highest of any files names with .xx extension)
 *    - the default files from catalog crawling      (ex:  $currentWorkingDir/[BuildCatalog.BUILD_SUBDIR]/[catalog].columns.tsv.default )
 *    - a previous catalog                           (ex:  $anotherCatalogDir/[oldCatalog].columns.tsv )
 *  Steps for columns.tsv merge
 *    1) Copy original file ($currentWorkingDir/[catalog].columns.tsv) to 1+(highest numbered existing file): EX: $currentWorkingDir/[catalog].columns.tsv.04
 *    2) Load X = $currentWorkingDir/[catalog].columns.tsv
 *    3) Load Y = $currentWorkingDir/[BuildCatalog.BUILD_SUBDIR]/[catalog].columns.tsv.default
 *    4) Load Z = $anotherCatalogDir/[catalog].columns.tsv
 *    5) Loop thru X, and for each missing column, check if it is present in Y.  If not, check Z
 *    6) Write contents of columns object to X location
 *  Steps for datasource.properties
 *    1) Copy original file ($currentWorkingDir/[catalog].datasource.properties) to 1+(highest numbered existing file): Ex: $currentWorkingDir/[catalog].datasource.properties.04
 *    2) Load X = $currentWorkingDir/[catalog].datasource.properties
 *    3) Load Y = $currentWorkingDir/[BuildCatalog.BUILD_SUBDIR]/[catalog].datasource.properties.default
 *    4) Load Z = $anotherCatalogDir/[catalog].datasource.properties
 *    5) Loop thru X, and for each missing value, check if it is present in Y.  If not, check Z
 *    6) Write contents of columns object to X location
 *  Similar for the blacklist file
 * @author Michael Meiners (m054457)
 * 2016-04-27  */
public class MergeMetadataStep implements BuildCatalogStepInterface {
	private BuildInfo mBuildInfo;
	private StepLogger mStepLogger;
	private File mCatalogDir;
	// The full file path to the previous catalog, ending in .tsv.bgz
	private File mPreviousCatalogFile;
	
	public MergeMetadataStep(BuildInfo buildInfo, StepLogger logger) throws IOException {
		mBuildInfo = buildInfo;
		mStepLogger = logger;
		mCatalogDir = new File(buildInfo.getTargetDirectory());
		
		// Previous catalog columns tsv, datasource.properties, blacklist are obtained by looking at the 
		// previous catalog name and replacing the '.tsv.bgz' suffix with the applicable prop file suffix.
		// If this catalog file does not exist, do not continue because the merge will not properly bring
		// in the previous catalog's property file info.
		mPreviousCatalogFile = null;
		if( buildInfo.getPreviousCatalogPath() != null  && buildInfo.getPreviousCatalogPath().trim().length() > 0 ) {
			mPreviousCatalogFile = new File(buildInfo.getPreviousCatalogPath()); 
			checkPreviousCatalogValue(mPreviousCatalogFile);
		}
	}

	public void execute() throws BuildCatalogStepExecuteException {
		try {
			mergeDataSourceProperties();

			mergeColumnsTsv();

			File currentCatalog 	  = new File(mCatalogDir, mBuildInfo.getCatalogPrefix() + ".tsv.bgz");

			// Blacklist
			File defaultCtgBlacklist  		= new File(mCatalogDir.getCanonicalPath() + "/" + BuildCatalog.BUILD_SUBDIR,  mBuildInfo.getCatalogPrefix() + ".columns.tsv.blacklist.default");
			mergeBlacklist(getBlacklistFromCatalog(currentCatalog), getBlacklistFromCatalog(mPreviousCatalogFile), defaultCtgBlacklist, /*isBiorWebBlacklist=*/false);
			
			// Blacklist BioRWeb
			File defaultCtgBlacklistBiorWeb = new File(mCatalogDir.getCanonicalPath() + "/" + BuildCatalog.BUILD_SUBDIR,  mBuildInfo.getCatalogPrefix() + ".columns.tsv.blacklist.biorweb.default");
			mergeBlacklist(getBlacklistBiorwebFromCatalog(currentCatalog), getBlacklistBiorwebFromCatalog(mPreviousCatalogFile), defaultCtgBlacklistBiorWeb, /*isBiorWebBlacklist=*/true);
		} catch(Exception e) {
			throw new BuildCatalogStepExecuteException(e);
		}
	}

	private void mergeDataSourceProperties() throws Exception {
		MergeDataSourceProperties mergeProps = new MergeDataSourceProperties(mBuildInfo, mStepLogger);
		mergeProps.mergeDatasourceProperties();
	}

	private void mergeColumnsTsv() throws IOException, CatalogMetadataInputException {
		MergeColumnsTsv mergeCols = new MergeColumnsTsv(mCatalogDir, mBuildInfo.getCatalogPrefix(), getColumnsTsvFromCatalog(mPreviousCatalogFile), mStepLogger);
		mergeCols.mergeColumnsTsv();
	}


	protected File getColumnsTsvFromCatalog(File catalogFile) {
		if( catalogFile == null )
			return null;
		return new File(catalogFile.getParentFile(), catalogFile.getName().replace(".tsv.bgz", ".columns.tsv"));
	}

	protected File getBlacklistFromCatalog(File catalogFile) {
		if( catalogFile == null )
			return null;
		return new File(catalogFile.getParentFile(), catalogFile.getName().replace(".tsv.bgz", ".columns.tsv.blacklist"));
	}

	protected File getBlacklistBiorwebFromCatalog(File catalogFile) {
		if( catalogFile == null )
			return null;
		return new File(catalogFile.getParentFile(), catalogFile.getName().replace(".tsv.bgz", ".columns.tsv.blacklist.biorweb"));
	}

	private void mergeBlacklist(File currentFile, File previousCtgFile, File defaultFile, boolean isBiorWebBlacklist)  throws IOException {
		String name = isBiorWebBlacklist ? "blacklist.biorweb" : "blacklist";
		// If the current catalog blacklist file is DIFFERENT FROM the default one,
		// then just return since we don't want to overwrite any user changes
		String currentStr = isExists(currentFile) ? FileUtils.readFileToString(currentFile) : "";
		String defaultStr = isExists(defaultFile) ? FileUtils.readFileToString(defaultFile) : "";
		String previousStr= isExists(previousCtgFile) ? FileUtils.readFileToString(previousCtgFile) : "";
		
		// If non-blank:
		if( currentStr.length() > 0 ) {
			//    If same as default, then update with previous (if given and not the same as current)
			if( currentStr.equals(defaultStr)  &&  previousStr.length() > 0  &&  ! currentStr.equals(previousStr) ) {
				mStepLogger.logAndSummary("Copying previous catalog " + name + " file");
				FileUtils.copyFile(previousCtgFile, currentFile);
			} else { //    Else leave same
				mStepLogger.logAndSummary("Current " + name + " file has been previously modified by user.  No changes made.");
			}
		}
		// Else If blank, get from previous catalog (if available), else from default
		else {
			if( previousStr.length() > 0 ) {
				mStepLogger.logAndSummary("Copying previous catalog " + name + " file");
				FileUtils.copyFile(previousCtgFile, currentFile);
			} else { // from default
				mStepLogger.logAndSummary("Copying default catalog " + name + " file.");
				FileUtils.copyFile(defaultFile, currentFile);
			}
		}
	}


	private boolean isExists(File file) {
		return file != null  &&  file.exists();
	}

	/* If the user has specified a previous catalog to merge with, the catalog file itself
	 * must exist, AND at least one of the prop files to be merged. Otherwise, we can't 
	 * tell what the user's intension is for specifying this as a previous catalog with which to merge.
	 */
	private void checkPreviousCatalogValue(File previousCatalogFile) throws IOException {
		String prevCatalogFilePath = mBuildInfo.getPreviousCatalogPath();
		if (!previousCatalogFile.canRead()) {
			throw new IOException(BuildInfoKey.PREVIOUS_CATALOG_PATH.name() + " property value is not a valid readable catalog file: " + prevCatalogFilePath);
		}

		File dataSrcProps = MergeDataSourceProperties.getDataSourcePropertiesFromCatalog(previousCatalogFile);
		File colsTsv = getColumnsTsvFromCatalog(previousCatalogFile);
		File colsTsvBlacklist = getBlacklistFromCatalog(previousCatalogFile);
		File colsTsvBlacklistBiorweb = getBlacklistBiorwebFromCatalog(previousCatalogFile);
		
		boolean isAtLeastOnePropFileReadable = 
				isExistsAndReadable(dataSrcProps)
			||  isExistsAndReadable(colsTsv)
			||  isExistsAndReadable(colsTsvBlacklist)
			||  isExistsAndReadable(colsTsvBlacklistBiorweb);
		
		if( ! isAtLeastOnePropFileReadable ) {
			throw new IOException("The catalog specified by " + BuildInfoKey.PREVIOUS_CATALOG_PATH.name() 
				+ " must have at least one properties file accompanying it "
				+ "(datasource.properties, columns.tsv, columns.tsv.blacklist, columns.tsv.blacklist.biorweb).  "
				+ "Ensure the files exists and are readable, or remove the " + BuildInfoKey.PREVIOUS_CATALOG_PATH.name()
				+ " value from the build information file.");			
		}
	}

	private boolean isExistsAndReadable(File f) {
		if( f == null )
			return false;
		return  f.exists()  &&  f.canRead();
	}
}
