package edu.mayo.bior.buildcatalog;

import java.io.File;
import java.io.IOException;
import java.net.URISyntaxException;
import java.util.regex.Pattern;

import edu.mayo.bior.util.progress.NumLineProgressHandler;
import org.apache.commons.io.FileUtils;

import edu.mayo.bior.catalog.Catalog;
import edu.mayo.bior.catalog.CatalogMetadataConstant;
import edu.mayo.bior.cli.cmd.CreateCatalogPropsCommand;
import edu.mayo.bior.util.validation.CatalogFileValidator;
import edu.mayo.cli.InvalidDataException;

public class CreatePropFilesStep implements BuildCatalogStepInterface
{
   private BuildInfo mBuildInfo = null;
   private StepLogger mStepLogger = null;

   private String mCatalogDirectoryPath = null;
   private String mCatalogFilePath = null;

   private String mDataSrcFilePath = null;
   private String mColumnsTsvFilePath = null;
   private String mBlacklistFilePath = null;
   private String mBlacklistBiowebFilePath = null;

   public CreatePropFilesStep(BuildInfo buildInfo, StepLogger stepLogger) throws IOException
   {
      mBuildInfo = buildInfo;
      mStepLogger = stepLogger;
      init(mBuildInfo);
   }

   public void execute() throws BuildCatalogStepInputException, BuildCatalogStepExecuteException
   {
      mStepLogger.log("Starting create catalog property files step :\n");
      mStepLogger.log("------------------------------------\n");

      mStepLogger.log("Create catalog property files step: Checking inputs... \n");
      checkInput();

      mStepLogger.log("Create catalog property files step: creating property files ... \n");
      CreateCatalogPropsCommand cmd = new CreateCatalogPropsCommand();
      cmd.setStepLogger(mStepLogger);
      try
      {
         cmd.execNoCmd(mCatalogFilePath, null, mCatalogDirectoryPath, /*isVcfSpecified=*/false, new ProgressHandler());
      }
      catch (InvalidDataException cliInvalidDataException)
      {
         throw new BuildCatalogStepExecuteException("Error executing creating catalog property files: " + cliInvalidDataException.getMessage());
      }
      catch (IOException ioe)
      {
         throw new BuildCatalogStepExecuteException("IO Exception occurred: " + ioe.getMessage());
      }
      catch (URISyntaxException e)
      {
         throw new BuildCatalogStepExecuteException("CreatePropFilesStep.execute(): Expected output file not found for catalog [" + mCatalogFilePath + "]:" + "Exception: " + e.getMessage());
      }
      catch (Exception e)
      {
         throw new BuildCatalogStepExecuteException("CreatePropFilesStep.execute(): Unexpected exception Error creating properties files: " + e.getMessage());
      }

      mStepLogger.log("Create catalog property files step: Checking outputs ... \n");
      checkOutput();

      mStepLogger.log("Create catalog property files step completed.\n");
      mStepLogger.log("------------------------------------\n");
   }

   private void init(BuildInfo buildInfo) throws IOException
   {
      mCatalogFilePath = buildInfo.getTargetDirectory() + "/" + buildInfo.getCatalogPrefix() + CatalogMetadataConstant.CATALOG_FILE_SUFFIX;
   }

   private void checkInput() throws BuildCatalogStepInputException
   {
      File catalogFile = new File(mCatalogFilePath);
      if (!CatalogFileValidator.catalogExists(catalogFile))
      {
         throw new BuildCatalogStepInputException("CreateMetadataStep.checkInput(): Catalog file is not a valid file. Please check catalog build, create catalog step results.");
      }
      if (!CatalogFileValidator.hasExpectedSuffix(catalogFile))
      {
         throw new BuildCatalogStepInputException("CreateMetadataStep.checkInput(): Catalog file is not a valid file. Does not have expected catalog file name suffix.");
      }


      File catalogDir = null;
      try
      {
         catalogDir = catalogFile.getCanonicalFile().getParentFile();
         if (catalogDir != null)
         {
            if (!catalogDir.isDirectory() || !catalogDir.canWrite())
            {
               throw new BuildCatalogStepInputException("Acquired catalog directory doesn't appear to be a valid, writable directory: " + catalogDir.getAbsolutePath());
            }
            mCatalogDirectoryPath = catalogDir.getAbsolutePath();
            if (mCatalogDirectoryPath == null)
            {
               throw new BuildCatalogStepInputException("CreateMetadataStep.checkInput(): Could not successfully set parent directory of catalog file: " + mCatalogFilePath);
            }
         }
      }
      catch (IOException io)
      {
         throw new BuildCatalogStepInputException("Not able to acquire catalog directory from catalog file name: " + mCatalogFilePath);
      }

      String catalogFilePrefix = Catalog.getCatalogPrefix(mCatalogFilePath);
      if (catalogFilePrefix == null)
      {
         throw new BuildCatalogStepInputException("IndexesStep.checkInput(): catalog file not valid suffix [" + mCatalogFilePath + "]. Please check catalog file name.");
      }

      mDataSrcFilePath = mCatalogDirectoryPath + "/" + catalogFilePrefix + CatalogMetadataConstant.DATASRC_PROPS_SUFFIX;
      backupToNumeric(mDataSrcFilePath);
      mColumnsTsvFilePath = mCatalogDirectoryPath + "/" + catalogFilePrefix + CatalogMetadataConstant.COLUMN_INFO_SUFFIX;
      backupToNumeric(mColumnsTsvFilePath);
      mBlacklistFilePath = mCatalogDirectoryPath + "/" + catalogFilePrefix + CatalogMetadataConstant.BLACKLIST_SUFFIX;
      backupToNumeric(mBlacklistFilePath);
      mBlacklistBiowebFilePath = mCatalogDirectoryPath + "/" + catalogFilePrefix + CatalogMetadataConstant.BLACKLIST_BIORWEB_SUFFIX;
      backupToNumeric(mBlacklistBiowebFilePath);
   }

   private void checkOutput() throws BuildCatalogStepExecuteException
   {
      StringBuilder outputCheckMsg = new StringBuilder();

      String msgBegin = "CreatePropFilesStep.checkOutput():  ";
      
      outputCheckMsg.append(getMsg(new File(mDataSrcFilePath),    "datasource.properties"));
      outputCheckMsg.append(getMsg(new File(mColumnsTsvFilePath), "columns.tsv"));
      outputCheckMsg.append(getMsg(new File(mBlacklistFilePath),  "blacklist"));
      outputCheckMsg.append(getMsg(new File(mBlacklistBiowebFilePath), "blacklist biorweb"));

      if (outputCheckMsg.length() > 0)
      {
         throw new BuildCatalogStepExecuteException(msgBegin + outputCheckMsg);
      }
   }


   private String getMsg(File propsFile, String fileShortName) {
	   File newDataSrcFile = new File(mDataSrcFilePath);
	   if (!newDataSrcFile.exists()) {
		   return fileShortName + " was not created: " + propsFile.getAbsolutePath();
	   } else if (!newDataSrcFile.canRead())  {
		   return fileShortName + " is not readable: " + propsFile.getAbsolutePath();
	   }
	   return "";
   }

   

   /**
    * Backup the current file to the BuildCatalog.BUILD_SUBDIR subdirectory, creating the next backup file (with incrementing number at the end: .1, .2, .3, etc)
    *
    * @param catalogPropFileName
    * @throws BuildCatalogStepInputException
    */
   private void backupToNumeric(String catalogPropFileName) throws BuildCatalogStepInputException
   {

      File catalogDir = new File(mCatalogDirectoryPath);
      File backupDir = new File(catalogDir.getAbsolutePath() + "/" + BuildCatalog.BUILD_SUBDIR + "/");
      if (!backupDir.exists() && catalogDir.canWrite())
      {
         if (!backupDir.mkdir())
         {
            System.err.println("Unable to create directory: " + catalogDir.getAbsolutePath());
            throw new BuildCatalogStepInputException("Unable to build the " + BuildCatalog.BUILD_SUBDIR + " subdirectory in: " + catalogDir.getAbsolutePath());
         }
      }

      File curFile = new File(catalogPropFileName);
      if (!curFile.exists())
      {
         return;  // nothing to backup if prop file doesn't exist yet.
      }

      File newBackupFile = null;
      File mostRecentNumericBackup = getMostRecentNumericBackup(backupDir, curFile.getName());
      if (mostRecentNumericBackup == null)
      {
         newBackupFile = new File(backupDir.getAbsolutePath() + "/" + curFile.getName() + ".1");
      }
      else
      {
         String backupBasename = mostRecentNumericBackup.getName();
         String backupNumber = backupBasename.substring(curFile.getName().length(), backupBasename.length());
         backupNumber = backupNumber.replace(".", "");
         Integer nextNum = -1;
         try
         {
            Integer curNum = new Integer(backupNumber);
            nextNum = curNum.intValue() + 1;
         }
         catch (NumberFormatException nf)
         {
            System.err.println("Error getting next backup integer for: " + mostRecentNumericBackup.getAbsolutePath());
         }
         newBackupFile = new File(backupDir.getAbsolutePath() + "/" + curFile.getName() + "." + nextNum.intValue());
      }

      // Ok, COPY the current file to next integer value backup file now:
      try
      {
         FileUtils.copyFile(curFile, newBackupFile);
      }
      catch (Exception e)
      {
         mStepLogger.log("CreatePropFilesStep.backupToNumeric: FAILED to copy the current file: " + curFile.getAbsolutePath() + " to backup file: " + newBackupFile.getAbsolutePath());
      }
   }

   private File getMostRecentNumericBackup(File inDir, String filenamePrefix)
   {

      File mostRecentWithPrefix = null;

      File[] fileList = inDir.listFiles();
      for (File each : fileList)
      {
         if (!Pattern.matches(filenamePrefix + "\\.\\d+", each.getName()))
         {
            continue; // only looking through number suffix backups, not the '.default' file.
         }

         if (mostRecentWithPrefix == null)
         {
            mostRecentWithPrefix = each;
            continue; // keep looking for newer numeric backup files for this prop file.
         }
         else
         {
            if (each.lastModified() > mostRecentWithPrefix.lastModified())
            {
               mostRecentWithPrefix = each;
            }
         }
      }
      return mostRecentWithPrefix;
   }

   private class ProgressHandler implements NumLineProgressHandler
   {
      @Override
      public long getNumLinesToCallbackOn()
      {
         return 100000;
      }

      @Override
      public void startingRead()
      {
      }

      @Override
      public void readNumLines(long lineNumber, long numLinesInFile)
      {
         System.out.print(String.format("Read %d lines from catalog\r", lineNumber));
      }

      @Override
      public void readAllLines(long lineNumber)
      {
         System.out.println(String.format("Read %d lines from catalog\r", lineNumber));
      }
   }

}
