package edu.mayo.bior.buildcatalog;

import java.io.File;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.util.*;

import com.google.common.base.Joiner;
import com.jayway.jsonpath.InvalidPathException;

import edu.mayo.bior.catalog.CatalogMetadataConstant;
import edu.mayo.bior.util.validation.CatalogFileValidator;

import org.apache.commons.io.IOUtils;
import org.apache.commons.lang.StringUtils;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * The BuildInfo object can be built from a properties file given by the path to the file or constructed from a
 * Map where the key is a BuildInfo.BuildInfoKey (enum) and the value for each is a String. These are the names of
 * the properties (which correspond to the name for the BuildInfoKey enum) and constraints/discussion on each key.
 * These constraints are enforced when you construct the object and the values can't be changed.
 * <ul>
 * <li>MAKE_JSON_SCRIPT_PATH - must be supplied and have non-empty value. It is verified this path exists and is executable.</li>
 * <li>MAKE_JSON_ARGS - must be supplied but can have empty value. No validation is done on the value. This is passed to
 * the MAKE_JSON_SCRIPT_PATH</li>
 * <li>MAKE_JSON_OUTPUT_FILE_PATH - doesn't have to be supplied and can have empty value. If not supplied or empty value,
 * it is created in the temporary working directory.</li>
 * <li>TARGET_DIR - doesn't have to be supplied and can have empty value. If not supplied or empty value,
 * the target directory is the current working directory. The target_dir is verified to exist, be a directory and is
 * readable, writeable and executable.</li>
 * <li>TEMP_DIR - doesn't have to be supplied and can have empty value. If not supplied or empty value, a
 * temporary directory is created by looking in /local2/tmp, then /local1/tmp, the value in java.io.tmpdir and finally
 * /tmp. If created, the user name and a directory name "bior_build_catalog_tmp" is added as sub-directories. Then under
 * this temp dir a working directory is made so it can be deleted at the end of a successful build. You should call
 * getWorkingTempDirectory() to get a File handle to this directory and put any temp files here.</li>
 * <li>DATA_SOURCE - must be supplied and have non-empty value. No validation is done on the value.</li>
 * <li>DATA_SOURCE_VERSION - must be supplied and have non-empty value. No validation is done on the value.</li>
 * <li>DATA_SOURCE_BUILD - must be supplied but can have empty value. No validation is done on the value.</li>
 * <li>CATALOG_PREFIX - must be supplied and have non-empty value. No validation is done on the value.</li>
 * <li>INDEXES - doesn't have to be supplied and can have empty value. If supplied, it is split on ',' and
 * then each value is trimmed. You can only get at it by asking by calling getIndexes() which returns a List of Strings.</li>
 * <li>PREVIOUS_CATALOG_PATH - doesn't have to be supplied and can have empty value. If given, it is
 * verified that this path exists but no other catalog related files are examined.</li>
 * </ul>
 *
 */

public class BuildInfo
{
   private static Logger sLogger = LoggerFactory.getLogger(BuildInfo.class);

   private Map<String, String> mKeyMap = new HashMap<String, String>();

   private File mMakeJsonScriptPathFile;

   protected static final String DEFAULT_MAKE_JSON_OUTPUT_FILENAME = "make_json_output.tsv";

   private static final String BUILD_CATALOG_DIR = "bior_build_catalog_tmp";
   private File mWorkingTempDir;

   private File mTargetDirFile;

   // One-based TJSON input column to use to create the catalog.
   // -1 will default to last column of TJSON input; Set if JSON input
   // for catalog is not the last column in input.
   private int inputJSONColumnNum = -1;

   // User-specified chromosome sort order list is optional unless over-riding
   // default BIOR behavior.
   private String chromosomeSortOrderFileName;

   // true is default, so optional. set to false to leave input data in input order.
   private boolean sortCatalogOnCreate = true;

   // Is the input file already in the correct catalog format (3 tabix columns plus JSON), and already bgzipped?
   // If the make_json script creates a bgzip file that is already in the final format, then we can skip the part
   // that extracts the 3 tabix columns from the JSON as well as the bgzipping of the input file, which can save
   // hours or days on large catalogs.
   // false is default, so is optional.
   private boolean isInputFileAlreadyInFinalCtgFormatAndBgzipped = false;

   public BuildInfo(Map<String, String> inputMap) throws BuildCatalogStepInputException
   {
      copyMapToMap(inputMap);
      verifyData(null);
      resolveRelativePaths();
   }

   /**
    * Load the BuildInfo properties from a properties file (usually called build_info.txt)
    *
    * @param buildInfoPath Path to build_info.txt
    * @throws IOException if a problem reading the file
    * @throws BuildCatalogStepInputException if there is a problem with the properties read from the file
    */
   public BuildInfo(String buildInfoPath) throws BuildCatalogStepInputException, IOException
   {
      InputStream fin = null;
      try
      {
         fin = new FileInputStream(buildInfoPath);
         Properties tempProperties = new Properties();
         tempProperties.load(fin);
         putPropertiesInMap(tempProperties);
         resolveRelativePaths();
      }
      finally
      {
         IOUtils.closeQuietly(fin);
      }

      verifyData(buildInfoPath);
      resolveRelativePaths();
   }

   /** Resolve several of the properties related to files and directories to their full canonical paths 
    * @throws BuildCatalogStepInputException */
   private void resolveRelativePaths() throws BuildCatalogStepInputException {
	   String currentPath = "";
	   try {
		   List<BuildInfoKey> pathsToResolve = Arrays.asList(
				   BuildInfoKey.MAKE_JSON_OUTPUT_FILE_PATH,
				   BuildInfoKey.MAKE_JSON_SCRIPT_PATH,
				   BuildInfoKey.PREVIOUS_CATALOG_PATH,
				   BuildInfoKey.TARGET_DIR, 
				   BuildInfoKey.TEMP_DIR
				   );
		   for(BuildInfoKey key : pathsToResolve) {
			   if( isGiven(key) ) {
				   currentPath = getVal(key);
				   String canonicalPath = new File(currentPath).getCanonicalPath();
				   mKeyMap.put(key.name(), canonicalPath);
			   }
		   }
	   } catch(Exception e) {
		   String msg = "Could not canonicalize the path: " + currentPath;
		   sLogger.error(msg);
		   throw new BuildCatalogStepInputException(msg, e);
	   }
   }

   private boolean isGiven(BuildInfoKey key) {
	   String val = getVal(key);
	   return val != null  &&  val.length() > 0;
   }

   private void putPropertiesInMap(Properties properties)
   {
	   for(Object key : properties.keySet()) {
		   String val = properties.getProperty((String)key);
		   mKeyMap.put((String)key, val);
		   if( ! isRecognizedKey((String)key) ) {
			   // Warn if there are any keys that are not in the enum
			   String msg = "Warning: Unrecognized key in build_info.txt file: " + key.toString(); 
			   sLogger.warn(msg);
			   System.err.println(msg);
		   }
	   }
   }

   /** Copy the passed in map to the one contained in this class */
   private void copyMapToMap(Map<String, String> inputMap) {
	   for(String key : inputMap.keySet()) {
		   String val = inputMap.get(key);
		   if (val != null)  {
			   mKeyMap.put((String)key, val);
		   }
	   }
   }

   /**
    * Saves the BuildInfo properties to a properties file (build_info.txt)
    * WARNING: This will not save any comments, so overwriting an existing build info file will destroy the comments!
    *
    * @throws IOException
    */
   public void save(String buildInfoPath) throws IOException
   {
      Properties props = getPropertiesFromFields();
      FileOutputStream fout = null;
      try
      {
         fout = new FileOutputStream(new File(buildInfoPath));
         props.store(fout, "## Build Info file for BioR build catalog command");
      }
      finally
      {
         IOUtils.closeQuietly(fout);
      }
   }

   private void verifyData(String buildInfoPath)
      throws BuildCatalogStepInputException
   {
      // Make sure required keys are present
      verifyRequiredKeys(buildInfoPath);

      // Make sure keys that require non empty values have them
      verifyRequiredNonEmptyValues(buildInfoPath);

      // verify data settings as much as possible
      validateSettingsAndGenerateDefaults();
   }


   private void verifyRequiredKeys(String buildInfoPath)
      throws BuildCatalogStepInputException
   {
      List<BuildInfoKey> keysNotFound = new ArrayList<BuildInfoKey>();
      for (BuildInfoKey reqKey : BuildInfoKey.getRequiredKeys()) {
         String val = getVal(reqKey);
         if (val == null) {
            keysNotFound.add(reqKey);
         }
      }
      throwInputExceptionIfNotEmpty(buildInfoPath, "is missing", keysNotFound);
   }

    private void verifyRequiredNonEmptyValues(String buildInfoPath)
      throws BuildCatalogStepInputException
   {
      List<BuildInfoKey> keysWithoutValues = new ArrayList<BuildInfoKey>();
      for (BuildInfoKey reqKey : BuildInfoKey.getNonEmptyValueRequiredKeys()) {
         String val = getVal(reqKey);
         if (StringUtils.isBlank(val)) {
            keysWithoutValues.add(reqKey);
         }
      }
      throwInputExceptionIfNotEmpty(buildInfoPath, "has an empty value for", keysWithoutValues);
   }

   private void throwInputExceptionIfNotEmpty(String buildInfoPath, String msgPart,
                                              List<BuildInfoKey> keysWithProblem)
      throws BuildCatalogStepInputException
   {
      if (!keysWithProblem.isEmpty())
      {
         String keyDescription = composeItemNumDescription("key", keysWithProblem.size());
         String error = "Error: ";
         if (!StringUtils.isBlank(buildInfoPath))
         {
            error = error + "file '" + buildInfoPath + "'";
         }
         else
         {
            error = error + "data supplied to create BuildInfo object";
         }
         throw new BuildCatalogStepInputException(String.format("%s %s %s: %s",
            error, msgPart, keyDescription, Joiner.on(", ").join(keysWithProblem)));
      }
   }

   private void validateSettingsAndGenerateDefaults() throws BuildCatalogStepInputException
   {
      // MAKE_JSON_SCRIPT_PATH
      validateJsonScriptPath();

      // no additional validation for MAKE_JSON_ARGS

      // TEMP_DIR
      validateTempDirOrGenerate();

      // MAKE_JSON_OUTPUT_FILE_PATH
      validateJsonOutputPathOrGenerate();

      // TARGET_DIR
      validateTargetDirOrGenerate();

      // no additional validation for DATA_SOURCE
      // no additional validation for DATA_SOURCE_VERSION
      // no additional validation for DATA_SOURCE_BUILD
      // no additional validation for CATALOG_PREFIX
      // no additional validation for INDEXES possible until after generating catalog

      // PREVIOUS_CATALOG_PATH
      validatePreviousCatalogPath();
      
      // SORT_ROWS
      validateSortRowsFlag();
   }
   
   private void validateSortRowsFlag() throws BuildCatalogStepInputException {
	   String sortRowsFlag = getVal(BuildInfoKey.SORT_ROWS);
	   if( sortRowsFlag != null  &&  ! sortRowsFlag.equalsIgnoreCase("true")  &&  ! sortRowsFlag.equalsIgnoreCase("false") ) {
		   throw new BuildCatalogStepInputException(BuildInfoKey.SORT_ROWS.toString() + " key was specified but should have a value of either 'true' or 'false'");
	   }
   }


   private boolean isRecognizedKey(String key) {
	   try {
		   // Test that we can create a BuildInfoKey from the key
		   BuildInfoKey.valueOf(key);
		   return true;
	   } catch(Exception e) {
		   return false;
	   }
   }


   private void validatePreviousCatalogPath() throws BuildCatalogStepInputException
   {
      String previousCatalogPath = getPreviousCatalogPath();
      if (!StringUtils.isBlank(previousCatalogPath))
      {
         File previousCatalogFile = new File(previousCatalogPath);
         if (!previousCatalogFile.exists())
         {
            throw new BuildCatalogStepInputException(String.format("Could not find '%s' from %s",
               previousCatalogPath, BuildInfoKey.PREVIOUS_CATALOG_PATH));
         }
         if (!CatalogFileValidator.hasExpectedSuffix(previousCatalogPath))
         {
            throw new BuildCatalogStepInputException(String.format("'%s' from %s does not have expected catalog " +
               "suffix '%s'", previousCatalogPath, BuildInfoKey.PREVIOUS_CATALOG_PATH,
               CatalogMetadataConstant.CATALOG_FILE_SUFFIX));

         }
      }
   }

   private void validateTargetDirOrGenerate() throws BuildCatalogStepInputException
   {
      String targetDir = getTargetDirectory();
      if (StringUtils.isBlank(targetDir))
      {
         targetDir = ".";
         mKeyMap.put(BuildInfoKey.TARGET_DIR.name(), targetDir);
      }
      File targetDirFile = new File(targetDir);
      if (!targetDirFile.exists())
      {
         throw new BuildCatalogStepInputException(String.format("Could not find dir '%s' from %s",
            targetDir, BuildInfoKey.TARGET_DIR));
      }
      if (!targetDirFile.isDirectory())
      {
         throw new BuildCatalogStepInputException(String.format("'%s' from %s is not a directory",
            targetDir, BuildInfoKey.TARGET_DIR));
      }
      if (!targetDirFile.canWrite())
      {
         throw new BuildCatalogStepInputException(String.format("'%s' from %s is not writeable",
            targetDir, BuildInfoKey.MAKE_JSON_OUTPUT_FILE_PATH));

      }
      if (!targetDirFile.canRead())
      {
         throw new BuildCatalogStepInputException(String.format("'%s' from %s is not readable",
            targetDir, BuildInfoKey.MAKE_JSON_OUTPUT_FILE_PATH));

      }
      if (!targetDirFile.canExecute())
      {
         throw new BuildCatalogStepInputException(String.format("'%s' from %s is not executable",
            targetDir, BuildInfoKey.MAKE_JSON_OUTPUT_FILE_PATH));

      }
      mTargetDirFile = targetDirFile;
   }

   private void validateJsonOutputPathOrGenerate() throws BuildCatalogStepInputException
   {
      String outputPath = getMakeJsonOutputFilePath();
      if (!StringUtils.isBlank(outputPath))
      {
         File outputFile = new File(outputPath);
         if (outputFile.exists() && outputFile.isDirectory())
         {
            throw new BuildCatalogStepInputException(String.format("'%s' from %s is a directory, not a file",
               outputPath, BuildInfoKey.MAKE_JSON_OUTPUT_FILE_PATH));
         }
         File parentDir = outputFile.getParentFile();
         if (parentDir != null && parentDir.exists() && !parentDir.canWrite())
         {
            throw new BuildCatalogStepInputException(String.format("Parent dir for '%s' from %s is not writeable",
               outputPath, BuildInfoKey.MAKE_JSON_OUTPUT_FILE_PATH));
         }
         if (parentDir != null && parentDir.exists() && !parentDir.canRead())
         {
            throw new BuildCatalogStepInputException(String.format("Parent dir for '%s' from %s is not readable",
               outputPath, BuildInfoKey.MAKE_JSON_OUTPUT_FILE_PATH));
         }
         if (parentDir != null && parentDir.exists() && !parentDir.canExecute())
         {
            throw new BuildCatalogStepInputException(String.format("Parent dir for '%s' from %s is not executable",
               outputPath, BuildInfoKey.MAKE_JSON_OUTPUT_FILE_PATH));
         }
      }
      // need to generate the output path
      else
      {
         File outputFile = new File(getWorkingTempDirectory(), DEFAULT_MAKE_JSON_OUTPUT_FILENAME);
         mKeyMap.put(BuildInfoKey.MAKE_JSON_OUTPUT_FILE_PATH.name(), outputFile.getPath());
      }
   }

   private void validateTempDirOrGenerate() throws BuildCatalogStepInputException
   {
      String tempDir = getBaseTempDirectory();
      File baseTempDirFile;
      if (StringUtils.isBlank(tempDir))
      {
    	  try {
    		  baseTempDirFile = new File(TempDirectory.findBaseTempDir(BUILD_CATALOG_DIR));
    	  } catch(InvalidPathException e) {
    		  throw new BuildCatalogStepInputException(e.getMessage());
    	  }
    	  // fill in the temp dir with something that makes sense
    	  mKeyMap.put(BuildInfoKey.TEMP_DIR.name(), baseTempDirFile.getPath());
      }
      else
      {
         baseTempDirFile = new File(tempDir);
      }
      
      if (!baseTempDirFile.exists())
      {
         // let's try to make it
         if (!baseTempDirFile.mkdirs())
         {
            throw new BuildCatalogStepInputException(String.format("Could not create temp dir '%s'. It is likely " +
               "due to permissions.", getBaseTempDirectory()));
         }
      }
      
      if (!baseTempDirFile.isDirectory())
      {
         throw new BuildCatalogStepInputException(String.format("'%s' from %s is not a directory", getBaseTempDirectory(),
            BuildInfoKey.TEMP_DIR));
      }
      
      File workingTempDir;
      try
      {
         workingTempDir = File.createTempFile("tmp_", "_bior_build_catalog", baseTempDirFile);
         if (!workingTempDir.delete())
         {
            sLogger.error("Trying to delete a temp file made from File.createTempFile but couldn't delete the file using File.delete");
            sLogger.error("A more generic message is put in the exception");
            throw new BuildCatalogStepInputException("Had trouble creating a temporary working directory in " +
               getBaseTempDirectory());
         }
         if (!workingTempDir.mkdir())
         {
            sLogger.error("Trying to create a temp working directory but couldn't make the directory with File.mkdir");
            sLogger.error("A more generic message is put in the exception");
            throw new BuildCatalogStepInputException("Had trouble creating a temporary working directory in " +
               getBaseTempDirectory());
         }
      }
      catch (IOException e)
      {
         throw new BuildCatalogStepInputException("Had trouble creating a temporary working directory in " +
            getBaseTempDirectory());
      }
      mWorkingTempDir = workingTempDir;
      mKeyMap.put(BuildInfoKey.TEMP_DIR.name(), mWorkingTempDir.getAbsolutePath());
   }

   private void validateJsonScriptPath() throws BuildCatalogStepInputException
   {
      String scriptPath = getMakeJsonScriptPath();
      File scriptPathFile = new File(scriptPath);
      if (!scriptPathFile.exists())
      {
         throw new BuildCatalogStepInputException(String.format("Error: could not find script '%s' from %s",
            scriptPath, BuildInfoKey.MAKE_JSON_SCRIPT_PATH));
      }
      if (!scriptPathFile.canExecute())
      {
         throw new BuildCatalogStepInputException(String.format("Error: script '%s' from %s is not executable: ",
            scriptPath, BuildInfoKey.MAKE_JSON_SCRIPT_PATH));
      }
      mMakeJsonScriptPathFile = scriptPathFile;
   }

   private static String composeItemNumDescription(String baseDescription, int numItems)
   {
      if (numItems == 1)
      {
         return baseDescription;
      }
      return baseDescription + "s";
   }


   /**
    * Get the File for getMakeJsonScriptPath. This File should exist and be executable.
    */
   File getMakeJsonScriptPathFile()
   {
      return mMakeJsonScriptPathFile;
   }

   /**
    * Get the supplied path. For example: /home/m054457/omim/compile.sh.
    */
   String getMakeJsonScriptPath()
   {
      return getVal(BuildInfoKey.MAKE_JSON_SCRIPT_PATH);
   }

   /**
    * The arguments to pass to the script from getMakeJsonScriptPath(). For example:
    * --omimRefData /research/bsi/data/refdata/omim/20160420/omim.tsv
    */
   String getMakeJsonArgs()
   {
      return getVal(BuildInfoKey.MAKE_JSON_ARGS);
   }

   /**
    * If the make json output file path was supplied, return that (e.g. omim_json.tsv). If the path was not supplied or
    * was blank ("" or "    " for example), return a path to the output file in the temporary working directory space.
    */
   String getMakeJsonOutputFilePath()
   {
      return getVal(BuildInfoKey.MAKE_JSON_OUTPUT_FILE_PATH);
   }

   /**
    * This is the path to the directory where all the catalog files will be put. If supplied (e.g. /research/bsi/development/TEMP/target_dir),
    * that path will be returned. Otherwise '.' (current working directory) will be returned. You can get a File
    * handle to this folder from getTargetDirectoryFile().
    */
   public String getTargetDirectory()
   {
      return getVal(BuildInfoKey.TARGET_DIR);
   }

   /**
    * File corresponding to path getTargetDirectory().
    */
   File getTargetDirectoryFile()
   {
      return mTargetDirFile;
   }

   /**
    * This is the path to the base temp directory and corresponds to the value (if supplied) by TEMP_DIR. If not
    * supplied, a temp directory is generated by looking in /local2/tmp, then /local1/tmp, then the path given
    * by the java property java.io.tmpdir and finally /tmp. A directory with the user name and another directory
    * with bior_catalog is then appended. For example, let's say no value was supplied for TEMP_DIR and we did find
    * /local2/tmp on the system and our user name was user2, we would get back a value from here that was
    * "/local2/tmp/user2/bior_build_catalog_tmp"
    */
   String getBaseTempDirectory()
   {
      return getVal(BuildInfoKey.TEMP_DIR);
   }

   /**
    * This is a file handle to the working temp directory which is a directory in getBaseTempDirectory(). This is where
    * all the temporary files should be placed and if everything works, it should be removed at the end of the run. This
    * is guaranteed to be a valid directory to work in, and should be unique for each run
    * (so multiple runs of bior_build_catalog can be executed at the same time without clobbering each other)
    */
   File getWorkingTempDirectory()
   {
      return mWorkingTempDir;
   }

   /**
    * @return guaranteed to be a non empty value. e.g. "dbNSP", "HGMD"
    */
   String getDataSourceName()
   {
      return getVal(BuildInfoKey.DATA_SOURCE);
   }

    /**
    * @return guaranteed to be a non empty value.  e.g. "142" (dbSNP), "2016.1" (HGMD)
    */
   String getDataSourceVersion()
   {
      return getVal(BuildInfoKey.DATA_SOURCE_VERSION);
   }

   /**
    * @return the build value. This can be an empty string but not null. e.g. "GRCh38", "GRCh37.p13", ""
    */
   String getDataSourceBuild()
   {
      return getVal(BuildInfoKey.DATA_SOURCE_BUILD);
   }

   /**
    * @return the data source release date.  e.g. "2018-05-23"
    */
   String getDataSourceReleaseDate()
   {
      return getVal(BuildInfoKey.DATA_SOURCE_RELEASE_DATE);
   }

   /**
    * The catalog prefix is the portion of the catalog base filename before the ".tsv.bgz" suffix. This value will be
    * used to name the catalog data and related files.
    * @return guaranteed to be a non empty value. e.g. "00-All.vcf" (dbSNP), "HGMD_PRO_2016.1" (HGMD)
    */
   public String getCatalogPrefix()
   {
      return getVal(BuildInfoKey.CATALOG_PREFIX);
   }

   /**
    * Comma-separated list of H2 indexes to build. Each value is a JSON key in the json column of the catalog. e.g.
    * "ID" (dbSNP), "GENE,ID,RSID" (HGMD)
    * @return comma-separated list of indexes. Can be an empty list.
    */
   List<String> getIndexes()
   {
      return splitByCommaAndTrim(getVal(BuildInfoKey.INDEXES));
   }

   private static List<String> splitByCommaAndTrim(String s)
   {
      List<String> valList = new ArrayList<String>();
      if (s == null || s.length() <= 0)
      {
         return valList;
      }

      String[] vals = s.split(",");
      for (String val : vals)
      {
         String eachIndex = val.trim();
         if (eachIndex.length() > 0)
         {
            valList.add(eachIndex);
         }
      }
      return valList;
   }



   /**
    * Get the previous catalog path that will be used to gather info for properties files merge.
    * This is the catalog path as given ending in ".tsv.bgz".  Ex:  /research/bsi/development/TEMP/mycatalog/mycatalog.tsv.bgz"
    *
    * @return previous catalog path (not required, so could be null or empty string ("").
    *         e.g. "/research/bsi/development/TEMP/mycatalog/mycatalog.tsv.bgz", "mycatalog.tsv.bgz"
    */
   String getPreviousCatalogPath()
   {
      return getVal(BuildInfoKey.PREVIOUS_CATALOG_PATH);
   }

   /**
    * @return the one-based JSON column number that should be used to build the catalog content. This default to '-1'
    *         meaning the last column of the file output by getMakeJsonScriptPath(). 7 would mean that column 7 of the
    *         file output by getMakeJsonScriptPath() would be the JSON that is used.
    */
   int getinputJSONColumnNum()
   {
      return this.inputJSONColumnNum;
   }

   /**
    * @param inputJSONColumnNum one-based column number of JSON column used to build catalog content.
    */
   void setInputJSONColumnNum(int inputJSONColumnNum)
   {
      this.inputJSONColumnNum = inputJSONColumnNum;
   }


   String getChromosomeSortOrderFileName()
   {
      return chromosomeSortOrderFileName;
   }

   void setChromosomeSortOrderFileName(String chromosomeSortOrderFileName)
   {
      this.chromosomeSortOrderFileName = chromosomeSortOrderFileName;
   }

   boolean getSortCatalogOnCreate()
   {
	   String isSortStr = getVal(BuildInfoKey.SORT_ROWS);
	   if( isSortStr != null ) {
		   return "TRUE".equalsIgnoreCase(isSortStr);
	   } else {
		   return this.sortCatalogOnCreate;
	   }
   }

   void setSortCatalogOnCreate(String sortOnCreate)  {
      if (sortOnCreate.toUpperCase().equals("TRUE")) {
         this.sortCatalogOnCreate = true;
      } else if (sortOnCreate.toUpperCase().equals("FALSE")) {
         this.sortCatalogOnCreate = false;
      }
   }

   boolean getInFinalFormat()
   {
	   String isFinalFormatStr = getVal(BuildInfoKey.FINAL_FORMAT);
	   if( isFinalFormatStr != null ) {
		   return "TRUE".equalsIgnoreCase(isFinalFormatStr);
	   } else {
		   return this.isInputFileAlreadyInFinalCtgFormatAndBgzipped;
	   }
   }

   void setInFinalFormat(String inFinalFormat)  {
      if (inFinalFormat.toUpperCase().equals("TRUE")) {
         this.isInputFileAlreadyInFinalCtgFormatAndBgzipped = true;
      } else if (inFinalFormat.toUpperCase().equals("FALSE")) {
         this.isInputFileAlreadyInFinalCtgFormatAndBgzipped = false;
      }
   }

   private Properties getPropertiesFromFields() throws IOException
   {
      Properties props = new Properties();
      for (BuildInfoKey key : BuildInfoKey.values())
      {
         String val = getVal(key);
         if (val != null)
         {
            props.setProperty(key.name(), val);
         }
      }
      return props;
   }
   
   public Map<String, String> getMap() {
	   return mKeyMap;
   }


   public String getVal(BuildInfoKey key) {
	   return mKeyMap.get(key.name());
   }


   /**
    * @return the data source description.  e.g. "NCBI's dbSNP Variant Database"
    */
   public String getDataSourceDescription() {
	   return getVal(BuildInfoKey.DATA_SOURCE_DESCRIPTION);
   }

   /**
    * @return the data source dataset.  e.g. "Variants"
    */
   public String getDataSourceDataset() {
	   return getVal(BuildInfoKey.DATA_SOURCE_DATASET);
   }

}
