package edu.mayo.bior.pipeline.createcatalog;

import static org.junit.Assert.assertEquals;
import static org.junit.Assert.fail;

import java.io.BufferedReader;
import java.io.FileReader;

import org.json.JSONObject;
import org.junit.Ignore;
import org.junit.Test;

import edu.mayo.pipes.history.History;

public class TjsonToCatalogPipeTest {
	
    public static String JSON = "{'_type':'gene','_landmark':'1','_strand':'+','_minBP':10954,'_maxBP':11507,'gene':'LOC100506145','note':'Derived by automated computational analysis using gene prediction method: GNOMON. Supporting evidence includes similarity to: 1 Protein','pseudo':'','GeneID':'100506145'}".replaceAll("'", "\"");

    
    @Test
    public void testExtractLandmark(){
        TjsonToCatalogPipe tjson2Catalog = new TjsonToCatalogPipe(/*jsonCol=*/0, /*isJsonOnly=*/false, /*isModChrom=*/true);
        
        // TODO:  Maintain the JSON keys in the order that they were in the initial string
        //JSONObject jsonObj = new JSONObject(json);
        JSONObject jsonObj = new JSONObject(JSON);
        assertEquals("1", tjson2Catalog.getLandmark(jsonObj, false));
        
        jsonObj = new JSONObject("{'_landmark':'chr1'}");
        assertEquals("1", tjson2Catalog.getLandmark(jsonObj, true));
        
        jsonObj = new JSONObject("{'_landmark':1}");
        assertEquals("1", tjson2Catalog.getLandmark(jsonObj, false));

        jsonObj = new JSONObject("{'_landmark':chr1}");
        assertEquals("1", tjson2Catalog.getLandmark(jsonObj, true));

        jsonObj = new JSONObject("{'_landmark':c17}");
        assertEquals("17", tjson2Catalog.getLandmark(jsonObj, true));

        jsonObj = new JSONObject("{'_landmark':chr}");
        assertEquals("chr", tjson2Catalog.getLandmark(jsonObj, false));

        // Modify the chrom, and change _minBP and _maxBP to integers
        String actual = tjson2Catalog.processLine(new History("{'_landmark':chr1,'_minBP':'100','_maxBP':'200','someRandomKey':'someVal'}")).getMergedData("\t");
        String expected = "1	100	200	{'_landmark':'1','_minBP':100,'_maxBP':200,'someRandomKey':'someVal'}".replaceAll("'", "\"");
        assertEquals(expected, actual);

        
        // Since chrom, min, max not in JSON, just put ".", 0, 0 for them in the 3 tabix columns
        actual = tjson2Catalog.processLine(new History("{'CHROM':'chr1','gene':'AZTA','someRandomKey':'someVal'}")).getMergedData("\t");
        expected = "UNKNOWN	0	0	{'CHROM':'chr1','gene':'AZTA','someRandomKey':'someVal'}".replaceAll("'", "\"");
        assertEquals(expected, actual);

        
        // Try not modifying the chromosome
        tjson2Catalog = new TjsonToCatalogPipe(/*jsonCol=*/0, /*isJsonOnly=*/false, /*isModChrom=*/false);
        actual = tjson2Catalog.processLine(new History("{'_landmark':chr1,'_minBP':'100','_maxBP':'200','someRandomKey':'someVal'}")).getMergedData("\t");
        expected = "chr1	100	200	{'_landmark':'chr1','_minBP':100,'_maxBP':200,'someRandomKey':'someVal'}".replaceAll("'", "\"");
        assertEquals(expected, actual);

        
        // Change the chromosome and build a line that has ONLY the JSON object (not chrom,min,max,json)
        tjson2Catalog = new TjsonToCatalogPipe(/*jsonCol=*/0, /*isJsonOnly=*/true, /*isModChrom=*/true);
        actual = tjson2Catalog.processLine(new History("{'_landmark':'chr1'}")).getMergedData("\t");
        expected = "{'_landmark':'1'}".replaceAll("'", "\"");
        assertEquals(expected, actual);

        System.out.println("JSON object: " + new JSONObject(JSON).toString());
    }

    @Test
    public void testExtractMinBP(){
        TjsonToCatalogPipe tjson2Catalog = new TjsonToCatalogPipe(/*jsonCol=*/0, /*isJsonOnly=*/false, /*isModChrom=*/true);
        JSONObject jsonObj = new JSONObject(JSON);
        assertEquals(10954L, (long)(tjson2Catalog.getMinBP(jsonObj)));

        // Convert _minBP from string to integer
        String actual = tjson2Catalog.processLine(new History("{'_landmark':chr1,'_minBP':'100','_maxBP':'102','_refAllele':'ACG'}")).getMergedData("\t");
        String expected = "1	100	102	{'_landmark':'1','_minBP':100,'_maxBP':102,'_refAllele':'ACG'}".replaceAll("'", "\"");
        assertEquals(expected, actual);
    }

    @Test
    public void testExtractMaxBP(){
        TjsonToCatalogPipe tjson2Catalog = new TjsonToCatalogPipe(/*jsonCol=*/0, /*isJsonOnly=*/false, /*isModChrom=*/true);
        JSONObject jsonObj = new JSONObject(JSON);
        assertEquals(11507L, (long)(tjson2Catalog.getMaxBP(jsonObj)));
        
        
        // Error because ref allele length does not equal (max-min+1)
        try {
        	History history = tjson2Catalog.processLine(new History("{'_landmark':'chr1','_minBP':'100','_maxBP':'101','_refAllele':'A'}"));
        	fail("Should have thrown an exception before this");
        } catch(Exception e) {
        	// OK
        }

        // But, this should work because the _refAllele length == (max-min+1)
        String actual = tjson2Catalog.processLine(new History("{'_landmark':chr1,'_minBP':'100','_maxBP':'102','_refAllele':'ACG','someRandomKey':'someVal'}")).getMergedData("\t");
        String expected = "1	100	102	{'_landmark':'1','_minBP':100,'_maxBP':102,'_refAllele':'ACG','someRandomKey':'someVal'}".replaceAll("'", "\"");
        assertEquals(expected, actual);

        // _maxBP not given, so calculate it - this is useful when the user doesn't want to calculate the _maxBP themselves
        actual = tjson2Catalog.processLine(new History("{'_landmark':chr1,'_minBP':'100','_refAllele':'ACG'}")).getMergedData("\t");
        expected = "1	100	102	{'_landmark':'1','_minBP':100,'_refAllele':'ACG','_maxBP':102}".replaceAll("'", "\"");
        assertEquals(expected, actual);

        // Change the _maxBP from a string to an integer
        actual = tjson2Catalog.processLine(new History("{'_landmark':chr1,'_minBP':'100','_maxBP':'102','_refAllele':'ACG'}")).getMergedData("\t");
        expected = "1	100	102	{'_landmark':'1','_minBP':100,'_maxBP':102,'_refAllele':'ACG'}".replaceAll("'", "\"");
        assertEquals(expected, actual);

    }

    @Test
    public void testCreateLine() {
        TjsonToCatalogPipe tjson2Catalog = new TjsonToCatalogPipe(/*jsonCol=*/0, /*isJsonOnly=*/false, /*isModChrom=*/true);
        String expected = "1" + "\t" + "10954" + "\t" + "11507" + "\t" + JSON;
        String actual = tjson2Catalog.processLine(new History(JSON)).getMergedData("\t");
        assertEquals(expected, actual);
    }
    
    @Test
    public void testJsonArraysAndObjects(){
        TjsonToCatalogPipe tjson2Catalog = new TjsonToCatalogPipe(/*jsonCol=*/0, /*isJsonOnly=*/false, /*isModChrom=*/true);
        JSONObject jsonObj = new JSONObject(JSON);

        String actual = tjson2Catalog.processLine(new History("{'_landmark':chr1,'_minBP':'100','_maxBP':'102','_refAllele':'ACG','_altAlleles':['A','C','G'],'INFO':{'key':'val'},'Path':'/data5/bsi/bior','SpecialChars':'~`!@#$%^&*()_-+=[]{}|;:,.<>?','Quotes':'\\\'\\\"','Slashes':'\\\\,/'}")).getMergedData("\t");
        //																																																																	'\"					  \\,/
        String expected = "1	100	102	{\"_landmark\":\"1\",\"_minBP\":100,\"_maxBP\":102,\"_refAllele\":\"ACG\",\"_altAlleles\":[\"A\",\"C\",\"G\"],\"INFO\":{\"key\":\"val\"},\"Path\":\"/data5/bsi/bior\",\"SpecialChars\":\"~`!@#$%^&*()_-+=[]{}|;:,.<>?\",\"Quotes\":\"\'\\\"\",\"Slashes\":\"\\\\,/\"}";
        assertEquals(expected, actual);
    }

    @Test
    public void testEscapingOfSpecialCharacters(){
        TjsonToCatalogPipe tjson2Catalog = new TjsonToCatalogPipe(/*jsonCol=*/0, /*isJsonOnly=*/false, /*isModChrom=*/true);
        JSONObject jsonObj = new JSONObject(JSON);

        String actual = tjson2Catalog.processLine(new History("{'_landmark':'chr1','_minBP':100,'_maxBP':102,'Path':'/data5/bsi/bior','SpecialChars':'~`!@#$%^&*()_-+=[]{}|;:,.<>?','Quotes':'\\\'\\\"','Slashes':'\\\\,/'}")).getMergedData("\t");
        //	(see comments at end of this line for which characters should be visible)-->							 																    '\"					    \\,/
        String expected = "1	100	102	{\"_landmark\":\"1\",\"_minBP\":100,\"_maxBP\":102,\"Path\":\"/data5/bsi/bior\",\"SpecialChars\":\"~`!@#$%^&*()_-+=[]{}|;:,.<>?\",\"Quotes\":\"\'\\\"\",\"Slashes\":\"\\\\,/\"}";
        assertEquals(expected, actual);
    }
    
    // 6/9/16: This is failing right now but we need to fix it in future (bug fix)
    //         The 'en dash' in Lenox-Gastaut is being replaced with a \u2013 unicode char
    //         and should not be. Because TjsonToCatalog is using a org.json.JSONObject.
    //         We may have needed that impl because gives consistent ordering of json elements
    //         but com.google.gson.JsonObject is not producing unicode. It may not maintiain order though.
    @Ignore
    public void testUnicodeReplacement() {
    	TjsonToCatalogPipe tjson2Catalog = new TjsonToCatalogPipe(/*jsonCol=*/0, /*isJsonOnly=*/true, /*isModChrom=*/true);
    	JSONObject jsonObj = new JSONObject(JSON);
        String jsonStr = "{\"_landmark\":\"1\",\"_minBP\":100,\"_maxBP\":102,\"PHEN\":\"Lenox–Gastaut_syndrome\"}";
        String actual = tjson2Catalog.processLine(new History(jsonStr)).getMergedData("\t");							 																
        String expected = jsonStr;
        assertEquals(expected, actual);
    }

}
