/**
 * bior_pipeline
 *
 * <p>@author Gregory Dougherty</p>
 * Copyright Mayo Clinic, 2014
 *
 */
package edu.mayo.bior.pipeline;

import static edu.mayo.bior.pipeline.Variant2JSONPipe.kNoHeader;
import static edu.mayo.bior.pipeline.Variant2JSONPipe.kBadHeader;
import static edu.mayo.bior.pipeline.Variant2JSONPipe.kFlipAllelesWarning;
import static edu.mayo.bior.pipeline.Variant2JSONPipe.kPositionMatchFailedWarning;
import static org.junit.Assert.*;

import com.tinkerpop.pipes.Pipe;
import com.tinkerpop.pipes.util.Pipeline;
import edu.mayo.bior.pipes.history.MakeFirstLineHeaderPipe;
import edu.mayo.genomicutils.refassembly.RefAssemblyFinder;
import edu.mayo.pipes.JSON.lookup.LookupPipe;
import edu.mayo.pipes.history.*;
import edu.mayo.pipes.iterators.FileLineIterator;
import edu.mayo.pipes.util.metadata.Metadata;
import edu.mayo.pipes.util.test.PipeTestUtils;
import java.io.File;
import java.io.IOException;
import java.util.*;
import org.apache.commons.io.FileUtils;
import org.apache.commons.lang.StringUtils;
import org.junit.After;
import org.junit.Before;
import org.junit.Test;

/**
 * Class to implement JUnit tests for the {@linkplain Variant2JSONPipe}, testing code that generates JSON 
 * from an rsID
 *
 * <p>@author Gregory Dougherty</p>
 */
public class Variant2JsonPipeTest
{
	
	private static final String		kLookupFile = "src/test/resources/testData/tabix/00-First1K_GRCh37.tsv.bgz";
	private static final String		kLookupIndexFile = "src/test/resources/testData/tabix/index/00-First1K_GRCh37.ID.idx.h2.db";
	
	/**
	 * Do any needed setup.  If History still statically stored MetaData, this would create it
	 * @throws IOException 
	 */
	@Before
	public void setup () throws IOException
	{
		// Set the reference assembly base directory 
		File	refAssemblyBaseDir = new File("src/test/resources/ref_assembly");
		RefAssemblyFinder.setRefAssemblyBaseDirStatic(refAssemblyBaseDir.getCanonicalPath());
	}
	
	
	/**
	 * Do any needed tear down.  If History still statically stored MetaData, this would tear it down
	 */
	@After
	public void teardown()
	{
		// Do any needed tear down here
	}
	
	
	/**
	 * Test that validates chromosome and position to JSON using a Variant2JSONPipe
	 * 
	 * @throws IOException	For any file problems
	 */
	@Test
	public void testJSONGenerationFromPosition () throws IOException
	{
		Pipeline<String, String>	thePipe = getVariant2JsonPipeline ("testJSONGenerationFromPosition");
		
		List<String>	input = Arrays.asList ("ID	Chrom	Pos	Allele1	Allele2", "Mike	1	10583", "Ike	1	13327	A	C", 
												"rs368469931	1	10139", "Bike	1	10139", "rs376643643", "rs368469931	1	13980", 
												"rs1	1	51476	G	T", "rs376007522		", "rs368469931", "Tyke	26	13327	A	C", 
												"Trike	25	13327	A	C", "Sike	24	13327	A	C", "Hike	23	13327	A	C");
		List<String>	expected = Arrays.asList (
			"##BIOR=<ID='bior.ToTJson',Operation='testJSONGenerationFromPosition',DataType='JSON',ShortUniqueName='ToTJson'>", 
			"#ID	Chrom	Pos	Allele1	Allele2	bior.ToTJson", 
			"Mike	1	10583			{'_landmark':'1','_refAllele':'G','_altAlleles':['A'],'_minBP':10583," + 
					"'_maxBP':10583}", 
			"Mike	1	10583			{'_landmark':'1','_refAllele':'G','_altAlleles':['C'],'_minBP':10583," + 
					"'_maxBP':10583}", 
			"Mike	1	10583			{'_landmark':'1','_refAllele':'G','_altAlleles':['T'],'_minBP':10583," + 
					"'_maxBP':10583}", 
			"Ike	1	13327	A	C	{'_landmark':'1','_refAllele':'G','_altAlleles':['T'],'_minBP':13327," + 
				"'_maxBP':13327,'WARNING':'" + kFlipAllelesWarning + "'}", 
			"rs368469931	1	10139			{'CHROM':'1','POS':'10139','ID':'rs368469931','REF':'A','ALT':'T'," + 
				"'QUAL':'.','FILTER':'.','INFO':{'RS':368469931,'RSPOS':10139,'dbSNPBuildID':138,'SSR':0," + 
				"'SAO':0,'VP':'0x050000020005000002000100','WGT':1,'VC':'SNV','R5':true,'ASP':true}," + 
				"'_id':'rs368469931','_type':'variant','_landmark':'1','_refAllele':'A','_altAlleles':['T']," + 
				"'_minBP':10139,'_maxBP':10139}", 
			"Bike	1	10139			{'_landmark':'1','_refAllele':'A','_altAlleles':['C'],'_minBP':10139,'_maxBP':10139}", 
			"Bike	1	10139			{'_landmark':'1','_refAllele':'A','_altAlleles':['G'],'_minBP':10139,'_maxBP':10139}", 
			"Bike	1	10139			{'_landmark':'1','_refAllele':'A','_altAlleles':['T'],'_minBP':10139,'_maxBP':10139}", 
			"rs376643643					{'WARNING':'No match found for rs376643643'}", 
			"rs368469931	1	13980			{'CHROM':'1','POS':'10139','ID':'rs368469931','REF':'A','ALT':'T'," + 
				"'QUAL':'.','FILTER':'.','INFO':{'RS':368469931,'RSPOS':10139,'dbSNPBuildID':138,'SSR':0," + 
				"'SAO':0,'VP':'0x050000020005000002000100','WGT':1,'VC':'SNV','R5':true,'ASP':true}," + 
				"'_id':'rs368469931','_type':'variant','_landmark':'1','_refAllele':'A','_altAlleles':['T']," + 
				"'_minBP':10139,'_maxBP':10139,'WARNING':'" + kPositionMatchFailedWarning + "'}", 
			"rs1	1	51476	G	T	{'_landmark':'1','_refAllele':'T','_altAlleles':['G'],'_minBP':51476," + 
				"'_maxBP':51476,'WARNING':'No match found for rs1'}", 
			"rs376007522					{'CHROM':'1','POS':'10109','ID':'rs376007522','REF':'A','ALT':'T'," + 
				"'QUAL':'.','FILTER':'.','INFO':{'RS':376007522,'RSPOS':10109,'dbSNPBuildID':138,'SSR':0," + 
				"'SAO':0,'VP':'0x050000020005000002000100','WGT':1,'VC':'SNV','R5':true,'ASP':true}," + 
				"'_id':'rs376007522','_type':'variant','_landmark':'1','_refAllele':'A','_altAlleles':['T']," + 
				"'_minBP':10109,'_maxBP':10109}", 
			"rs368469931					{'CHROM':'1','POS':'10139','ID':'rs368469931','REF':'A','ALT':'T'," + 
				"'QUAL':'.','FILTER':'.','INFO':{'RS':368469931,'RSPOS':10139,'dbSNPBuildID':138,'SSR':0," + 
				"'SAO':0,'VP':'0x050000020005000002000100','WGT':1,'VC':'SNV','R5':true,'ASP':true}," + 
				"'_id':'rs368469931','_type':'variant','_landmark':'1','_refAllele':'A','_altAlleles':['T']," + 
				"'_minBP':10139,'_maxBP':10139}", // M:13327 differs between GRCh37 and GRCh37.p13. Using line for GRCh37 GTD 6/24/15
//			"Tyke	26	13327	A	C	{'_landmark':'M','_refAllele':'A','_altAlleles':['C'],'_minBP':13327,'_maxBP':13327,}", 
			"Tyke	26	13327	A	C	{'_landmark':'M','_refAllele':'T','_altAlleles':['G'],'_minBP':13327," + 
				"'_maxBP':13327,'WARNING':'" + kFlipAllelesWarning + "'}", 
			"Trike	25	13327	A	C	{'ERROR':'Chromosome `XY`, position `13327` is not valid'}", 
			"Sike	24	13327	A	C	{'_landmark':'Y','_refAllele':'C','_altAlleles':['A'],'_minBP':13327,'_maxBP':13327}", 
			"Hike	23	13327	A	C	{'_landmark':'X','_refAllele':'N','_altAlleles':['A'],'_minBP':13327,'_maxBP':13327}", 
			"Hike	23	13327	A	C	{'_landmark':'X','_refAllele':'N','_altAlleles':['C'],'_minBP':13327,'_maxBP':13327}");
		
		strsToJSON (expected);
		thePipe.setStarts (input);
		
    	List<String>	actual = PipeTestUtils.getResults (thePipe);
//    	PipeTestUtils.printLines (actual);
		PipeTestUtils.assertListsEqual (expected, actual);
	}
	
	
	/**
	 * Test case of single base from Chromosome M
	 */
	@Test
	public void testJSONGenerationFromPosition_singleLine ()
	{
		System.out.println ("From testJSONGenerationFromPosition () - for debugging single failed line");
		
		Pipeline<String, String>	thePipe = getVariant2JsonPipeline ("testJSONGenerationFromPosition_singleLine");
		List<String>				input = Arrays.asList ("ID	Chrom	Pos	Allele1	Allele2", "Tyke	26	13327	A	C");
		List<String>				expected = Arrays.asList (
				"##BIOR=<ID='bior.ToTJson',Operation='testJSONGenerationFromPosition_singleLine',DataType='JSON',ShortUniqueName='ToTJson'>",
				"#ID	Chrom	Pos	Allele1	Allele2	bior.ToTJson",
				"Tyke	26	13327	A	C	{'_landmark':'M','_refAllele':'T','_altAlleles':['G'],'_minBP':13327,'_maxBP':13327," + 
					"'WARNING':'Neither of the alleles supplied matched the reference, so swapped strands'}");
		
		strsToJSON (expected);
		thePipe.setStarts (input);
		
    	List<String>	actual = PipeTestUtils.getResults (thePipe);
//    	PipeTestUtils.printLines (actual);
		PipeTestUtils.assertListsEqual (expected, actual);
	}
	
	
	/**
	 * Test that validates that an rsID with header will be processed successfully
	 * 
	 * @throws IOException	For any file problems
	 */
	@Test
	public void testJSONGenerationRsID () throws IOException
	{
		Pipeline<String, String>	thePipe = getVariant2JsonPipeline ("testJSONGenerationRsID");
		List<String>				input = Arrays.asList ("id", "rs1", "", "foo", "rs2", "rs3", "rs376007522", "");
		List<String>				expected = Arrays.asList (
			"##BIOR=<ID='bior.ToTJson',Operation='testJSONGenerationRsID',DataType='JSON',ShortUniqueName='ToTJson'>", 
			"#id	bior.ToTJson", 
			"rs1	{'WARNING':'No match found for rs1'}", 
			"foo	{'ERROR':'foo is not a valid rsID'}", 
			"rs2	{'WARNING':'No match found for rs2'}", 
			"rs3	{'WARNING':'No match found for rs3'}", 
			"rs376007522	{'CHROM':'1','POS':'10109','ID':'rs376007522','REF':'A','ALT':'T','QUAL':'.'," + 
			"'FILTER':'.','INFO':{'RS':376007522,'RSPOS':10109,'dbSNPBuildID':138,'SSR':0,'SAO':0," + 
			"'VP':'0x050000020005000002000100','WGT':1,'VC':'SNV','R5':true,'ASP':true},'_id':'rs376007522'," + 
			"'_type':'variant','_landmark':'1','_refAllele':'A','_altAlleles':['T'],'_minBP':10109,'_maxBP':10109}");
		
		strsToJSON (expected);
		thePipe.setStarts (input);
		
    	List<String>	actual = PipeTestUtils.getResults (thePipe);
//    	PipeTestUtils.printLines (actual);
		PipeTestUtils.assertListsEqual (expected, actual);
	}
	
	
	/**
	 * Test that validates that the bim file with all its issues will be processed successfully
	 * 
	 * @throws IOException	For any file problems
	 */
	@Test
	public void testJSONGenerationBimFile () throws IOException
	{
		String						bimFile = "src/test/resources/testData/VariantToJson/bb.test.bim";
		String						bimResultsFile = "src/test/resources/testData/VariantToJson/bimResults.txt";
		Pipeline<String, String>	thePipe = getVariant2JsonPipeline ("testJSONGenerationBimFile");
		FileLineIterator			input = new FileLineIterator (bimFile);
		List<String>				expected = FileUtils.readLines (new File (bimResultsFile));
		
		thePipe.setStarts (input);
		
    	List<String>	actual = PipeTestUtils.getResults (thePipe);
    	
//    	PipeTestUtils.printLines (actual);
		PipeTestUtils.assertListsEqual (expected, actual);
	}
	
	
	/**
	 * Test that validates that bad chromosomes with header will be processed successfully
	 * 
	 * @throws IOException	For any file problems
	 */
	@Test
	public void testJSONGenerationBadChrom () throws IOException
	{
		Pipeline<String, String>	thePipe = getVariant2JsonPipeline ("testJSONGenerationBadChrom");
		
		List<String>	input = Arrays.asList ("id	chrom	pos", "t	57	32446842", "chr1.1	100	1", "blah	Q	50000000");
		List<String>	expected = Arrays.asList (
			"##BIOR=<ID='bior.ToTJson',Operation='testJSONGenerationBadChrom',DataType='JSON',ShortUniqueName='ToTJson'>", 
			"#id	chrom	pos	bior.ToTJson", 
			"t	57	32446842	{'ERROR': 'Chromosome `57` is not valid'}", 
			"chr1.1	100	1	{'ERROR': 'Chromosome `100` is not valid'}", 
			"blah	Q	50000000	{'ERROR': 'Chromosome `Q` is not valid'}");
		
		strsToJSON (expected);
		thePipe.setStarts (input);
		
    	List<String>	actual = PipeTestUtils.getResults (thePipe);
//    	PipeTestUtils.printLines (actual);
		PipeTestUtils.assertListsEqual (expected, actual);
	}
	
	
	/**
	 * Test that validates that bad alleles with header will be processed successfully
	 * 
	 * @throws IOException	For any file problems
	 */
	@Test
	public void testJSONGenerationBadAllele () throws IOException
	{
		Pipeline<String, String>	thePipe = getVariant2JsonPipeline ("testJSONGenerationBadAllele");
		
		List<String>	input = Arrays.asList ("id	chrom	pos	allele1	allele2", "test	1	100	N	A", "test2	1	100	G	N");
		List<String>	expected = Arrays.asList (
			"##BIOR=<ID='bior.ToTJson',Operation='testJSONGenerationBadAllele',DataType='JSON',ShortUniqueName='ToTJson'>", 
			"#id	chrom	pos	allele1	allele2	bior.ToTJson", 
			"test	1	100	N	A	{'ERROR': 'Invalid allele `N`'}", 
			"test2	1	100	G	N	{'ERROR': 'Invalid allele `N`'}");
		
		strsToJSON (expected);
		thePipe.setStarts (input);
		
    	List<String>	actual = PipeTestUtils.getResults (thePipe);
//    	PipeTestUtils.printLines (actual);
		PipeTestUtils.assertListsEqual (expected, actual);
	}
	
	
	/**
	 * Test that validates that bad positions with header will be processed successfully
	 * 
	 * @throws IOException	For any file problems
	 */
	@Test
	public void testJSONGenerationBadPosition () throws IOException
	{
		Pipeline<String, String>	thePipe = getVariant2JsonPipeline ("testJSONGenerationBadPosition");
		
		List<String>	input = Arrays.asList ("id	chrom	pos", "test-1	13	-1", "test0	13	0", "chr1.x	1	10000000000", "chr1.x	1	1000000000");
		List<String>	expected = Arrays.asList (
				"##BIOR=<ID='bior.ToTJson',Operation='testJSONGenerationBadPosition',DataType='JSON',ShortUniqueName='ToTJson'>", 
				"#id	chrom	pos	bior.ToTJson", "test-1	13	-1	{'ERROR': 'Invalid position `-1`'}", 
				"test0	13	0	{'ERROR': 'Invalid position `0`'}", 
				"chr1.x	1	10000000000	{'ERROR': 'Invalid position `10000000000`'}", 
				"chr1.x	1	1000000000	{'ERROR': 'Chromosome `1`, position `1000000000` is not valid'}");
		
		strsToJSON (expected);
		thePipe.setStarts (input);
		
    	List<String>	actual = PipeTestUtils.getResults (thePipe);
//    	PipeTestUtils.printLines (actual);
		PipeTestUtils.assertListsEqual (expected, actual);
	}
	
	
	/**
	 * Test that validates that bad chromosomes with header and spaces not tabs will be processed successfully
	 * 
	 * @throws IOException	For any file problems
	 */
	@Test
	public void testJSONGenerationBadChromSpace () throws IOException
	{
		Pipeline<String, String>	thePipe = getVariant2JsonPipeline ("testJSONGenerationBadChromSpace");
		List<String>	input = Arrays.asList ("id      chrom     pos", "rs3     13      32446842", "chr1.1  1       1", "blah    5       50000000");
		boolean			success = false;
		
		thePipe.setStarts (input);
		
		try
		{
	    	PipeTestUtils.getResults (thePipe);
		}
		catch (RuntimeException oops)
		{
			success = oops.getMessage ().equals (kNoHeader);
		}
    	assertTrue ("Did not throw a 'no header' exception", success);
	}
	
	
	/**
	 * Test that validates that a chromosome and position with no header will be caught successfully
	 * 
	 * @throws IOException	For any file problems
	 */
	@Test
	public void testJSONGenerationNoHeader () throws IOException
	{
		Pipeline<String, String>	thePipe = getVariant2JsonPipeline ("testJSONGenerationNoHeader");
		List<String>	input = Arrays.asList ("Mike	1	10583", "Ike	1	13327	A	C", 
												"rs368469931	1	10139", "Bike	1	10139", "rs376643643", "rs368469931	1	13980", 
												"Sun	1	51476	G	T", "rs376007522		", "rs368469931");
		thePipe.setStarts (input);
		
		boolean	success = false;
		try
		{
	    	PipeTestUtils.getResults (thePipe);
		}
		catch (RuntimeException oops)
		{
			success = oops.getMessage ().equals (kNoHeader);
		}
		
    	assertTrue ("Did not throw a 'no header' exception", success);
	}
	
	
	/**
	 * Test that validates that a header missing one of chromosome and position  will be caught successfully
	 * 
	 * @throws IOException	For any file problems
	 */
	@Test
	public void testJSONGenerationBadHeader () throws IOException
	{
		Pipeline<String, String>	thePipe = getVariant2JsonPipeline ("testJSONGenerationBadHeader");
		List<String>				input = Arrays.asList ("id	pos", "chr1.1	1");
		boolean						success = false;
		
		thePipe.setStarts (input);
		
		try
		{
	    	PipeTestUtils.getResults (thePipe);
		}
		catch (RuntimeException oops)
		{
			success = oops.getMessage ().equals (kBadHeader);
		}
		
    	assertTrue ("Did not throw a 'no header' exception", success);
	}
	
	
	/**
	 * Test that validates that an rsID with no header will be caught successfully
	 * 
	 * @throws IOException	For any file problems
	 */
	@Test
	public void testJSONGenerationRsIDNoHeader () throws IOException
	{
		Pipeline<String, String>	thePipe = getVariant2JsonPipeline ("testJSONGenerationRsIDNoHeader");
		List<String>				input = Arrays.asList ("rs1", "rs2", "rs3");
		boolean						success = false;
		
		thePipe.setStarts (input);
		
		try
		{
	    	PipeTestUtils.getResults (thePipe);
		}
		catch (RuntimeException oops)
		{
			success = oops.getMessage ().equals (kNoHeader);
		}
    	assertTrue ("Did not throw a 'no header' exception", success);
	}
	
	
	/** 
	 * Test header with "IDs" - ERROR since none of the required headers are there
	 */
	@Test (expected = RuntimeException.class)
	public void badHeader_error ()
	{
		Pipeline<String, String>	thePipe = getVariant2JsonPipeline ("badHeader_error");
		List<String>				input = Arrays.asList ("IDs", "rs376643643");
		
		thePipe.setStarts (input);
		
    	PipeTestUtils.getResults (thePipe);
	}
	
	
	/**
	 * Validate that a header with no starting "#" works
	 */
	@Test
	public void headerNoHash_ok ()
	{
		Pipeline<String, String>	thePipe = getVariant2JsonPipeline ("headerNoHash_ok");
		
		List<String>	input = Arrays.asList ("ID", "rs368469931");
		List<String>	expected = Arrays.asList (
				"##BIOR=<ID='bior.ToTJson',Operation='headerNoHash_ok',DataType='JSON',ShortUniqueName='ToTJson'>",
				"#ID	bior.ToTJson", 
				"rs368469931	{'CHROM':'1','POS':'10139','ID':'rs368469931','REF':'A','ALT':'T'," + 
						"'QUAL':'.','FILTER':'.','INFO':{'RS':368469931,'RSPOS':10139,'dbSNPBuildID':138,'SSR':0," + 
						"'SAO':0,'VP':'0x050000020005000002000100','WGT':1,'VC':'SNV','R5':true,'ASP':true}," + 
						"'_id':'rs368469931','_type':'variant','_landmark':'1','_refAllele':'A','_altAlleles':['T']," + 
						"'_minBP':10139,'_maxBP':10139}"
				);
		
		strsToJSON (expected);
		thePipe.setStarts (input);
		
    	List<String>	actual = PipeTestUtils.getResults (thePipe);
//    	PipeTestUtils.printLines (actual);
		PipeTestUtils.assertListsEqual (expected, actual);
	}
	
	
	/** 
	 * Test a multi-base ref allele that wraps across multiple lines in the ref assembly catalog 
	 */
	@Test
	public void refWrapsMultipleLinesInRefAssemblyCatalog ()
	{
		Pipeline<String, String>	thePipe = getVariant2JsonPipeline ("refWrapsMultipleLinesInRefAssemblyCatalog");
		
		// Ref at pos = 'T'
		// 1	10011	10080	CCTAACCCTAACCCTAACCCTAACCCTAACCCTAACCCTAACCCTAACCCTAACCCTAACCCTAACCCTA
		// 1	10081	10150	ACCCTAACCCTAACCCTAACCCTAACCCAACCCTAACCCTAACCCTAACCCTAACCCTAACCCTAACCCC
		List<String>	input = Arrays.asList ("CHROM	ID	CM	POS	ALLELE1	ALLELE2", "1	.	.	10079	TAACC	T");
		List<String>	expected = Arrays.asList (
				"##BIOR=<ID='bior.ToTJson',Operation='refWrapsMultipleLinesInRefAssemblyCatalog',DataType='JSON',ShortUniqueName='ToTJson'>",
				"#CHROM	ID	CM	POS	ALLELE1	ALLELE2	bior.ToTJson",
				"1	.	.	10079	TAACC	T	{'_landmark':'1','_refAllele':'TAACC','_altAlleles':['T'],'_minBP':10079,'_maxBP':10083}"
				);
		
		strsToJSON (expected);
		thePipe.setStarts (input);
		
    	List<String>	actual = PipeTestUtils.getResults (thePipe);
//    	PipeTestUtils.printLines (actual);
		PipeTestUtils.assertListsEqual (expected, actual);
	}
	
	
	/** 
	 * Test that validates rsID to JSON using a LookupPipe 
	 */
	@Test
	public void testJSONGenerationFromRsIDHeader ()
	{
		Pipeline<String, String>	thePipe = getVariant2JsonPipeline ("testJSONGenerationFromRsIDHeader");
		List<String>	input = Arrays.asList ("ID", "rs376643643", "rs376007522", "rs368469931", "rs371194064", "rs201752861");
		List<String>	expected = Arrays.asList (
			"##BIOR=<ID='bior.ToTJson',Operation='testJSONGenerationFromRsIDHeader',DataType='JSON',ShortUniqueName='ToTJson'>", 
			"#ID	bior.ToTJson", 
			"rs376643643	{'WARNING':'No match found for rs376643643'}", 
			"rs376007522	{'CHROM':'1','POS':'10109','ID':'rs376007522','REF':'A','ALT':'T'," + 
				"'QUAL':'.','FILTER':'.','INFO':{'RS':376007522,'RSPOS':10109,'dbSNPBuildID':138," + 
				"'SSR':0,'SAO':0,'VP':'0x050000020005000002000100','WGT':1,'VC':'SNV','R5':true," + 
				"'ASP':true},'_id':'rs376007522','_type':'variant','_landmark':'1','_refAllele':'A'," + 
				"'_altAlleles':['T'],'_minBP':10109,'_maxBP':10109}", 
			"rs368469931	{'CHROM':'1','POS':'10139','ID':'rs368469931','REF':'A','ALT':'T'," + 
				"'QUAL':'.','FILTER':'.','INFO':{'RS':368469931,'RSPOS':10139,'dbSNPBuildID':138," + 
				"'SSR':0,'SAO':0,'VP':'0x050000020005000002000100','WGT':1,'VC':'SNV','R5':true," + 
				"'ASP':true},'_id':'rs368469931','_type':'variant','_landmark':'1','_refAllele':'A'," + 
				"'_altAlleles':['T'],'_minBP':10139,'_maxBP':10139}", 
			"rs371194064	{'CHROM':'1','POS':'10150','ID':'rs371194064','REF':'C','ALT':'T'," + 
				"'QUAL':'.','FILTER':'.','INFO':{'RS':371194064,'RSPOS':10150,'dbSNPBuildID':138," + 
				"'SSR':0,'SAO':0,'VP':'0x050000020005000002000100','WGT':1,'VC':'SNV','R5':true," + 
				"'ASP':true},'_id':'rs371194064','_type':'variant','_landmark':'1','_refAllele':'C'," + 
				"'_altAlleles':['T'],'_minBP':10150,'_maxBP':10150}", 
			"rs201752861	{'CHROM':'1','POS':'10177','ID':'rs201752861','REF':'A','ALT':'C'," + 
				"'QUAL':'.','FILTER':'.','INFO':{'RS':201752861,'RSPOS':10177,'dbSNPBuildID':137," + 
				"'SSR':0,'SAO':0,'VP':'0x050000020005000002000100','WGT':1,'VC':'SNV','R5':true," + 
				"'ASP':true},'_id':'rs201752861','_type':'variant','_landmark':'1','_refAllele':'A'," + 
				"'_altAlleles':['C'],'_minBP':10177,'_maxBP':10177}");
		
		strsToJSON (expected);
		thePipe.setStarts (input);
		
    	List<String>	actual = PipeTestUtils.getResults (thePipe);
//    	PipeTestUtils.printLines (actual);
		PipeTestUtils.assertListsEqual (expected, actual);
	}
	
	
	
	/**
	 * Test behavior when 
	 * From dbSNP: 
	 *  11015010150{"CHROM":"1","POS":"10150","ID":"rs371194064","REF":"C","ALT":"T","QUAL":".","FILTER":".",
	 * "INFO":{"RS":371194064,"RSPOS":10150,"dbSNPBuildID":138,"SSR":0,"SAO":0,"VP":"0x050000020001000002000100",
	 * 			"WGT":1,"VC":"SNV","R5":true,"OTHERKG":true},"_id":"rs371194064","_type":"variant","_landmark":"1",
	 * 			"_refAllele":"C","_altAlleles":["T"],"_minBP":10150,"_maxBP":10150}
	 * From ref assembly:
	 * 11008110150ACCCTAACCCTAACCCTAACCCTAACCCAACCCTAACCCTAACCCTAACCCTAACCCTAACCCTAACCCC
	 * 
	 */
	@Test
	public void rsIdMatchesKnownVariant_butRefDoesNotMatch ()
	{
		Pipeline<String, String>	thePipe = getVariant2JsonPipeline ("rsIdMatchesKnownVariant_butRefDoesNotMatch");
		List<String>	input = Arrays.asList ("CHROM	ID	CM	POS	ALLELE1	ALLELE2", "1	rs371194064	.	10150	A	G");
		List<String>	expected = Arrays.asList (
				"##BIOR=<ID='bior.ToTJson',Operation='rsIdMatchesKnownVariant_butRefDoesNotMatch',DataType='JSON',ShortUniqueName='ToTJson'>",
				"#CHROM	ID	CM	POS	ALLELE1	ALLELE2	bior.ToTJson",
				"1	rs371194064	.	10150	A	G	{'CHROM':'1','POS':'10150','ID':'rs371194064','REF':'C','ALT':'T','QUAL':'.'," + 
						"'FILTER':'.','INFO':{'RS':371194064,'RSPOS':10150,'dbSNPBuildID':138,'SSR':0,'SAO':0,'VP':'0x050000020005000002000100'," + 
						"'WGT':1,'VC':'SNV','R5':true,'ASP':true},'_id':'rs371194064','_type':'variant','_landmark':'1','_refAllele':'C'," + 
						"'_altAlleles':['T'],'_minBP':10150,'_maxBP':10150}" 
		);
		
		strsToJSON (expected);
		thePipe.setStarts (input);
		
    	List<String>	actual = PipeTestUtils.getResults (thePipe);
//    	PipeTestUtils.printLines (actual);
		PipeTestUtils.assertListsEqual (expected, actual);
	//	fail("NOTE:  SHOULD THERE BE A WARNING IN THE ABOVE CASE ABOUT THE REF NOT MATCHING???");
	}
	
	
	/** 
	 * Test a multi-base ref allele that wraps across multiple lines in the ref assembly catalog, 
	 * and the passed in alleles are not valid
	 */
	@Test
	public void refWrapsTwoLinesInRefAssemblyCatalog_butNoMatch ()
	{
		Pipeline<String, String>	thePipe = getVariant2JsonPipeline ("refWrapsTwoLinesInRefAssemblyCatalog_butNoMatch");
		// This line should wrap the first two lines of the ref assembly catalog
		List<String>	input = Arrays.asList ("CHROM	ID	CM	POS	ALLELE1	ALLELE2", "1	.	.	10079	AAAAA	TTTTT");
		List<String>	expected = Arrays.asList (
			"##BIOR=<ID='bior.ToTJson',Operation='refWrapsTwoLinesInRefAssemblyCatalog_butNoMatch',DataType='JSON',ShortUniqueName='ToTJson'>",
			"#CHROM	ID	CM	POS	ALLELE1	ALLELE2	bior.ToTJson",
			"1	.	.	10079	AAAAA	TTTTT	{'_landmark':'1','_refAllele':'T','_altAlleles':['AAAAA'],'_minBP':10079,'_maxBP':10079}", 
			"1	.	.	10079	AAAAA	TTTTT	{'_landmark':'1','_refAllele':'T','_altAlleles':['TTTTT'],'_minBP':10079,'_maxBP':10079}"
		);
		
		strsToJSON (expected);
		thePipe.setStarts (input);
		
    	List<String>	actual = PipeTestUtils.getResults (thePipe);
//    	PipeTestUtils.printLines (actual);
		PipeTestUtils.assertListsEqual (expected, actual);
	}
	
	
	/** When dbSNP went from having "POS" as a string in version 142 to an int in the JSON in 150, it broke the bior_variant_to_tjson command */
	@Test
	public void testClassCastExceptionWhenDbsnpPosFieldIsInteger() {
		String dir = "src/test/resources/testData/testVariantToTjsonWhenDbsnpHasPosFieldAsInteger/";
		String dbsnpCatalog = dir + "dbsnp150_withPosAsInt.tsv.bgz";
		String dbsnpH2Index = dir + "index/dbsnp150_withPosAsInt.ID.idx.h2.db";
		Pipeline<String, String>	thePipe = getVariant2JsonPipeline ("testOddCaseOfClassCastException", dbsnpCatalog, dbsnpH2Index);
		// This line should wrap the first two lines of the ref assembly catalog
		List<String>	input = Arrays.asList (
				"ID	CHROM	POS",
				"rs62651026	1	10108",
				"rs376007522	1	10109"
				);
		List<String>	expected = Arrays.asList (
				"##BIOR=<ID=\"bior.ToTJson\",Operation=\"testOddCaseOfClassCastException\",DataType=\"JSON\",ShortUniqueName=\"ToTJson\">",
				"#ID	CHROM	POS	bior.ToTJson",
				"rs62651026	1	10108	{'CHROM':'1','POS':10108,'ID':'rs62651026','REF':'C','ALT':'T','QUAL':'.','FILTER':'.','INFO':{'RS':62651026,'RSPOS':10108,'dbSNPBuildID':129,'SSR':0,'SAO':0,'VP':'0x050000020005000002000100','WGT':1,'VC':'SNV','R5':true,'ASP':true},'_id':'rs62651026','_type':'variant','_landmark':'1','_refAllele':'C','_altAlleles':['T'],'_minBP':10108,'_maxBP':10108}".replace("'", "\""),
				"rs376007522	1	10109	{'CHROM':'1','POS':10109,'ID':'rs376007522','REF':'A','ALT':'T','QUAL':'.','FILTER':'.','INFO':{'RS':376007522,'RSPOS':10109,'dbSNPBuildID':138,'SSR':0,'SAO':0,'VP':'0x050000020005000002000100','WGT':1,'VC':'SNV','R5':true,'ASP':true},'_id':'rs376007522','_type':'variant','_landmark':'1','_refAllele':'A','_altAlleles':['T'],'_minBP':10109,'_maxBP':10109}"
		);
		
		strsToJSON (expected);
		thePipe.setStarts (input);
		
    	List<String>	actual = PipeTestUtils.getResults (thePipe);
    	PipeTestUtils.printLines (actual);
		assertEquals(StringUtils.join(expected, "\n"), StringUtils.join(actual, "\n"));
	}
	
	/**
	 * Iterate over a List, taking every String that might have "'" in it and replace them with "\""
	 * 
	 * @param theStrs	List of Strings to "JSONify"
	 */
	private static final void strsToJSON (List<String> theStrs)
	{
		int	len = theStrs.size ();
		
		for (int i = 0; i < len; ++i)
			theStrs.set (i, strToJSON (theStrs.get (i)));
	}
	
	
	/**
	 * Take a String that might have "'" in it and replace them with "\"", 
	 * then find any "`" and replace them with "'"
	 * 
	 * @param theString	String to "JSONify"
	 * @return	JSONified string
	 */
	private static final String strToJSON (String theString)
	{
		return theString.replace ('\'', '"').replace ('`', '\'');
	}
	

	/**
	 * Create the {@linkplain Pipeline} to be used in the tests
	 *  
	 * @param meta	Name of the test being run
	 * @return	A String in, String out {@linkplain Pipeline}
	 */
	private Pipeline<String, String> getVariant2JsonPipeline (String meta) {
		return getVariant2JsonPipeline(meta, kLookupFile, kLookupIndexFile);
	}
	
	/**
	 * Create the {@linkplain Pipeline} to be used in the tests
	 *  
	 * @param meta	Name of the test being run
	 * @return	A String in, String out {@linkplain Pipeline}
	 */
	private Pipeline<String, String> getVariant2JsonPipeline (String meta, String dbsnpLookupFile, String dbsnpH2IdIndexFile)
	{
		Metadata					metadata = new Metadata (meta);
		Pipe<String, String>		addHeader = new MakeFirstLineHeaderPipe ();
		Pipe<String, History>		setup = new HistoryInPipe (metadata);
		Pipe<String, History>		preLogic = new Pipeline<String, History> (addHeader, setup);
		LookupPipe					lookup = new LookupPipe (dbsnpLookupFile, dbsnpH2IdIndexFile);
		
		Variant2JSONPipe			logic = new Variant2JSONPipe (lookup, null);
		Pipe<History, String>		postLogic = new HistoryOutPipe ();
		Pipeline<String, String>	thePipe = new Pipeline<String, String> (preLogic, logic, postLogic);
		
		return thePipe;
	}
	
}
