package edu.mayo.bior.cli.func;

import static org.junit.Assert.assertEquals;
import static org.junit.Assert.fail;

import java.io.File;
import java.io.IOException;

import org.junit.Test;

import com.jayway.jsonpath.JsonPath;

import edu.mayo.bior.cli.cmd.VCF2VariantCommand;
import edu.mayo.pipes.util.test.PipeTestUtils;

public class VcfToTjsonITCase extends BaseFunctionalTest {

	@Test
	public void test() throws IOException, InterruptedException {
        System.out.println("VcfToTjsonITCase.test");

        String stdin = loadFile(new File("src/test/resources/test.vcf"));

		CommandOutput out = executeScript("bior_vcf_to_tjson", stdin);

		assertEquals("STDERR:"+out.stderr+"\n"+"STDOUT:"+out.stdout, 0, out.exit);
		assertEquals("", out.stderr);

		String header = getHeader(out.stdout);
		assertEquals(
				"##fileformat=VCFv4.0" +"\n" +
				"##fileDate=20090805" +"\n" +
				"##source=myImputationProgramV3.1" +"\n" +
				"##reference=1000GenomesPilot-NCBI36" +"\n" +
				"##phasing=partial" +"\n" +
				"##INFO=<ID=NS,Number=1,Type=Integer,Description=\"Number of Samples With Data\">" +"\n" +
				"##INFO=<ID=DP,Number=1,Type=Integer,Description=\"Total Depth\">" +"\n" +
				"##INFO=<ID=AF,Number=.,Type=Float,Description=\"Allele Frequency\">" +"\n" +
				"##INFO=<ID=AA,Number=1,Type=String,Description=\"Ancestral Allele\">" +"\n" +
				"##INFO=<ID=DB,Number=0,Type=Flag,Description=\"dbSNP membership, build 129\">" +"\n" +
				"##INFO=<ID=H2,Number=0,Type=Flag,Description=\"HapMap2 membership\">" +"\n" +
				"##BIOR=<ID=\"bior.ToTJson\",Operation=\"bior_vcf_to_tjson\",DataType=\"JSON\",ShortUniqueName=\"ToTJson\">" + "\n" +
 				"#CHROM\tPOS\tID\tREF\tALT\tQUAL\tFILTER\tINFO\tbior.ToTJson"+"\n", 
				header);
		
		// pull out just data rows
		String data = out.stdout.replace(header, "");
		
		// JSON should be added as last column (9th)
		String[] cols = data.split("\t");
		int		numCols = cols.length;
		assertEquals("Columns should be 9 but it is " + numCols, 9, numCols);
		
		String json = cols[numCols - 1];

        assertEquals("20",	JsonPath.compile("CHROM").read(json));
        assertEquals(14370,	JsonPath.compile("POS").read(json));
        assertEquals(3,		JsonPath.compile("INFO.NS").read(json));
	}
	
	/** When an error occurs in a column, make sure that the error goes to stderr, and NOT stdout or it will interfere with the data
	 *  Also, in this case, the error is on an integer type, so the offending field should be ignored 
	 * @throws InterruptedException 
	 * @throws IOException */
	@Test
	public void badField() throws IOException, InterruptedException {
		String inputVcf = "##INFO=<ID=MyKey,Number=1,Type=Integer,Description=\"A field that normally only has one value\">\n"
				+ 		  "##INFO=<ID=YourKey,Number=1,Type=Integer,Description=\"A field that normally only has one value\">\n"
			+ PipeTestUtils.combine(new String[] { "#CHROM", "POS", "ID", "REF", "ALT", "QUAL", "FILTER", "INFO" }, "\t") + "\n"
			+ PipeTestUtils.combine(new String[] { "1",      "100", "rs123", "A","C",   ".",    ".",      "MyKey=0|0|0;YourKey=1"}, "\t") + "\n";
		
		CommandOutput out = executeScript("bior_vcf_to_tjson", inputVcf);

		String EXPECTED_ERR = "WARNING: Invalid VCF Line, id=MyKey, 0|0|0 does not appear to be an Integer : 1	100	rs123	A	C	.	.	MyKey=0|0|0;YourKey=1\n";
		// NOTE: The MyKey pair will be excluded from the JSON because it could not be parsed!
		String EXPECTED_OUT = "##INFO=<ID=MyKey,Number=1,Type=Integer,Description=\"A field that normally only has one value\">\n"
						 	+ "##INFO=<ID=YourKey,Number=1,Type=Integer,Description=\"A field that normally only has one value\">\n"
							+ "##BIOR=<ID=\"bior.ToTJson\",Operation=\"bior_vcf_to_tjson\",DataType=\"JSON\",ShortUniqueName=\"ToTJson\">\n"
							+ "#CHROM	POS	ID	REF	ALT	QUAL	FILTER	INFO	bior.ToTJson\n"
							+ ("1	100	rs123	A	C	.	.	MyKey=0|0|0;YourKey=1	{'CHROM':'1','POS':100,'ID':'rs123','REF':'A','ALT':'C',"
							+  "'INFO':{'YourKey':1},'_id':'rs123','_type':'variant','_landmark':'1','_refAllele':'A','_altAlleles':['C'],'_minBP':100,'_maxBP':100}\n").replaceAll("'", "\"");
		assertEquals("STDERR:" + out.stderr + "\n"+"STDOUT:" + out.stdout, 0, out.exit);
		assertEquals(EXPECTED_ERR, out.stderr);
		assertEquals(EXPECTED_OUT, out.stdout);
		
	}


	@Test
	/** Convert escaped characters back into their original form for TJSON.
	 *  %3A   : (colon)
		%3B   ; (semicolon)
		%3D   = (equal sign)
		%25   % (percent sign)
		%2C   , (comma)
		%0D   CR
		%0A   LF
		%09   TAB */
	public void decodeSpecialCharacters() throws IOException, InterruptedException {
		String inputVcf = "##INFO=<ID=Str,Number=.,Type=String,Description=\"\">\n"
				+ 		  "##INFO=<ID=Int,Number=.,Type=Integer,Description=\"\">\n"
			+ PipeTestUtils.combine(new String[] { "#CHROM", "POS", "ID", "REF", "ALT", "QUAL", "FILTER", "INFO" }, 					"\t") + "\n"
			// These should not be converted
			+ PipeTestUtils.combine(new String[] { "1",      "100", "rs123", "A","C",   ".",    ".",      "Str=a,b,c|d:e%f;Int=0,1,2"},	"\t") + "\n"
			// These WILL be converted - same as line above, but with encoded characters (and a tab at end of Str)
			+ PipeTestUtils.combine(new String[] { "1",      "101", "rs1234","A","C",   ".",    ".",      "Str=a%2Cb%2Cc|d%3Ae%25f%09;Int=3"},	"\t") + "\n";
		
		CommandOutput out = runCmdApp(inputVcf, new VCF2VariantCommand(), "bior_vcf_to_tjson");
		
		printOutputsForDebugging(out, "");

		String EXPECTED_OUT = "##INFO=<ID=Str,Number=.,Type=String,Description=\"\">\n"
						 	+ "##INFO=<ID=Int,Number=.,Type=Integer,Description=\"\">\n"
							+ "##BIOR=<ID=\"bior.ToTJson\",Operation=\"bior_vcf_to_tjson\",DataType=\"JSON\",ShortUniqueName=\"ToTJson\">\n"
							+ "#CHROM	POS	ID	REF	ALT	QUAL	FILTER	INFO	bior.ToTJson\n"
				+ swapQuotes( "1	100	rs123	A	C	.	.	Str=a,b,c|d:e%f;Int=0,1,2	{'CHROM':'1','POS':100,'ID':'rs123','REF':'A','ALT':'C','INFO':{'Str':['a','b','c|d:e%f'],'Int':[0,1,2]},'_id':'rs123','_type':'variant','_landmark':'1','_refAllele':'A','_altAlleles':['C'],'_minBP':100,'_maxBP':100}\n")
				+ swapQuotes( "1	101	rs1234	A	C	.	.	Str=a%2Cb%2Cc|d%3Ae%25f%09;Int=3	{'CHROM':'1','POS':101,'ID':'rs1234','REF':'A','ALT':'C','INFO':{'Str':['a,b,c|d:e%f%09'],'Int':[3]},'_id':'rs1234','_type':'variant','_landmark':'1','_refAllele':'A','_altAlleles':['C'],'_minBP':101,'_maxBP':101}\n");
		assertEquals("STDERR:" + out.stderr + "\n"+"STDOUT:" + out.stdout, 0, out.exit);
		assertEquals("", out.stderr);
		assertEquals(EXPECTED_OUT, out.stdout);
		
	}

}
