package edu.mayo.bior.cli.func;

import static org.junit.Assert.*;

import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.sql.SQLException;
import java.sql.Savepoint;
import java.util.Properties;



import org.apache.commons.io.FileUtils;
//import org.h2.util.Utils;
import org.junit.After;
import org.junit.Before;
import org.junit.BeforeClass;
import org.junit.Test;
import org.junit.rules.TemporaryFolder;

import edu.mayo.pipes.JSON.lookup.lookupUtils.IndexUtils;
import edu.mayo.bior.catalog.index.IndexDatabaseCreator;

//import edu.mayo.pipes.util.index.IndexDatabaseCreator;

//==========================================================================================
// NOTE: These testcases will require these steps to be performed before running:
// 1) From the command line, run (inserting the path to the bior-toolkit dir):
//		export BIOR_LITE_HOME=/home/bior/bior_lite/bior-toolkit/target/bior-toolkit-0.0.2-SNAPSHOT
// 2) From the bior-toolkit project, run command:
//		To update the code:
//			svn up
//		To build the target directory and skip the tests initially:
//			mvn clean package install -DskipTests
// 3) Also, if the pipes or pipeline code changes, then the mvn command must be run again to push the code into the target directory
//==========================================================================================
public class IndexCommandITCase extends BaseFunctionalTest {
	private File mCatalogToCopy = new File("src/test/resources/sameVariantCatalog.tsv.bgz");
	private File mCatalog = null;
	private File mIndexId = null;
	private File mIndexChrom = null;
	private File mIndexNested = null;
	private File mIndexNotInUsualPlace = null;
	
	
	private File mTempDir = null;
	private File mTempOutText = null;
	
	final String JSON_PATH = "ID";  //rsIds

	@BeforeClass
	public static void beforeAll() throws FileNotFoundException {
		BaseFunctionalTest.setBiorToolkitCmdsRequired(true);
	}
	
	@Before
	public void beforeEach() throws IOException {
		// Copy the catalog over to the temp directory (so the indexes we create don't have to be repeatedly checked into SVN
		TemporaryFolder tempFolder = new TemporaryFolder();
		tempFolder.create();
		mTempDir = tempFolder.newFolder();
		mCatalog = new File(mTempDir, mCatalogToCopy.getName());
		FileUtils.copyFile(mCatalogToCopy, mCatalog);
		
		mIndexId 		= new File(mTempDir, "index/sameVariantCatalog.ID.idx.h2.db");
		mIndexChrom 	= new File(mTempDir, "index/sameVariantCatalog.CHROM.idx.h2.db");
		mIndexNested 	= new File(mTempDir, "index/sameVariantCatalog.INFO.different_bc_ref_notmatch.idx.h2.db");
		mIndexNotInUsualPlace = new File(mTempDir, "someOtherDir/tempIndex/myIndexFile.Hgnc.idx.h2.db");
		
		mTempOutText    = new File(mTempDir, "temp.txt");
	}
	
	
	@Test
	public void help() throws IOException, InterruptedException {
        System.out.println("IndexCommandITCase.help");
        CommandOutput out = executeScript("bior_index_catalog", null);
		String expected = loadFile(new File("src/test/resources/index/IndexCommand.expectedOutput.help.txt"));
		// bior_index --help
		out = executeScript("bior_index_catalog", null, "--help");
		assertEquals(out.stdout, expected, out.stdout);
		assertFalse(mIndexId.exists());
		assertNoErrors(out);

		// bior_index_catalog -h
		out = executeScript("bior_index_catalog", null, "-h");
		assertEquals(out.stdout, expected, out.stdout);
		IndexUtils.writeToFile(out.stdout, mTempOutText.getCanonicalPath());
		assertFalse(mIndexId.exists());
		assertNoErrors(out);
	}

	@Test
	public void badCmd_noCatalogOrJsonFlag() throws IOException, InterruptedException {
        System.out.println("IndexCommandITCase.badCmd_noCatalogOrJsonFlag");
        // bior_index_catalog
		CommandOutput out = executeScript("bior_index_catalog", null);
		String expected = loadFile(new File("src/test/resources/index/IndexCommand.expectedOutput.missingOptions.txt"));
		IndexUtils.writeToFile(out.stdout, mTempOutText.getCanonicalPath());
		assertEquals(out.stderr, expected, out.stderr);
		assertFalse(mIndexId.exists());
		assertEquals(1, out.exit);
	}

	@Test
	public void badCmd_noCatalog() throws IOException, InterruptedException {
        System.out.println("IndexCommandITCase.badCmd_noCatalog");
        // bior_index_catalog -k ID
		CommandOutput out = executeScript("bior_index_catalog", null, "-p", JSON_PATH);
		String expected = loadFile(new File("src/test/resources/index/IndexCommand.expectedOutput.missingCatalog.txt"));
		IndexUtils.writeToFile(out.stdout, mTempOutText.getCanonicalPath());
		assertEquals(out.stderr, expected, out.stderr);
		assertFalse(mIndexId.exists());
		assertEquals(1, out.exit);
	}

	@Test
	public void badCmd_noJsonFlag() throws IOException, InterruptedException {
        System.out.println("IndexCommandITCase.badCmd_noJsonFlag");
        // bior_index_catalog -d src/test/resources/sameVariantCatalog.tsv.bgz
		CommandOutput out = executeScript("bior_index_catalog", null, "-d", mCatalog.getCanonicalPath());
		String expected = loadFile(new File("src/test/resources/index/IndexCommand.expectedOutput.missingKeyOption.txt"));
		IndexUtils.writeToFile(out.stdout, mTempOutText.getCanonicalPath());
		assertEquals(out.stderr, expected, out.stderr);
		assertFalse(mIndexId.exists());
		assertEquals(1, out.exit);
	}

	@Test
	public void badOption() throws IOException, InterruptedException, SQLException, ClassNotFoundException {
        System.out.println("IndexCommandITCase.badOption");
        // bior_index_catalog -p ID -k ID -d src/test/resources/sameVariantCatalog.tsv.bgz
		CommandOutput out = executeScript("bior_index_catalog", null, "-p", mCatalog.getCanonicalPath(), "-k", JSON_PATH, "-d", "badArg1", "-z", "badArg2");
		String expected = loadFile(new File("src/test/resources/index/IndexCommand.expectedOutput.badOption.txt"));
		IndexUtils.writeToFile(out.stdout, mTempOutText.getCanonicalPath());
		assertEquals(out.stderr, expected, out.stderr);
		assertFalse(mIndexId.exists());
		assertEquals(1, out.exit);
	}


	@Test
	public void tooManyArgs() throws IOException, InterruptedException, SQLException, ClassNotFoundException {
        System.out.println("IndexCommandITCase.tooManyArgs");
        // bior_index_catalog -p ID -d src/test/resources/sameVariantCatalog.tsv.bgz  someUnnecessaryArg
		CommandOutput out = executeScript("bior_index_catalog", null, "-p", JSON_PATH, "-d", mCatalog.getCanonicalPath(), "someUnnecessaryArg");
		String expected = loadFile(new File("src/test/resources/index/IndexCommand.expectedOutput.tooManyArgs.txt"));
		IndexUtils.writeToFile(out.stdout, mTempOutText.getCanonicalPath());
		assertEquals(out.stderr, expected, out.stderr);
		assertFalse(mIndexId.exists());
		assertEquals(1, out.exit);
	}

	/** Json path not found in ANY rows - should return "1" for exit code when it realizes the index is empty */
	public void jsonPathNotFound() throws IOException, InterruptedException, SQLException, ClassNotFoundException {
        System.out.println("IndexCommandITCase.jsonPathNotFound");
        // bior_index_catalog  -p SomeBadJsonPath  -d src/test/resources/sameVariantCatalog.tsv.bgz
		CommandOutput out = executeScript("bior_index_catalog", null, "-p", "SomeBadJsonPath", "-d", mCatalog.getCanonicalPath());
		IndexUtils.writeToFile(out.stdout, mTempOutText.getCanonicalPath());
		assertTrue(out.stderr, out.stderr.contains("java.lang.IllegalArgumentException: There were no keys indexed!  Check your inputs and try again."));
		assertEquals(1, out.exit);
	}
	
	
	@Test
	public void jsonPath() throws IOException, InterruptedException, SQLException, ClassNotFoundException {
        System.out.println("IndexCommandITCase.jsonPath");
        // bior_index_catalog    -p ID  -d src/test/resources/sameVariantCatalog.tsv.bgz
		System.out.println("bior_index_catalog " +" -p " + JSON_PATH +" -d " + mCatalog.getCanonicalPath());
		CommandOutput out = executeScript("bior_index_catalog", null, "-p", JSON_PATH, "-d", mCatalog.getCanonicalPath());
		
		IndexUtils.writeToFile(out.stdout, mTempOutText.getCanonicalPath());
		assertDbRows(38, mIndexId.getCanonicalPath());
		assertEquals(loadFile(new File("src/test/resources/index/IndexCommand.expectedOutput.ID.txt")),
			     IndexDatabaseCreator.getTableAsString(mIndexId));
		assertNoErrors(out);
	}

	@Test
	public void indexParm() throws IOException, InterruptedException, SQLException, ClassNotFoundException {
        System.out.println("IndexCommandITCase.indexParam");
        CommandOutput out = executeScript("bior_index_catalog", null, "-p", JSON_PATH, "-d", mCatalog.getCanonicalPath(), "-i", mIndexId.getCanonicalPath());
		IndexUtils.writeToFile(out.stdout, mTempOutText.getCanonicalPath());
		assertDbRows(38, mIndexId.getCanonicalPath());
		assertEquals(loadFile(new File("src/test/resources/index/IndexCommand.expectedOutput.ID.txt")),
			     IndexDatabaseCreator.getTableAsString(mIndexId));
		assertNoErrors(out);
	}

	@Test
	public void longOptionNames() throws IOException, InterruptedException, SQLException, ClassNotFoundException {
        System.out.println("IndexCommandITCase.longOptionNames");
        // bior_index_catalog --path ID  --database src/test/resources/sameVariantCatalog.tsv.bgz  --index  src/test/resources/index/sameVariantCatalog.ID.idx.h2.db
		CommandOutput out = executeScript("bior_index_catalog", null, "--path", JSON_PATH, "--database", mCatalog.getCanonicalPath(), "--index", mIndexId.getCanonicalPath());
		IndexUtils.writeToFile(out.stdout, mTempOutText.getCanonicalPath());
		assertDbRows(38, mIndexId.getCanonicalPath());
		assertEquals(loadFile(new File("src/test/resources/index/IndexCommand.expectedOutput.ID.txt")),
			     IndexDatabaseCreator.getTableAsString(mIndexId));
		assertNoErrors(out);
	}

	@Test
	public void userIndexNotInDefaultDir() throws IOException, InterruptedException, SQLException, ClassNotFoundException {
        System.out.println("IndexCommandITCase.userIndexNotInDefaultDir");
        // bior_index_catalog -p ID  -d src/test/resources/sameVariantCatalog.tsv.bgz  -i  src/test/resources/index/sameVariantCatalog.ID.idx.h2.db
		CommandOutput out = executeScript("bior_index_catalog", null, "-p", JSON_PATH, "-d", mCatalog.getCanonicalPath(), "-i", mIndexNotInUsualPlace.getCanonicalPath());
		IndexUtils.writeToFile(out.stdout, mTempOutText.getCanonicalPath());
	//	assertNoErrors(out);
		assertDbRows(38, mIndexNotInUsualPlace.getCanonicalPath());
		assertEquals(loadFile(new File("src/test/resources/index/IndexCommand.expectedOutput.ID.txt")),
			     IndexDatabaseCreator.getTableAsString(mIndexNotInUsualPlace));
	}

	@Test
	public void keyIsInt() throws IOException, InterruptedException, SQLException, ClassNotFoundException {
        System.out.println("IndexCommandITCase.keyIsInt");
        CommandOutput out = executeScript("bior_index_catalog", null, "-p", "CHROM",  "-d", mCatalog.getCanonicalPath());
		IndexUtils.writeToFile(out.stdout, mTempOutText.getCanonicalPath());
		assertDbRows(38, mIndexChrom.getCanonicalPath());
		assertEquals(loadFile(new File("src/test/resources/index/IndexCommand.expectedOutput.CHROM.txt")),
			     IndexDatabaseCreator.getTableAsString(mIndexChrom));
		assertNoErrors(out);
	}

	@Test
	public void keyIsString() throws IOException, InterruptedException, SQLException, ClassNotFoundException {
        System.out.println("IndexCommandITCase.keyIsString");
        CommandOutput out = executeScript("bior_index_catalog", null, "-p", JSON_PATH, "-d", mCatalog.getCanonicalPath());
		IndexUtils.writeToFile(out.stdout, mTempOutText.getCanonicalPath());
		assertDbRows(38, mIndexId.getCanonicalPath());
		assertEquals(loadFile(new File("src/test/resources/index/IndexCommand.expectedOutput.ID.txt")),
			     IndexDatabaseCreator.getTableAsString(mIndexId));
		assertNoErrors(out);
	}


	@Test
	public void jsonPathNested_matchOne() throws IOException, InterruptedException, SQLException, ClassNotFoundException {
        System.out.println("IndexCommandITCase.jsonPathNested_matchOne");
        CommandOutput out = executeScript("bior_index_catalog", null, "-p", "INFO.different_bc_ref_notmatch", "-d", mCatalog.getCanonicalPath());
		IndexUtils.writeToFile(out.stdout, mTempOutText.getCanonicalPath());
		assertDbRows(1, mIndexNested.getCanonicalPath());
		assertEquals("	KEY	FILEPOS\n1)	true	5403\n", 
			     IndexDatabaseCreator.getTableAsString(mIndexNested));
		assertNoErrors(out);
	}

	
	@Test
	/** The key points to a JSON array that yields multiple rows in the index per catalog row.
	 *  Ex:  {"Genes":["BRCA1","MTHFR","FOX1"]}  */
	public void jsonArrayForKey() throws IOException, InterruptedException, SQLException, ClassNotFoundException {
        System.out.println("IndexCommandITCase.jsonArrayForKey");
        File catalogPathToCopy =  new File("src/test/resources/catalogWithJsonArray.tsv.bgz");
        
        // First copy the catalog to a temp folder so we don't generate new files that have to be checked in repeatedly to SVN
        File catalog = new File(mTempDir, catalogPathToCopy.getName());
        FileUtils.copyFile(catalogPathToCopy, catalog);
        
        CommandOutput out = executeScript("bior_index_catalog", null, "-p", "Genes", "-d", catalog.getCanonicalPath());
		IndexUtils.writeToFile(out.stdout, mTempOutText.getCanonicalPath());
		//assertDbRows(6, indexPath);
		String expected = "\tKEY\tFILEPOS\n"
				+         "1)\tGeneA\t20\n"
				+         "2)\tGeneB\t20\n"
				+         "3)\tGeneC\t20\n"
				+         "4)\tGeneD\t78\n"
				+         "5)\tGeneE\t78\n"
				+         "6)\tGeneF\t78\n";
		File indexFile = new File(mTempDir, "index/catalogWithJsonArray.Genes.idx.h2.db");
		String actual = IndexDatabaseCreator.getTableAsString(indexFile);
		assertEquals(expected, actual);
		assertNoErrors(out);
	}

	//========================================================================
	
	private void assertNoErrors(CommandOutput out) {
		// Assert no errors and nothing in stderr
		assertEquals(out.stderr, 0, out.exit);
		assertEquals("", out.stderr);
	}
	
	private void assertDbRows(int rowsExpected, String h2DbPath) throws SQLException, ClassNotFoundException, IOException {
		File indexFile = new File(h2DbPath);
		assertTrue(indexFile.exists());
		assertEquals(rowsExpected, IndexDatabaseCreator.countDatabaseRows(indexFile));
	}
	
}
