package edu.mayo.bior.cli.cmd;

import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertFalse;
import static org.junit.Assert.assertTrue;
import static org.junit.Assert.fail;

import java.io.File;
import java.io.FileFilter;
import java.io.IOException;
import java.util.Properties;

import org.apache.commons.cli.ParseException;
import org.apache.commons.io.FileUtils;
import org.junit.Before;
import org.junit.Rule;
import org.junit.Test;
import org.junit.rules.TemporaryFolder;

import edu.mayo.bior.catalog.HumanBuildAssembly;
import edu.mayo.bior.catalog.stats.StatsBuilder;
import edu.mayo.bior.cli.func.BaseFunctionalTest;
import edu.mayo.bior.pipeline.createcatalog.TjsonToCatalog;
import edu.mayo.cli.test.CommandOutput;
import edu.mayo.cli.test.CommandTestDriver;

public class CatalogStatsCommandTest extends BaseFunctionalTest
{
   private CommandTestDriver driver;

   @Rule
   public TemporaryFolder tempFolder = new TemporaryFolder();

   @Before
   public void setUp() throws IOException, ParseException
   {
      driver = new CommandTestDriver(CatalogStatsCommand.class);
      tempFolder.create();
      StatsBuilder.setTestLargeRowCountsToDefault();
      StatsBuilder.setValuesToInitializeToIntMaxToDefault();
   }

   @Test
   public void happyPath() throws IOException
   {

      File catalogFile = new File("src/test/resources/catalogStats/clinvar_catalog/macarthur-lab_xml_txt.tsv.bgz");
      File outputDir = tempFolder.newFolder("happyPath");

      CommandOutput output = driver.run(new String[]{"-d", catalogFile.getAbsolutePath(), "-o", outputDir.getAbsolutePath(),
         "-n", "2000", "--numLines", /*ALL=*/"0", "--startLine", "1"});

      assertEquals("", output.stderr);
      assertEquals(0, output.exitCode);

      File expectedOutputDir = new File("src/test/resources/catalogStats/clinvar_stats");

      // should be same # of files (1 per column, excluding golden attributes)
      assertEquals(expectedOutputDir.listFiles().length, outputDir.listFiles().length);

      // check that all stats files match
      for (File expectedFile : expectedOutputDir.listFiles())
      {
         File actualFile = new File(outputDir, expectedFile.getName());
         assertTrue(actualFile.exists());
         assertTrue(FileUtils.contentEquals(expectedFile, actualFile));
      }
   }
   
   
   

   @Test
   /** Test with a small subset of the entire file, and with only a subset of the values (for the count)
    *  Also, test with a catalog that is just the JSON column
    * @throws IOException
    */
   public void testSubsetOfLinesAndValueCount_andOneColCtg() throws IOException, InterruptedException
   {
      final String EOL = "\n";
      final String TAB = "\t";
      String ctgStr = "{'A':1}" + EOL   // Sampled
         + "{'A':1}" + EOL
         + "{'A':1}" + EOL
         + "{'A':2}" + EOL   // Sampled
         + "{'A':2}" + EOL
         + "{'A':2}" + EOL
         + "{'A':2}" + EOL   // Sampled
         + "{'A':3}" + EOL
         + "{'A':3}" + EOL
         + "{'A':3}" + EOL   // Sampled
         + "{'A':44}" + EOL
         + "{'A':44}" + EOL
         + "{'A':555555}" + EOL; // Sampled
      String colsTsvStr = concat("#ColumnName", "Type", "Count", "Description", "HumanReadableName") + EOL
         + concat("A", "String", "1", "A key", "keyA") + EOL;
      File ctgFile = createCatalogFromStr(ctgStr, colsTsvStr);

      File statsOutDir = tempFolder.newFolder();
      // Sample only every 1 in 3 lines;  Only save stats on the first 4 unique values
      File logFile = tempFolder.newFile();
      CommandOutput output = driver.run(new String[]{"-d", ctgFile.getAbsolutePath(), "-o", statsOutDir.getAbsolutePath(), "-n", "4", "-s", "3", "--logfile", logFile.getCanonicalPath()});
      //String logContents = FileUtils.readFileToString(logFile);
      //printOutputs(output, logContents);
      assertEquals("", output.stderr);
      assertEquals(0, output.exitCode);

      String EXPECTED_STATS = FileUtils.readFileToString(new File("src/test/resources/catalogStats/A_stats.txt"));

      // Should only be 1 file in the stats output dir (for the "A" field)
      assertEquals(1, statsOutDir.listFiles().length);

      // check that the stats for field "A" match expected
      assertEquals(EXPECTED_STATS, FileUtils.readFileToString(new File(statsOutDir, "A_stats.txt")));
   }

   private File createCatalogFromStr(String ctgStr, String colsTsvStr) throws IOException, InterruptedException
   {
      File tempDir = tempFolder.newFolder();

      File ctgColsTsvFile = new File(tempDir, "myctg.columns.tsv");
      FileUtils.write(ctgColsTsvFile, colsTsvStr);

      File ctgStrFile = new File(tempDir, "myctg.tsv");
      FileUtils.write(ctgStrFile, ctgStr);
      File ctgBgzipOut = new File(tempDir, "myctg.tsv.bgz");

      TjsonToCatalog tjsonToCatalog = new TjsonToCatalog();
      tjsonToCatalog.createCatalog(ctgStrFile.getCanonicalPath(), ctgBgzipOut.getCanonicalPath(),
         /*isSort=*/false, /*chromSortOrderFilePath=*/null,
         /*jsoncol=*/0, /*isJsonOnly=*/true,
         tempDir.getCanonicalPath(), HumanBuildAssembly.GRCh37,
         /*isModifyChrom=*/false,
         /*isInFinalFormat=*/false);
      return ctgBgzipOut;
   }

   @Test
   public void catalogDoesNotExist() throws IOException
   {
      String garbagePath = "sdfjkdlsajfklsj";
      File outputDir = tempFolder.newFolder("catalogDoesNotExist");

      CommandOutput output = driver.run(String.format("-d %s -o %s", garbagePath, outputDir.getAbsolutePath()));

      assertEquals(1, output.exitCode);
      assertEquals("", output.stdout);
      assertTrue(output.stderr, output.stderr.contains(String.format("%s does not exist", garbagePath)));
   }

   @Test
   public void catalogNotReadable() throws IOException
   {
      // make a new copy and set readable FALSE
      File catalogDir = tempFolder.newFolder("catalogDir");
      FileUtils.copyFileToDirectory(new File("src/test/resources/catalogStats/clinvar_catalog/macarthur-lab_xml_txt.tsv.bgz"), catalogDir);
      File catalogFile = catalogDir.listFiles()[0];
      catalogFile.setReadable(false);

      File outputDir = tempFolder.newFolder("catalogNotReadable");

      CommandOutput output = driver.run(String.format("-d %s -o %s", catalogFile.getAbsolutePath(), outputDir.getAbsolutePath()));
      assertEquals(1, output.exitCode);
      assertEquals("", output.stdout);
      assertTrue(output.stderr, output.stderr.contains(String.format("%s permissions issue (not readable)", catalogFile.getAbsolutePath())));
   }

   @Test
   public void outputDirNotWriteable() throws IOException
   {
      File catalogFile = new File("src/test/resources/catalogStats/clinvar_catalog/macarthur-lab_xml_txt.tsv.bgz");
      File outputDir = tempFolder.newFolder("outputDirNotWriteable");
      outputDir.setWritable(false);

      CommandOutput output = driver.run(String.format("-d %s -o %s", catalogFile.getAbsolutePath(), outputDir.getAbsolutePath()));

      assertEquals(1, output.exitCode);
      assertEquals("", output.stdout);
      assertTrue(output.stderr.contains(outputDir.getAbsolutePath()) && output.stderr.contains("(Permission denied)"));
   }

   @Test
   public void badNumLinesOption() throws IOException
   {
      File catalogFile = new File("src/test/resources/catalogStats/clinvar_catalog/macarthur-lab_xml_txt.tsv.bgz");

      CommandOutput output = driver.run(String.format("-d %s --numLines -1", catalogFile.getAbsolutePath()));

      assertEquals(1, output.exitCode);
      assertEquals("", output.stdout);
      assertTrue(output.stderr.contains("Option 'numLines' must be 0 or higher"));
   }

   @Test
   public void badStartLineOption() throws IOException
   {
      File catalogFile = new File("src/test/resources/catalogStats/clinvar_catalog/macarthur-lab_xml_txt.tsv.bgz");

      CommandOutput output = driver.run(String.format("-d %s  --startLine 0", catalogFile.getAbsolutePath()));

      assertEquals(1, output.exitCode);
      assertEquals("", output.stdout);
      assertTrue(output.stderr.contains("Option 'startLine' must be 1 or higher"));
   }
   
   
   @Test
   public void testLargeRowCount() throws IOException
   {
	  StatsBuilder.setLogProgressEveryXLines(1);
      StatsBuilder.setTestLargeRowCounts(true);
      Properties values = new Properties();
      values.setProperty("INFO.GENEINFO", "DDX11L1:100287102");
      values.setProperty("INFO.RSPOS",    "10579");
      StatsBuilder.setValuesToInitializeToIntMax(values);

      File catalogFile = new File("src/test/resources/catalogStats/dbsnp_catalog/All_dbSNP.tsv.bgz");
      File tempDir = this.tempFolder.newFolder();
      CommandOutput output = driver.run(String.format("-d %s  --output-dir  %s", catalogFile.getAbsolutePath(), tempDir.getCanonicalPath()));

      assertEquals("", output.stderr);
      assertEquals("", output.stdout);
      assertEquals(0, output.exitCode);
      
      String geneInfoTxt = FileUtils.readFileToString(new File(tempDir, "INFO.GENEINFO_stats.txt"));
      
      // ------------------------------- Integer.MAX_VALUE = 2147483647
      assertContains(geneInfoTxt, "Total Lines in file:      2147483757");
      assertContains(geneInfoTxt, "Num lines sampled:        2147483757");
	  assertContains(geneInfoTxt, "Lines that had column:    2147483757 (100.0%)");
	  assertContains(geneInfoTxt, "Total column value chars: 2147485757");
	  
	  // Note: the '2' character should be over Integer.MAX_VALUE since this is explicitly set in the initialization for the test
	  //       'D' should have 200 character occurrences (CharCount) on 100 lines (LineCount) since it occurs twice in the "INFO.GENEINFO" value "DDX11L1:100287102"
	  //       'L' should have 100 character occurrences (CharCount) on 100 lines (LineCount) since it only occurs once in the "INFO.GENEINFO" value "DDX11L1:100287102"
	  //                           Char   LineCount           Line%       CharCount           Char%
	  assertContains(geneInfoTxt, "2      2147483757          100.0%      2147483857          100.0%");
	  assertContains(geneInfoTxt, "D             100            0.0%             200            0.0%");
	  assertContains(geneInfoTxt, "L             100            0.0%             100            0.0%");
   }
   
   
   @Test
   /** Not specifying an output directory should write the files to the current working directory and NOT the root dir */
   public void testOutputDirShouldNotBeRootDir() throws IOException
   {
      File catalogFile = new File("src/test/resources/catalogStats/clinvar_catalog/macarthur-lab_xml_txt.tsv.bgz");
      CommandOutput output = driver.run(String.format("-d %s ", catalogFile.getAbsolutePath()));

      printOutputs(output, "");
      
      // Should NOT contain "Permission denied"
      assertFalse(output.stderr.contains("(Permission denied)"));
      assertEquals("", output.stdout);
      assertEquals(0, output.exitCode);
      
      // Verify that the output stats were written to the current working directory
      String currentWorkingDir = System.getProperty("user.dir");
      File statsFile = new File(currentWorkingDir, "symbol_stats.txt");
      assertTrue(statsFile.exists());
      assertTrue(statsFile.length() > 1000);
      removeAllStatsFilesFromDir(new File(currentWorkingDir));
   }

	private void removeAllStatsFilesFromDir(File dir) {
		File[] statsFiles = getStatsFileInDir(dir);
		for(File f : statsFiles) {
			f.deleteOnExit();
		}
	}
	
	private File[] getStatsFileInDir(File dir) {
		return dir.listFiles(new FileFilter() {
			public boolean accept(File f) {
				return f.getName().endsWith("_stats.txt");
			}
		});		
	}

	private void printOutputs(CommandOutput output, String logText) {
    	System.err.println("exitCode: " + output.exitCode);

    	System.out.println("stdout:------------------------------------");
    	System.out.println(output.stdout);
    	
    	System.out.println("stderr:------------------------------------");
    	System.out.println(output.stderr);
    	
    	System.out.println("logText output:-----------------------------");
    	System.out.println(logText);
	}



}
