package edu.mayo.bior.cli.cmd;

import com.google.gson.Gson;
import edu.mayo.bior.catalog.CatalogFormatException;
import edu.mayo.bior.catalog.latest.LatestCatalog;
import edu.mayo.bior.catalog.latest.LatestCatalogFinder;
import edu.mayo.bior.catalog.latest.LatestCatalogNotFoundException;
import edu.mayo.cli.test.CommandOutput;
import edu.mayo.cli.test.CommandTestDriver;
import org.junit.Before;
import org.junit.Rule;
import org.junit.Test;
import org.junit.rules.TemporaryFolder;

import java.io.File;
import java.io.IOException;

import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertTrue;
import static org.mockito.Mockito.*;
import static org.mockito.Mockito.times;
import static org.unitils.reflectionassert.ReflectionAssert.assertReflectionEquals;

public class CatalogLatestCommandTest {

    @Rule
    public TemporaryFolder tempFolder = new TemporaryFolder();

    private CommandTestDriver driver;

    private LatestCatalogFinder mockFinder;

    private File rootDir;

    private LatestCatalog latest;

    @Before
    public void setUp() throws Exception {
        driver = new CommandTestDriver(CatalogLatestCommand.class);

        rootDir = tempFolder.newFolder("catalog_root_dir");

        mockFinder = mock(LatestCatalogFinder.class);
        CatalogLatestCommand.setLatestCatalogFinder(mockFinder);

        latest = new LatestCatalog();
        latest.setSource("FOO");
        latest.setVersion("v3");
        latest.setBuild("GRCh37");
        latest.setDescription("desc");
        latest.setDataset("dataset");
        latest.setDataSourceReleaseDate("2019-01-11");
        latest.setCatalogPath("/path/to/catalog");
    }

    @Test
    public void happyPath() throws IOException, CatalogFormatException, LatestCatalogNotFoundException {

        when(mockFinder.findLatestCatalog(rootDir, "FOO", "GRCh37", null)).thenReturn(latest);

        CommandOutput output = driver.run(new String[]{"-r", rootDir.getAbsolutePath(), "-s", "FOO", "-B", "GRCh37"});

        assertEquals("", output.stderr);
        assertEquals(0, output.exitCode);

        LatestCatalog actual = new Gson().fromJson(output.stdout, LatestCatalog.class);

        assertReflectionEquals(latest, actual);

        verify(mockFinder, times(1)).findLatestCatalog(rootDir, "FOO", "GRCh37", null);
    }

    @Test
    public void extractSingleField() throws IOException, CatalogFormatException, LatestCatalogNotFoundException {

        when(mockFinder.findLatestCatalog(rootDir, "FOO", "GRCh37", null)).thenReturn(latest);

        CommandOutput output = driver.run(new String[]{"-r", rootDir.getAbsolutePath(), "-s", "FOO", "-B", "GRCh37", "-p", "version"});

        assertEquals("", output.stderr);
        assertEquals(0, output.exitCode);

        verify(mockFinder, times(1)).findLatestCatalog(rootDir, "FOO", "GRCh37", null);

        assertEquals("v3", output.stdout.trim());
    }

    @Test
    public void extractMultipleFields() throws IOException, CatalogFormatException, LatestCatalogNotFoundException {

        when(mockFinder.findLatestCatalog(rootDir, "FOO", "GRCh37", null)).thenReturn(latest);

        CommandOutput output = driver.run(new String[]{"-r", rootDir.getAbsolutePath(), "-s", "FOO", "-B", "GRCh37", "-p", "catalogPath", "-p", "version", "-p", "dataSourceReleaseDate"});

        assertEquals("", output.stderr);
        assertEquals(0, output.exitCode);

        verify(mockFinder, times(1)).findLatestCatalog(rootDir, "FOO", "GRCh37", null);

        assertEquals("/path/to/catalog\tv3\t2019-01-11", output.stdout.trim());
    }

    @Test
    public void badJsonPath() throws IOException, CatalogFormatException, LatestCatalogNotFoundException {

        when(mockFinder.findLatestCatalog(rootDir, "FOO", "GRCh37", null)).thenReturn(latest);

        CommandOutput output = driver.run(new String[]{"-r", rootDir.getAbsolutePath(), "-s", "FOO", "-B", "GRCh37", "-p", "BAD_PATH"});

        verify(mockFinder, times(1)).findLatestCatalog(rootDir, "FOO", "GRCh37", null);

        assertEquals(1, output.exitCode);
        assertEquals("", output.stdout);
        assertTrue(output.stderr.contains("No results for path: $['BAD_PATH']"));
    }

    @Test
    public void happyPathWithOptionalContains() throws IOException, CatalogFormatException, LatestCatalogNotFoundException {

        when(mockFinder.findLatestCatalog(rootDir, "FOO", "GRCh37", "disease_no_spaces")).thenReturn(latest);

        CommandOutput output = driver.run(new String[]{"-r", rootDir.getAbsolutePath(), "-s", "FOO", "-B", "GRCh37", "-c", "disease_no_spaces"});

        assertEquals("", output.stderr);
        assertEquals(0, output.exitCode);

        LatestCatalog actual = new Gson().fromJson(output.stdout, LatestCatalog.class);

        assertReflectionEquals(latest, actual);

        verify(mockFinder, times(1)).findLatestCatalog(rootDir, "FOO", "GRCh37", "disease_no_spaces");
    }

    @Test
    public void errorFindingCatalog() throws IOException, CatalogFormatException, LatestCatalogNotFoundException {

        LatestCatalogNotFoundException exception = new LatestCatalogNotFoundException("Test message...");

        when(mockFinder.findLatestCatalog(rootDir, "FOO", "GRCh37", null)).thenThrow(exception);

        CommandOutput output = driver.run(new String[]{"-r", rootDir.getAbsolutePath(), "-s", "FOO", "-B", "GRCh37"});

        assertEquals(1, output.exitCode);
        assertEquals("", output.stdout);
        assertEquals("Error in bior_catalog_latest\n" + "Test message...", output.stderr.trim());

        verify(mockFinder, times(1)).findLatestCatalog(rootDir, "FOO", "GRCh37", null);
    }
}
