package edu.mayo.bior.catalog.verification;

import java.io.BufferedReader;
import java.io.File;
import java.io.IOException;
import java.util.Map.Entry;

import com.google.gson.JsonParseException;
import edu.mayo.bior.catalog.CatalogFormatException;
import edu.mayo.bior.catalog.CatalogTabixEntry;
import edu.mayo.bior.catalog.CatalogFileUtils;
import htsjdk.tribble.readers.TabixReader;
import org.junit.Assert;
import org.junit.Before;
import org.junit.Rule;
import org.junit.Test;

import com.google.gson.JsonElement;
import com.google.gson.JsonParser;
import com.google.gson.JsonPrimitive;
import org.junit.rules.ExpectedException;
import org.junit.rules.TemporaryFolder;

import static org.junit.Assert.*;

public class VerifyUtilsTest
{

   private String junit_DataDir = "src/test/resources/testData/verification/";

   @Rule
   public TemporaryFolder temporaryFolder = new TemporaryFolder();

   @Rule
   public ExpectedException expectedException = ExpectedException.none();

   private File tempFile = null;
   private MessageLogger mLogger = null;

   @Before
   public void setUp() throws IOException
   {
      tempFile = temporaryFolder.newFile();
      mLogger = new MessageLogger(tempFile.getPath());
   }

   @Test
   public void testReadCatalogRow()
   {
      String validCatRow1 = "." + TAB + "0" + TAB + "0" + TAB + "{\"key\":\"value\"}";
      String validCatRow2 = "1" + TAB + "123" + TAB + "456" + TAB + "{\"key\":\"value\"}";
      String validCatRow3 = "{\"key\":\"value\"}";

      Assert.assertNotNull(VerifyUtils.readCatalogRow(validCatRow1, mLogger));
      Assert.assertNotNull(VerifyUtils.readCatalogRow(validCatRow2, mLogger));
      Assert.assertNotNull(VerifyUtils.readCatalogRow(validCatRow3, mLogger));

      String invalidCatRow1 = TAB + TAB + TAB + "{\"key\":\"value\"}";
      String invalidCatRow2 = "." + TAB + "." + TAB + "." + TAB + "{\"key\":\"value\"}";
      String invalidCatRow3 = "1" + TAB + "123" + TAB + "456" + TAB;
      Assert.assertNull(VerifyUtils.readCatalogRow(invalidCatRow1, mLogger));
      Assert.assertNull(VerifyUtils.readCatalogRow(invalidCatRow2, mLogger));
      Assert.assertNull(VerifyUtils.readCatalogRow(invalidCatRow3, mLogger));
   }

   // TODO - fix this by testing validateRowTabix, verifyRowGoldenJson, verifyTabixToGoldenElements,
   @Test
   public void testValidateCatalogRowTabix()
   {

      try
      {
         CatalogVerifier verifier = new CatalogVerifier();
         MessageLogger logger = mLogger;

         // SETUP: NONVARIANT, NONPOSITIONAL TESTS:

         /////////////////////////////////////////////////////////
         // Setup test #1 for non-variant, non-positional:
         /////////////////////////////////////////////////////////
         BufferedReader r = CatalogFileUtils.getBufferedReader(junit_DataDir + "hgncCatalogEntries.txt");
         String hgnc_CatEntry1 = r.readLine();
         CatalogTabixEntry validNonPositionalTabixEntry = VerifyUtils.readCatalogRow(hgnc_CatEntry1, logger);
         Assert.assertNotNull(validNonPositionalTabixEntry);
         assertEquals(0, logger.numErrors());
         assertEquals(0, logger.numWarnings());
         assertEquals(0, logger.numInfos());

         //Run testing #1 for non-variant, non-positional:
         assertEquals(0, logger.numErrors());
         //verifier.validateRowTabix(validNonPositionalTabixEntry,isCatalogPositional);
         assertEquals(0, logger.numErrors());
         assertEquals(0, logger.numWarnings());
         assertEquals(0, logger.numInfos());

         com.google.gson.JsonObject catalogRowJsonObj1 = VerifyUtils.getJsonObject(validNonPositionalTabixEntry.getJsonString());
         //CatalogEntryGoldenJson goldenJsonObj = verifier.verifyRowGoldenJson(catalogRowJsonObj1,isCatalogPositional);
         //Assert.assertNotNull(goldenJsonObj);
         //Assert.assertNull(goldenJsonObj.getChr());
         //Assert.assertNull(goldenJsonObj.getMinBP());
         //Assert.assertNull(goldenJsonObj.getMaxBP());
         assertEquals(0, logger.numErrors());
         assertEquals(0, logger.numWarnings());
         assertEquals(0, logger.numInfos());

         //verifier.verifyTabixToGoldenElements(validNonPositionalTabixEntry,goldenJsonObj,isCatalogPositional);
         assertEquals(0, logger.numErrors());
         assertEquals(0, logger.numWarnings());
         assertEquals(0, logger.numInfos());

         /////////////////////////////////////////////////////////
         // Setup test #2 for non-variant, non-positional:
         /////////////////////////////////////////////////////////
         r = CatalogFileUtils.getBufferedReader(junit_DataDir + "jsonOnlyCatalogEntries.txt");
         String jsonOnly_CatEntry1 = r.readLine();
         CatalogTabixEntry jsonOnly_tabixEntry = VerifyUtils.readCatalogRow(jsonOnly_CatEntry1, logger);
         assertEquals(0, logger.numErrors());
         assertEquals(0, logger.numWarnings());
         assertEquals(0, logger.numInfos());

         //Run test #2 for non-variant, non-positional:
         assertEquals(0, logger.numErrors());
         //verifier.validateRowTabix(jsonOnly_tabixEntry,isCatalogPositional);
         assertEquals(0, logger.numErrors());
         assertEquals(0, logger.numWarnings());
         assertEquals(0, logger.numInfos());

         com.google.gson.JsonObject catalogRowJsonObj2 = VerifyUtils.getJsonObject(jsonOnly_tabixEntry.getJsonString());
         //CatalogEntryGoldenJson goldenVariant2 = verifier.verifyRowGoldenJson(catalogRowJsonObj2,isCatalogPositional);
         //Assert.assertNotNull(goldenVariant2);
         //Assert.assertNull(goldenVariant2.getChr());
         //Assert.assertNull(goldenVariant2.getMinBP());
         //Assert.assertNull(goldenVariant2.getMaxBP());
         assertEquals(0, logger.numErrors());
         assertEquals(0, logger.numWarnings());
         assertEquals(0, logger.numInfos());

         //verifier.verifyTabixToGoldenElements(jsonOnly_tabixEntry,goldenVariant2,isCatalogPositional);
         assertEquals(0, logger.numErrors());
         assertEquals(0, logger.numWarnings());
         assertEquals(0, logger.numInfos());
      }
      catch (Exception e)
      {
         fail("Unexpected Exception... failure: " + e.getMessage());
         e.printStackTrace();
      }

   }

   @Test
   public void testTabixRetrieval() throws IOException, CatalogFormatException
   {
      File catalogFile = new File("src/test/resources/testData/verification/catalog_with_column_issues/catalog.tsv.bgz");
      TabixReader tabixReader = new TabixReader(catalogFile.getPath());
      BufferedReader reader = CatalogFileUtils.getBufferedReader(catalogFile.getPath());
      if (reader == null)
      {
         fail("Couldn't get reader to read " + catalogFile.getPath());
         return;
      }
      String line;
      int lineNumber = 1;
      while ((line = reader.readLine()) != null)
      {
         CatalogTabixEntry entry = new CatalogTabixEntry(line);
         if (lineNumber == 1)
         {
            assertTrue(VerifyUtils.isTabixRetrievalSuccessful(tabixReader, entry, true));
         }
         if (lineNumber == 3)
         {
            assertTrue(VerifyUtils.isTabixRetrievalSuccessful(tabixReader, entry, false));
         }
         lineNumber++;
      }
   }

   @Test
   public void testDNA()
   {
      final String ERR_MSG = "Sequence valid allele test failed. Please investigate.";


      assertTrue(ERR_MSG, CatalogVariantVerifier.validDNANucleotides("A"));
      assertTrue(ERR_MSG, CatalogVariantVerifier.validDNANucleotides("C"));
      assertTrue(ERR_MSG, CatalogVariantVerifier.validDNANucleotides("G"));
      assertTrue(ERR_MSG, CatalogVariantVerifier.validDNANucleotides("T"));

      assertTrue(ERR_MSG, CatalogVariantVerifier.validDNANucleotides("AC"));
      assertTrue(ERR_MSG, CatalogVariantVerifier.validDNANucleotides("AA"));
      assertTrue(ERR_MSG, CatalogVariantVerifier.validDNANucleotides("GT"));
      assertTrue(ERR_MSG, CatalogVariantVerifier.validDNANucleotides("CC"));
      assertTrue(ERR_MSG, CatalogVariantVerifier.validDNANucleotides("GT"));

      assertTrue(ERR_MSG, CatalogVariantVerifier.validDNANucleotides("AAAAA"));
      assertTrue(ERR_MSG, CatalogVariantVerifier.validDNANucleotides("ACCCCT"));
      assertTrue(ERR_MSG, CatalogVariantVerifier.validDNANucleotides("GGGGGG"));
      assertTrue(ERR_MSG, CatalogVariantVerifier.validDNANucleotides("GTGTGT"));
      assertTrue(ERR_MSG, CatalogVariantVerifier.validDNANucleotides("TTTTATTTTT"));
      assertTrue(ERR_MSG, CatalogVariantVerifier.validDNANucleotides("TTTTT"));
      assertTrue(ERR_MSG, CatalogVariantVerifier.validDNANucleotides("CATTGCCCCCCAAAAATGAG"));

      assertTrue(ERR_MSG, CatalogVariantVerifier.validDNANucleotides("N"));
      assertTrue(ERR_MSG, CatalogVariantVerifier.validDNANucleotides("NANAAACTC"));
      assertTrue(ERR_MSG, CatalogVariantVerifier.validDNANucleotides("NAAGATCC"));
      assertTrue(ERR_MSG, CatalogVariantVerifier.validDNANucleotides("NNN"));

      String empty_not_allowed = "Empty allele not allowed at this time.";
      String lower_not_allowed = "Lower case alleles in sequence not valid.";
      String invalid_allele_char = "Invalid allele character in sequence.";
      assertFalse(ERR_MSG + empty_not_allowed, CatalogVariantVerifier.validDNANucleotides(" "));
      assertFalse(ERR_MSG + empty_not_allowed, CatalogVariantVerifier.validDNANucleotides(""));
      assertFalse(ERR_MSG + empty_not_allowed, CatalogVariantVerifier.validDNANucleotides(".")); // vcf empty field value
      assertFalse(ERR_MSG + lower_not_allowed, CatalogVariantVerifier.validDNANucleotides("n"));
      assertFalse(ERR_MSG + lower_not_allowed, CatalogVariantVerifier.validDNANucleotides("acgt"));
      assertFalse(ERR_MSG + lower_not_allowed, CatalogVariantVerifier.validDNANucleotides("aCgGtT"));
      assertFalse(ERR_MSG + lower_not_allowed, CatalogVariantVerifier.validDNANucleotides("AAAAa"));
      assertFalse(ERR_MSG + lower_not_allowed, CatalogVariantVerifier.validDNANucleotides("ANTCn"));
      assertFalse(ERR_MSG + invalid_allele_char, CatalogVariantVerifier.validDNANucleotides("TCrGGT"));
      assertFalse(ERR_MSG + invalid_allele_char, CatalogVariantVerifier.validDNANucleotides("HPWQMY"));

   }

   @Test
   public void testBadJson() throws JsonParseException
   {
      expectedException.expect(JsonParseException.class);
      VerifyUtils.getJsonObject("{12}");
   }

   @Test
   public void testInvalidJson_KeyWithPeriodWithin()
   {
      String catalogEntry_testInvalidJsonKey = "1\t10108\t10108\t{\"CHR.OM\":\"1\",\"POS\":\"10108\",\"ID\":\"rs62651026\",\"REF\":\"C\",\"ALT\":\"T\",\"QUAL\":\".\",\"FILTER\":\".\",\"INFO\":{\"RS\":62651026,\"RSPOS\":10108,\"dbSNPBuildID\":129,\"SSR\":0,\"SAO\":0,\"VP\":\"0x050000020005000002000100\",\"WGT\":1,\"VC\":\"SNV\",\"R5\":true,\"ASP\":true},\"_id\":\"rs62651026\",\"_type\":\"variant\",\"_landmark\":\"1\",\"_refAllele\":\"C\",\"_altAlleles\":[\"T\"],\"_minBP\":10108,\"_maxBP\":10108}";

      String[] elems = catalogEntry_testInvalidJsonKey.split("\t");
      String jsonToTest = elems[3];

      com.google.gson.JsonParser jp = new JsonParser();
      com.google.gson.JsonElement catalogRowJsonElem = null;
      com.google.gson.JsonObject catalogRowJsonObj = null;
      try
      {
         catalogRowJsonElem = jp.parse(jsonToTest);
         catalogRowJsonObj = catalogRowJsonElem.getAsJsonObject();
         for (Entry<String, com.google.gson.JsonElement> entry : catalogRowJsonObj.entrySet())
         {
            String key = entry.getKey();
            if (key.contains("."))
            {
               com.google.gson.JsonElement value = entry.getValue();
               if (value != null)
               {
                  System.out.println("Was able to parse json and get key-value pair for key with dot: Key: " + key + " value: " + value.getAsString());   // "CHR.OM"   is a valid Json key, parsable anyway. but we want to not have this happen in catalogs.
               }
            }
         }
      }
      catch (Exception e)
      {
         System.err.println("Exception testing json.");
      }

   }
	
	@Test
	public void testPathErrorHandling() 
	{
		// See: http://stackoverflow.com/questions/1099300/whats-the-difference-between-getpath-getabsolutepath-and-getcanonicalpath
		try 
		{
			File f = new File("/SomeBogusDir/SomeBadSubDir/../AnotherDir/someFile.txt");
			assertEquals("/SomeBogusDir/SomeBadSubDir/../AnotherDir/someFile.txt", f.getPath());
			assertEquals("/SomeBogusDir/SomeBadSubDir/../AnotherDir/someFile.txt", f.getAbsolutePath());
			// Hmm, this one no longer seems to throw an exception when the file is not found.   Did this change with Java 8?
			assertEquals("/SomeBogusDir/AnotherDir/someFile.txt", f.getCanonicalPath());
		}
		catch(Exception e) 
		{
			fail("Could not resolve one of the paths.  ");
		}
	}
	

   @Test
   public void testNumberDeserialization()
   {
      String maxJavaInt = "2147483647";  // 2 to the 31st power, minus 1 is the max int
      //String minJavaInt = "-2147483647";
      int maxint = Integer.MAX_VALUE;
      assertTrue(maxint == new Integer(maxJavaInt));  // sanity check before we start

      String largerNumThanMaxInt = "3147483647";
      Long longVal_largerNumThanMaxInt = new Long(largerNumThanMaxInt);
      //String smallerNumThanInt = "-3147483647";

      String jsonNumbers = "{\"my_int\":" + maxJavaInt + ",\"my_long\":" + largerNumThanMaxInt + "}";

      com.google.gson.JsonParser jp = new JsonParser();
      com.google.gson.JsonElement jsonElem = null;
      jsonElem = jp.parse(jsonNumbers);

      if (jsonElem.isJsonObject())
      {
         com.google.gson.JsonObject obj = jsonElem.getAsJsonObject();
         JsonElement myInt = obj.get("my_int");
         JsonElement myLong = obj.get("my_long");
         JsonPrimitive jsonPrim_validInt = myInt.getAsJsonPrimitive();
         JsonPrimitive jsonPrim_validLong = myLong.getAsJsonPrimitive();

         assertTrue(jsonPrim_validInt.isNumber());
         java.lang.Object int_number = VerifyUtils.jsonNumberToJavaNumber(jsonPrim_validInt);
         assertTrue(int_number instanceof Integer);
         try
         {
            Integer intNum = (Integer) int_number;
            assertTrue(intNum.equals(maxint));
         }
         catch (NumberFormatException nf)
         {
            fail("Integer handling: We should not get an exception with this operation");
         }

         assertTrue(jsonPrim_validLong.isNumber());
         java.lang.Object lnumber = VerifyUtils.jsonNumberToJavaNumber(jsonPrim_validLong);
         assertTrue(lnumber instanceof Long);
         try
         {
            Long longNum = (Long) lnumber;
            assertTrue(longNum.equals(longVal_largerNumThanMaxInt));
         }
         catch (NumberFormatException nf)
         {
            fail("Long handling: We should not get an exception with this operation");
         }

      } // end number json object
   }

   @Test
   public void testIsChrM()
   {
      assertFalse(VerifyUtils.isChrM(null));
      assertFalse(VerifyUtils.isChrM(""));
      assertFalse(VerifyUtils.isChrM("  "));
      assertFalse(VerifyUtils.isChrM("1"));
      assertFalse(VerifyUtils.isChrM("chrX"));
      assertTrue(VerifyUtils.isChrM(" MT "));
      assertTrue(VerifyUtils.isChrM(" ChRMT "));
      assertTrue(VerifyUtils.isChrM(" M "));
      assertTrue(VerifyUtils.isChrM(" MT "));
      assertTrue(VerifyUtils.isChrM("M"));
      assertTrue(VerifyUtils.isChrM("MT"));
      assertTrue(VerifyUtils.isChrM("chrm"));
      assertTrue(VerifyUtils.isChrM("chrmt"));
   }
   
   private final String TAB = "\t";

}
