package edu.mayo.bior.catalog.verification;

import edu.mayo.pipes.history.ColumnMetaData;
import edu.mayo.pipes.history.ColumnMetaDataOperations;
import org.junit.Before;
import org.junit.Rule;
import org.junit.Test;
import org.junit.rules.TemporaryFolder;

import java.io.*;
import java.util.HashMap;
import java.util.Set;

import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertTrue;

/**
 * Test CatalogJsonVerifier
 */
public class CatalogJsonVerifierTest
{
   private HashMap<String, ColumnMetaData> columnInfo;
   private MessageLogger stringLogger = new MessageLogger(new StringWriter());

   @Rule
   public TemporaryFolder tempFolder = new TemporaryFolder();

   @Before
   public void setUp() throws IOException
   {
	   File dbSnpColumnsFile = new File("src/test/resources/testData/verification/00-All.vcf.columns.tsv");
       columnInfo = new ColumnMetaDataOperations(dbSnpColumnsFile).load();
   }

   @Test
   public void testExtraJsonKeys()
   {
      // Parsable json, but embedded INFO JSON's key 'EXTRAPROP' is an extra prop-value pair compared to column info:
      String json = "{'CHROM':'1','POS':'10108','ID':'rs62651026','INFO':{'RS':62651026,'RSPOS':10108,'EXTRAPROP':666},'_refAllele':'C','_altAlleles':['T'],'TESTJSON':{'HELLO':'JAJ'},'LASTPROP':111}";

      CatalogJsonVerifier verifier = new CatalogJsonVerifier(columnInfo, stringLogger);
      verifier.verify(sqToDq(json));
      Set<String> propNamesNotInColumnsInfo = verifier.keysNotFoundInColumnInfo();
      assertTrue(propNamesNotInColumnsInfo.size() == 3);
      assertTrue(propNamesNotInColumnsInfo.contains("INFO.EXTRAPROP"));
      assertTrue(propNamesNotInColumnsInfo.contains("TESTJSON.HELLO"));
      assertTrue(propNamesNotInColumnsInfo.contains("LASTPROP"));
      assertEquals(2, stringLogger.numInfos());
      assertEquals(3, stringLogger.numWarnings());
      assertEquals(2, stringLogger.numErrors());
      assertEquals(7, stringLogger.numMessages());

      // Parsable json, but JAJ is an extra prop-value pair compared to column info:
      json = "{'CHROM':'1','POS':'10108','ID':'rs62651026','JAJ':'23d4567'}";
      stringLogger = new MessageLogger(new StringWriter());
      verifier = new CatalogJsonVerifier(columnInfo, stringLogger);
      verifier.verify(sqToDq(json));
      propNamesNotInColumnsInfo = verifier.keysNotFoundInColumnInfo();
      assertTrue(propNamesNotInColumnsInfo.size() == 1);
      assertTrue(propNamesNotInColumnsInfo.contains("JAJ"));
      assertEquals(2, stringLogger.numInfos());
      assertEquals(1, stringLogger.numWarnings());
      assertEquals(2, stringLogger.numErrors());
      assertEquals(5, stringLogger.numMessages());

      // Parsable json, but JAJ_ANOTHER_EXTRA is an extra prop-value pair compared to column info:
      json = "{'CHROM':'1','POS':'10108','ID':'rs62651026','JAJ_ANOTHER_EXTRA':'234567'}";
      stringLogger = new MessageLogger(new StringWriter());
      verifier = new CatalogJsonVerifier(columnInfo, stringLogger);
      verifier.verify(sqToDq(json));
      propNamesNotInColumnsInfo = verifier.keysNotFoundInColumnInfo();
      assertTrue(propNamesNotInColumnsInfo.size() == 1);
      assertTrue(propNamesNotInColumnsInfo.contains("JAJ_ANOTHER_EXTRA"));
   }

   @Test
   public void testFloatForJsonKey() throws Exception
   {
      File columnDescriptionsFile = new File("src/test/resources/testData/verification/onefloat.columns.tsv");
      HashMap<String, ColumnMetaData> oneFloatColumnInfo = new ColumnMetaDataOperations(columnDescriptionsFile).load();

      stringLogger = new MessageLogger(new StringWriter());
      String json = "{'f':1.25}";
      CatalogJsonVerifier verifier = new CatalogJsonVerifier(oneFloatColumnInfo, stringLogger);
      verifier.verify(sqToDq(json));
      assertEquals(0, stringLogger.numWarnings());
      assertEquals(0, stringLogger.numErrors());
   }

   @Test
   public void testValidQuotingAndValues()
   {
      // invalid json b/c the value for JAJ is alpha-numeric but it isn't quoted.
      StringWriter strWriter = new StringWriter();
      stringLogger = new MessageLogger(strWriter);
      String json = "{'CHROM':'1','POS':23d4567,'ID':'rs62651026'}";
      CatalogJsonVerifier verifier = new CatalogJsonVerifier(columnInfo, stringLogger);
      verifier.verify(sqToDq(json));
      assertEquals(3, stringLogger.numErrors());
      assertEquals(1, stringLogger.numInfos());
      assertEquals(0, stringLogger.numWarnings());
      assertEquals(4, stringLogger.numMessages());
      assertTrue(strWriter.toString().contains("issues parsing this json"));
      assertTrue(strWriter.toString().contains("parsing json property"));
      assertTrue(strWriter.toString().contains("look for unquoted"));

      // valid json b/c eventhough value is a number, it is quoted so it is treated as a number.
      strWriter = new StringWriter();
      stringLogger = new MessageLogger(strWriter);
      verifier = new CatalogJsonVerifier(columnInfo, stringLogger);
      json = "{'CHROM':'1','POS':234567,'ID':'rs62651026'}";
      verifier.verify(sqToDq(json));
      assertEquals(0, stringLogger.numMessages());

      strWriter = new StringWriter();
      stringLogger = new MessageLogger(strWriter);
      verifier = new CatalogJsonVerifier(columnInfo, stringLogger);
      json = "{CHROM:'1','POS':234567,'ID':'rs62651026'}";
      verifier.verify(sqToDq(json));
      assertEquals(2, stringLogger.numErrors());
      assertTrue(strWriter.toString().contains("Problem parsing json"));
      assertTrue(strWriter.toString().contains("look for unquoted"));
   }

   @Test
   public void testNoBraceJson()
   {
      String json = "";
      CatalogJsonVerifier verifier = new CatalogJsonVerifier(columnInfo, stringLogger);
      verifier.verify(json);
      assertEquals(1, stringLogger.numMessages());
      assertEquals(1, stringLogger.numErrors());
   }

   @Test
   public void testJustBracesJson()
   {
      String json = "{}";
      CatalogJsonVerifier verifier = new CatalogJsonVerifier(columnInfo, stringLogger);
      verifier.verify(json);
      assertEquals(0, stringLogger.numMessages());
   }

   @Test
   public void testSingleQuoteJson()
   {
      String json = "{'CHROM':'1','POS':234567,'ID':'rs62651026'}";
      StringWriter strWriter = new StringWriter();
      stringLogger = new MessageLogger(strWriter);
      CatalogJsonVerifier verifier = new CatalogJsonVerifier(columnInfo, stringLogger);
      verifier.verify(json);
      assertEquals(1, stringLogger.numInfos());
      assertEquals(2, stringLogger.numErrors());
      assertEquals(3, stringLogger.numMessages());
   }

   @Test
   public void testNoColumnInfo()
   {
      String json = "{'CHROM':'1','POS':234567,'ID':'rs62651026'}";
      StringWriter strWriter = new StringWriter();
      stringLogger = new MessageLogger(strWriter);
      CatalogJsonVerifier verifier = new CatalogJsonVerifier(null, stringLogger);
      verifier.verify(sqToDq(json));
      System.out.println(strWriter);
      assertEquals(1, stringLogger.numErrors());
      assertEquals(1, stringLogger.numMessages());
      assertTrue(strWriter.toString().contains("Unable to check json has valid keys"));
   }

   @Test
   public void testKeyWithDot()
   {
      String json = "{'CHROM':'1','POS':234567,'id.test':'rs62651026'}";
      StringWriter strWriter = new StringWriter();
      stringLogger = new MessageLogger(strWriter);
      CatalogJsonVerifier verifier = new CatalogJsonVerifier(columnInfo, stringLogger);
      verifier.verify(sqToDq(json));
      // 3 errors, 0 warnings, 1 info
      assertEquals(3, stringLogger.numErrors());
      assertEquals(1, stringLogger.numWarnings());
      assertEquals(2, stringLogger.numInfos());
      assertEquals(6, stringLogger.numMessages());

      json = "{'CHROM':'1','POS':234567,'json.test':{'key.withdot':'val'}}";
      strWriter = new StringWriter();
      stringLogger = new MessageLogger(strWriter);
      verifier = new CatalogJsonVerifier(columnInfo, stringLogger);
      verifier.verify(sqToDq(json));
      // 5 errors, 0 warnings, 1 info
      assertEquals(5, stringLogger.numErrors());
      assertEquals(1, stringLogger.numWarnings());
      assertEquals(2, stringLogger.numInfos());
      assertEquals(8, stringLogger.numMessages());
      assertTrue(strWriter.toString().contains("Not allowed to have '.' in json key 'key.withdot'"));
      assertTrue(strWriter.toString().contains("Not allowed to have '.' in json key 'json.test'"));
      assertTrue(strWriter.toString().contains("Keys in json not found in columns.tsv: 'json.test.key.withdot'"));
   }

   @Test
   public void testEmptyJsonValue()
   {
      String json = "{'CHROM':'1','POS':,'ID':'rs62651026'}";
      StringWriter strWriter = new StringWriter();
      stringLogger = new MessageLogger(strWriter);
      CatalogJsonVerifier verifier = new CatalogJsonVerifier(columnInfo, stringLogger);
      verifier.verify(sqToDq(json));
      assertEquals(1, stringLogger.numErrors());
      assertEquals(1, stringLogger.numMessages());
   }

   @Test
   public void testBlankJsonStringValue()
   {
      String json = "{'CHROM':'1','POS':234567,'ID':''}";
      stringLogger = new MessageLogger(new StringWriter());
      CatalogJsonVerifier verifier = new CatalogJsonVerifier(columnInfo, stringLogger);
      verifier.verify(sqToDq(json));
      assertEquals(0, stringLogger.numMessages());
   }

   @Test
   public void testEmptyArrayOk() throws IOException
   {
      String json = "{'CHROM':'1','POS':234567,'ID':[]}";
      stringLogger = new MessageLogger(new StringWriter());
      CatalogJsonVerifier verifier = new CatalogJsonVerifier(columnInfo, stringLogger);
      verifier.verify(sqToDq(json));
      assertEquals(0, stringLogger.numMessages());
   }

   @Test
   public void testIntegerValueForFloatColumn() throws IOException
   {
      File columnDescriptionsFile = new File("src/test/resources/testData/verification/onefloat.columns.tsv");
      HashMap<String, ColumnMetaData> oneFloatColumnInfo = new ColumnMetaDataOperations(columnDescriptionsFile).load();

      String json = "{'f':0}";
      StringWriter strWriter = new StringWriter();
      stringLogger = new MessageLogger(strWriter);
      CatalogJsonVerifier verifier = new CatalogJsonVerifier(oneFloatColumnInfo, stringLogger);
      verifier.verify(sqToDq(json));
      assertEquals(0, stringLogger.numMessages());
   }

   @Test
   public void testJsonArrayValueForIntegerAColumn() throws IOException
   {
      File columnDescriptionsFile = new File("src/test/resources/testData/verification/oneintwithcountA.columns.tsv");
      HashMap<String, ColumnMetaData> oneIntColumnInfo = new ColumnMetaDataOperations(columnDescriptionsFile).load();

      String json = "{'i':[0]}";
      StringWriter strWriter = new StringWriter();
      stringLogger = new MessageLogger(strWriter);
      CatalogJsonVerifier verifier = new CatalogJsonVerifier(oneIntColumnInfo, stringLogger);
      verifier.verify(sqToDq(json));
      assertEquals(0, stringLogger.numMessages());
   }

   @Test
   public void testJsonEMinusNotationValueInFloatAndFloatArray() throws IOException
   {
      File columnDescriptionsFile = new File("src/test/resources/testData/verification/twofloat.columns.tsv");
      HashMap<String, ColumnMetaData> oneIntColumnInfo = new ColumnMetaDataOperations(columnDescriptionsFile).load();

      String json = "{'f':0E-17,'fa':[1e-20, 0.6]}";
      StringWriter strWriter = new StringWriter();
      stringLogger = new MessageLogger(strWriter);
      CatalogJsonVerifier verifier = new CatalogJsonVerifier(oneIntColumnInfo, stringLogger);
      verifier.verify(sqToDq(json));
      //System.out.println(strWriter);
      assertEquals(0, stringLogger.numMessages());
   }

   // TODO - testing array 3 levels down
   //@Test
   public void testEmptyArrayThreeLevelsDownBug510728() throws IOException
   {
      File columnDescriptionsFile = new File("src/test/resources/testData/verification/array_3_levels.columns.tsv");
      HashMap<String, ColumnMetaData> oneFloatColumnInfo = new ColumnMetaDataOperations(columnDescriptionsFile).load();

      StringWriter strWriter = new StringWriter();
      stringLogger = new MessageLogger(strWriter);
      String json = "{'i':{'j':{'f':[]}}}";
      CatalogJsonVerifier verifier = new CatalogJsonVerifier(oneFloatColumnInfo, stringLogger);
      verifier.verify(sqToDq(json));
      assertEquals(0, stringLogger.numWarnings());
      assertEquals(0, stringLogger.numErrors());
   }

   @Test
   public void testNullJsonStringValue()
   {
      String json = "{'CHROM':'1','POS':234567,'ID':null}";
      StringWriter strWriter = new StringWriter();
      stringLogger = new MessageLogger(strWriter);
      CatalogJsonVerifier verifier = new CatalogJsonVerifier(columnInfo, stringLogger);
      verifier.verify(sqToDq(json));
      assertEquals(3, stringLogger.numErrors());
      assertTrue(strWriter.toString().contains("JSON null for key 'ID'"));
      assertTrue(strWriter.toString().contains("Please leave 'ID' out when the value is null"));
      // This is a duplicate message removed related to TFS work item 561394
      assertTrue(!(strWriter.toString().contains("value is JSON-null for key: ID")));
   }

   private static String sqToDq(String str)
   {
      return str.replaceAll("'", "\"");
   }
}
