package edu.mayo.bior.catalog.misses;

import java.io.File;
import java.io.IOException;
import java.util.Arrays;
import java.util.List;

import org.apache.commons.io.FileUtils;
import org.junit.Before;
import org.junit.Test;
import org.junit.rules.TemporaryFolder;

import edu.mayo.pipes.util.test.PipeTestUtils;

public class TabixReaderMissTest {

	
	private File tempDir;
	
	@Before
	public void beforeEach() throws IOException {
		TemporaryFolder temp = new TemporaryFolder();
		temp.create();
		tempDir = temp.newFolder();
	}
	
	@Test
	public void testMissesVcf() throws IOException {
		String catalog = new File("src/test/resources/testData/countMisses/hgnc.misses.tsv.bgz").getCanonicalPath();
		String vcf = new File("src/test/resources/testData/countMisses/misses.vcf").getCanonicalPath();

		File outFile = new File(tempDir, "misses.vcf.txt");
		TabixReaderTest.main(new String[] { "--catalog", catalog, "--output_file_path", outFile.getCanonicalPath(), "--vcf", vcf } );
		List<String> actual = FileUtils.readLines(outFile);
		
		keepFirst3Cols(actual);

		// Remove the first portion of the path from the second line as this is user-specific
		// Ex:  "## VCF File: /Users/m054457/EclipseWorkspace/bior-toolkit/src/test..." 
		//  to: "## VCF File: /src/test..."
		actual.set(1, "## VCF File: " + actual.get(1).substring(actual.get(1).indexOf("/src/test")));
		
		List<String> expected = Arrays.asList(
				"## These are the lines from the VCF file where there was a difference between using the old/bad TabixWriter and the new/fixed one.",
				"## VCF File: /src/test/resources/testData/countMisses/misses.vcf",
				"2	89156674	rs1",
				"2	89160080	rs2",
				"2	89160398	rs3",
				"2	89160733	rs4",
				"2	89161037	rs5",
				"2	89161398	rs6"
				);
		PipeTestUtils.assertListsEqual(expected, actual);
	}


	@Test
	public void testMissesCatalogOnly() throws IOException {
		String catalog = new File("src/test/resources/testData/countMisses/hgnc.misses.tsv.bgz").getCanonicalPath();

		File outFile = new File(tempDir, "misses.catalog.txt");
		TabixReaderTest.main(new String[] { "--catalog", catalog, "--output_file_path", outFile.getCanonicalPath() } );
		List<String> actual = FileUtils.readLines(outFile);
		
		keepFirst3Cols(actual);
		
		List<String> expected = Arrays.asList(
				"2	89156674	89157196",
				"2	89160080	89160117",
				"2	89160398	89160434",
				"2	89160733	89160770",
				"2	89161037	89161074",
				"2	89161398	89161435"
				);
		PipeTestUtils.assertListsEqual(expected, actual);
	}
	
	/** Modifies the list to keep only the first 3 cols to reduce the clutter. */
	private void keepFirst3Cols(List<String> actual) {
		for(int i=0; i < actual.size(); i++) {
			actual.set(i, keepFirstXCols(actual.get(i), 3));
		}
	}

	
	private String keepFirstXCols(String s, int numColsToKeep) {
		int pos = 0;
		for(int i=0; i < numColsToKeep; i++) {
			pos = s.indexOf("\t", pos+1);
			// Tab not found - couldn't get enough columns, so return whole string
			if( pos == -1 )
				return s;
		}
		return s.substring(0, pos);
	}
}
