package edu.mayo.bior.catalog.markdown.transformer.comparison;

import com.google.gson.JsonSyntaxException;
import edu.mayo.bior.catalog.markdown.transformer.MarkdownTransformer;
import edu.mayo.bior.catalog.markdown.transformer.TransformException;
import org.junit.Rule;
import org.junit.Test;
import org.junit.rules.TemporaryFolder;

import java.io.File;
import java.io.IOException;

import static org.apache.commons.io.FileUtils.write;
import static org.junit.Assert.assertEquals;
import static org.junit.Assert.fail;

public class ComparisonStatsImplTest {

    @Rule
    public TemporaryFolder temporaryFolder = new TemporaryFolder();

    private static final char NEWLINE = '\n';

    /**
     * Happy path testing.
     */
    @Test
    public void transform() throws IOException, TransformException {

        // Previous Catalog Rows    = 50
        // Removed Rows             = 10
        // Added Rows               = 60
        // Intersect Rows           = 40
        // Unchanged Intersect Rows = 0
        // Changed Intersect Rows   = 40
        // Current Catalog Rows     = 100

        // COLUMN A is a new column on all intersect and added rows
        // COLUMN B is a removed column on all intersect and removed rows
        // COLUMN C stayed the same on all intersect rows
        // COLUMN D value changed on 30 intersect rows
        // COLUMN E datatype changed on all intersect rows

        final String JSON =
                "{" +
                "  \"diffInfo\": {" +
                "    \"oldCatalogTsvBgz\": \"/path/to/old\"," +
                "    \"newCatalogTsvBgz\": \"/path/to/new\"" +
                "  }," +
                "  \"columnStats\": [" +
                "    {" +
                "      \"name\":            \"COLUMN_A\"," +
                "      \"added\":           100," +
                "      \"valueChanged\":    0," +
                "      \"removed\":         0," +
                "      \"valueUnchanged\":  0," +
                "      \"datatypeChanged\": 0" +
                "    }," +
                "    {" +
                "      \"name\":            \"COLUMN_B\"," +
                "      \"added\":           0," +
                "      \"valueChanged\":    0," +
                "      \"removed\":         50," +
                "      \"valueUnchanged\":  0," +
                "      \"datatypeChanged\": 0" +
                "    }," +
                "    {" +
                "      \"name\":            \"COLUMN_C\"," +
                "      \"added\":           0," +
                "      \"valueChanged\":    0," +
                "      \"removed\":         0," +
                "      \"valueUnchanged\":  40," +
                "      \"datatypeChanged\": 0" +
                "    }," +
                "    {" +
                "      \"name\":            \"COLUMN_D\"," +
                "      \"added\":           0," +
                "      \"valueChanged\":    25," +
                "      \"removed\":         0," +
                "      \"valueUnchanged\":  0," +
                "      \"datatypeChanged\": 0" +
                "    }," +
                "    {" +
                "      \"name\":            \"COLUMN_E\"," +
                "      \"added\":           0," +
                "      \"valueChanged\":    0," +
                "      \"removed\":         0," +
                "      \"valueUnchanged\":  0," +
                "      \"datatypeChanged\": 30" +
                "    }" +
                "    ]," +
                "  \"summaryStats\": {" +
                "    \"catalogColumnDataTypeChange\": 30," +
                "    \"catalogColumnRemoved\":        50," +
                "    \"catalogColumnAdded\":          100," +
                "    \"catalogColumnValueChanged\":   25," +
                "    \"catalogColumnValueUnchanged\": 40," +
                "    \"catalogColumnComparisons\":    245," +
                "    \"catalogColumnAllChanges\":     205," +
                "    \"catalogOldEntries\":           50," +
                "    \"catalogNewEntries\":           100," +
                "    \"catalogEntriesAdded\":         60," +
                "    \"catalogEntriesRemoved\":       10," +
                "    \"catalogEntriesChanged\":       110" +
                "  }" +
                "}";
        File jsonFile = temporaryFolder.newFile("test.json");
        write(jsonFile, JSON);

        MarkdownTransformer transformer = new ComparisonStatsImpl(jsonFile);

        final String ORIGINAL_MARKDOWN =
                "BEGIN" + NEWLINE +
                CatalogTableImpl.TEMPLATE_KEYWORD + NEWLINE +
                CatalogVolumeImpl.TEMPLATE_KEYWORD + NEWLINE +
                CatalogRowUpdateImpl.TEMPLATE_KEYWORD + NEWLINE +
                CatalogTop5ColumnUpdateImpl.TEMPLATE_KEYWORD + NEWLINE +
                CatalogColumnChangesImpl.TEMPLATE_KEYWORD + NEWLINE +
                CatalogContentChangesImpl.TEMPLATE_KEYWORD + NEWLINE +
                "END";

        final String EXPECTED_MARKDOWN =
                "BEGIN" + NEWLINE +


                "Catalog | Path" + NEWLINE +
                "--- | ---" + NEWLINE +
                "Previous | /path/to/old" + NEWLINE +
                "Current | /path/to/new" + NEWLINE +


                "Metric | Description | Count" + NEWLINE +
                "--- | --- | ---" + NEWLINE +
                "Previous Rows | # of rows in the previous catalog | 50" + NEWLINE +
                "Removed Rows | # of rows removed from the previous catalog | -10" + NEWLINE +
                "Added Rows | # of rows added to the previous catalog | +60" + NEWLINE +
                "Current Rows | # of rows in the current catalog | 100" + NEWLINE +
                "Net Difference | Difference between Previous Rows and Current Rows | +50" + NEWLINE +


                "Metric | Description | Count | % of Intersection" + NEWLINE +
                "--- | --- | --- | ---" + NEWLINE +
                "Intersection Rows | # of rows present in both previous and current catalogs | 40 | 100.0%" + NEWLINE +
                "Unchanged Rows | intersection rows with no column changes | 0 | 0.0%" + NEWLINE +
                "Updated Rows | intersection rows that had at least 1 column change | 40 | 100.0%" + NEWLINE +


                "Rank | Updated Column | Update Type | # Updated Rows | % Updated Rows" + NEWLINE +
                "--- | --- | --- | --- | ---" + NEWLINE +
                "1 | COLUMN\\_A | Column Added | 40 | 100.0%" + NEWLINE +
                "2 | COLUMN\\_B | Column Removed | 40 | 100.0%" + NEWLINE +
                "3 | COLUMN\\_E | Column Datatype Changed | 30 | 75.0%" + NEWLINE +
                "4 | COLUMN\\_D | Column Value Updated | 25 | 62.5%" + NEWLINE +
                "5 | | | " + NEWLINE +


                "Metric | Description | Count" + NEWLINE +
                "--- | --- | ---" + NEWLINE +
                "Previous Columns | # of columns in the previous catalog | 4" + NEWLINE +
                "Removed Columns | # of columns removed from the previous catalog | -1" + NEWLINE +
                "Added Columns | # of columns added to the previous catalog | +1" + NEWLINE +
                "Current Columns | # of columns in the current catalog | 4" + NEWLINE +
                "Net Difference | Difference between Previous Columns and Current Columns | 0" + NEWLINE +


                "ColumnName | Removed | Added | Changed | Datatype | Unchanged" + NEWLINE +
                "--- | --- | --- | --- | --- | ---" + NEWLINE +
                "COLUMN\\_A | 0 (0.000%) | 100 (100.000%) | 0 (0.000%) | 0 (0.000%) | 0 (0.000%)" + NEWLINE +
                "COLUMN\\_B | 50 (100.000%) | 0 (0.000%) | 0 (0.000%) | 0 (0.000%) | 0 (0.000%)" + NEWLINE +
                "COLUMN\\_C | 0 (0.000%) | 0 (0.000%) | 0 (0.000%) | 0 (0.000%) | 40 (100.000%)" + NEWLINE +
                "COLUMN\\_D | 0 (0.000%) | 0 (0.000%) | 25 (100.000%) | 0 (0.000%) | 0 (0.000%)" + NEWLINE +
                "COLUMN\\_E | 0 (0.000%) | 0 (0.000%) | 0 (0.000%) | 30 (100.000%) | 0 (0.000%)" + NEWLINE +
                "TOTAL | 50 (20.408%) | 100 (40.816%) | 25 (10.204%) | 30 (12.245%) | 40 (16.327%)" + NEWLINE +


                "END";

        String actualMarkdown = transformer.transform(ORIGINAL_MARKDOWN);
        assertEquals(EXPECTED_MARKDOWN, actualMarkdown);
    }

    /**
     * Test having a JSON file with bad content.
     */
    @Test
    public void transformBadJsonContent() throws IOException {

        final String JSON = "BAD JSON";

        File jsonFile = temporaryFolder.newFile("test.json");
        write(jsonFile, JSON);

        try {
            new ComparisonStatsImpl(jsonFile);
            fail();
        } catch (JsonSyntaxException e) {
            // expected
        }
    }
}
