package edu.mayo.bior.catalog.list;

import edu.mayo.bior.catalog.CatalogFormatException;
import org.apache.commons.lang.StringUtils;
import org.junit.Rule;
import org.junit.Test;
import org.junit.rules.ExpectedException;

import java.util.ArrayList;
import java.util.List;

import static junit.framework.Assert.*;

/**
 * Test CatalogInfo class
 */
public class CatalogInfoTest
{
   @Rule
   public ExpectedException expectedException = ExpectedException.none();

   private static final String TAB = "\t";

   private String status = "Active";
   private String path = "/research/bsi/data/catalogs/bior/v1/mirbase/release19/hsa_GRCh37.p5.tsv.bgz";
   private String lastUpdated = "2013-04-08 13:42:25";
   private String shortName = ".";
   private String indexes = ".";
   private String dataSource = "miRBase";
   private String version = "19";
   private String build = "GRCh37.p5";
   private String buildBase = "GRCh37";
   private String dataset = "miRNA sequences";

   @Test
   public void testBasic() throws CatalogFormatException
   {
      CatalogInfo info = new CatalogInfo(composeLine());
      assertTrue(info.isActive());
      assertFalse(info.isDeprecated());
      assertEquals(path, info.getPath());
      assertEquals(lastUpdated, info.getLastUpdated());
      assertNull(info.getShortName());
      assertEquals(0, info.getIndexesSize());
      assertEquals(dataSource, info.getDataSource());
      assertEquals(version, info.getVersion());
      assertEquals(build, info.getBuild());
      assertEquals(buildBase, info.getBuildBase());
      assertEquals(dataset, info.getDataset());

      status = "Deprecated";
      indexes = "f1,f2";
      info = new CatalogInfo(composeLine());
      assertFalse(info.isActive());
      assertTrue(info.isDeprecated());
      assertEquals(2, info.getIndexesSize());
   }

   @Test
   public void testBadLengthSize1() throws CatalogFormatException
   {
      expectedException.expect(CatalogFormatException.class);
      expectedException.expectMessage("lines with 10 tab");
      expectedException.expectMessage("but got 1 field ");
      new CatalogInfo("");
   }

   @Test
   public void testBadLengthSize3() throws CatalogFormatException
   {
      expectedException.expect(CatalogFormatException.class);
      expectedException.expectMessage("lines with 10 tab");
      expectedException.expectMessage("but got 3 fields ");
      List<String> fields = new ArrayList<String>();
      fields.add(status);
      fields.add(version);
      fields.add(build);
      new CatalogInfo(composeLineFromFields(fields));
   }

   @Test
   public void testBadStatus() throws CatalogFormatException
   {
      expectedException.expect(CatalogFormatException.class);
      expectedException.expectMessage("status");
      status = "none";
      new CatalogInfo(composeLine());
   }

   @Test
   public void testNull() throws CatalogFormatException
   {
      expectedException.expect(RuntimeException.class);
      expectedException.expectMessage("Must supply");
      new CatalogInfo(null);

   }
   private String composeLine()
   {
      List<String> fields = new ArrayList<String>();
      fields.add(status);
      fields.add(path);
      fields.add(lastUpdated);
      fields.add(shortName);
      fields.add(indexes);
      fields.add(dataSource);
      fields.add(version);
      fields.add(build);
      fields.add(buildBase);
      fields.add(dataset);
      return composeLineFromFields(fields);
   }

   private String composeLineFromFields(List<String> fields)
   {
      return StringUtils.join(fields, TAB);
   }
}
