package edu.mayo.bior.pipeline.VEP;

import java.util.Comparator;
import java.util.Map;
import java.util.TreeMap;

import org.apache.log4j.Logger;

/** Compare SIFT and Polyphen severities (first the term, then the scores.  If all tie, then choose first) 
 *    See: http://useast.ensembl.org/info/genome/variation/predicted_data.html?redirect=no */
public class VepCsqSiftPolyphenComparator implements Comparator<SiftPolyphenScore>{
	// Allow keys to be compared WITHOUT case sensitivity
	private static final Map<String, Integer> sTermMap = new TreeMap<String, Integer>(String.CASE_INSENSITIVE_ORDER);
	
	private static final Logger sLogger = Logger.getLogger(VepCsqSiftPolyphenComparator.class);

	static {
		// SIFT fields:
		sTermMap.put("deleterious",	5);
		sTermMap.put("tolerated",	0);
		
		// TODO: Check these!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!
		// PolyPhen fields:
		sTermMap.put("probably_damaging",	5);
		sTermMap.put("possibly_damaging", 	3);
		sTermMap.put("unknown",				1);
		sTermMap.put("benign",				0);
	}
	
	public int compare(SiftPolyphenScore score1, SiftPolyphenScore score2) {
		int termWeight1 = getTermWeight(score1);
		int termWeight2 = getTermWeight(score2);
		
		// If termWeight1 > termWeight2 return -1   (first one is worse than second) 
		if( termWeight1 > termWeight2 )
			return -1;
		// Else, if termWeight1 < termWeight2, return 1 (second is worse than first)
		else if( termWeight1 < termWeight2 )
			return 1;
		
		// Else, tied, so compare SIFT score:
		// If first is more damaging, return -1 (for SIFT, a smaller # is worse)
		if( score1.siftScore < score2.siftScore )
			return -1;
		// Else, if second is more damaging, then return 1
		else if( score2.siftScore < score1.siftScore )
			return 1;
		
		// Else, check Polyphen
		// If first is more damaging, return -1 (for polyphen a bigger # is worse)
		if( score1.polyphenScore > score2.polyphenScore )
			return -1;
		else if( score2.polyphenScore > score1.polyphenScore )
			return 1;
		
		// Else if STILL tied, return the first of the two feature #s
		return  score1.feature.compareToIgnoreCase(score2.feature);
	}

	/** Get the weight associated with each SIFT and PolyPhen term as ranked in sTermMap
	 *  NOTE: This is different from the SIFT and PolyPhen scores returned by VEP */
	protected int getTermWeight(SiftPolyphenScore score) {
		int siftWeight 		= getWeight(score.siftTerm, true);
		int polyphenWeight 	= getWeight(score.polyphenTerm, false);
		return siftWeight + polyphenWeight;
	}
	
	/** If SIFT or PolyPhen term not given (null or empty) assign weight of 0  (there will be a lot of blanks)
	 	Else find it in the map and get the score */
	protected int getWeight(String term, boolean isSiftTerm) {
		Integer weight = 0;
		if( term == null  ||  term.length() == 0 )
			weight = 0;
		else if( ! sTermMap.containsKey(term) ) {
			// If the term was not found in the map, assign its weight to zero and show error
			weight = 0;
			// term was given, but not recognized
			final String MSG = "Warning: VEP CSQ " + (isSiftTerm ? "SIFT" : "PolyPhen") + " term not recognized: " + term;
			System.err.println(MSG);
			sLogger.warn(MSG);
		} else
			weight = sTermMap.get(term);
		return weight;
	}
	
}
