package edu.mayo.bior.pipeline.VEP;

import java.util.Comparator;

import com.google.gson.JsonElement;
import com.google.gson.JsonObject;

/** Compare the CSQ consequences between two JsonObjects representing the VEP CSQ results.
 *  	Return a negative value if the first item is worst/more-damaging than the second.
 *  	Return a positive value if the first item is less damaging than the second
 *  Criteria:<br><ol>
 *  	<li>First compare the sequence ontology term (ex: missense_variant) based on severity</li>
 *   	<li>If those are the same, then compare the SIFT and PolyPhen terms if given, based on a weight given to each term (ex: "deleterious" = 5, "tolerated" = 0)</li>
 *      <li>If those are the same, then compare the SIFT and PolyPhen scores if given, if either the SIFT score OR the PolyPhen score is lower, choose that consequence as the worst.</li>
 *      </ol>
 */
public class VepCsqComparator implements Comparator<JsonObject> {

	private VepCsqSeqOntComparator mSeqOntComparator = new VepCsqSeqOntComparator();
	private VepCsqSiftPolyphenComparator mSiftPolyphenComparator = new VepCsqSiftPolyphenComparator();
	
	/** If the first csq object is worse than the second, return a negative number.
	 *  If the second is worse than the first, return a positive number  */
	public int compare(JsonObject csq1, JsonObject csq2) {
		// First, do a comparison of the sequence ontology value if they are given
		// Let's check the sequence ontology type - from worst to most-benign
		// (examples: stop-gained, missense_variant, upstream_gene_variant, downstream_gene_variant, intron_variant, etc)
		int comparison = compareSeqOnt(csq1, csq2);
		
		// Return the comparison value if not 0
		if( comparison != 0 ) 
			return comparison;
		
		// Else, let's weight the SIFT and Polyphen ***terms***
		return compareSiftPolTerms(csq1, csq2);
	}

	protected int compareSiftPolTerms(JsonObject csq1, JsonObject csq2) {
		SiftPolyphenScore score1 = getSiftPolyphenScore(csq1);
		SiftPolyphenScore score2 = getSiftPolyphenScore(csq2);
		return mSiftPolyphenComparator.compare(score1, score2);
	}

	protected SiftPolyphenScore getSiftPolyphenScore(JsonObject csq) {
		SiftPolyphenScore score = new SiftPolyphenScore(
				getString(csq.get("SIFT_TERM"), ""),
				getDouble(csq.get("SIFT_Score"), 1.0),  // NOTE: 1 is least damaging for SIFT
				getString(csq.get("PolyPhen_TERM"), ""),
				getDouble(csq.get("PolyPhen_Score"), 0.0),  // NOTE: 0 is least damaging for PolyPhen
				getString(csq.get("Feature"), "")
				);
		return score;
	}
	
	protected String getString(JsonElement elem, String defaultVal) {
		if( elem == null ) 
			return defaultVal;
		return elem.getAsString();
	}
	
	protected Double getDouble(JsonElement elem, Double defaultVal) {
		if( elem == null )
			return defaultVal;
		return elem.getAsDouble();
	}
	
	protected int compareSeqOnt(JsonObject csq1, JsonObject csq2) {
		JsonElement seqOnt1 = csq1.get("Consequence");
		JsonElement seqOnt2 = csq2.get("Consequence");
		
		if( seqOnt1 == null  &&  seqOnt2 == null )
			return 0;
		else if( seqOnt1 != null  &&  seqOnt2 == null )
			return -1;
		else if( seqOnt1 == null  &&  seqOnt2 != null )
			return 1;
		
		// Else return the comparison of the ontology terms 
		return mSeqOntComparator.compare(seqOnt1.getAsString(), seqOnt2.getAsString());
	}
	
}

