package edu.mayo.bior.cli.cmd;

import static edu.mayo.bior.cli.cmd.CommandUtil.handleFile;

import java.io.File;
import java.io.IOException;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.HashMap;
import java.util.List;
import java.util.Properties;

import org.apache.commons.cli.CommandLine;
import org.apache.commons.cli.Options;
import org.apache.log4j.Logger;

import edu.mayo.bior.cli.cmd.CommandUtil.FileAttributes;
import edu.mayo.cli.CommandPlugin;
import edu.mayo.pipes.history.ColumnMetaData;
import edu.mayo.pipes.history.ColumnMetaData.Type;
import edu.mayo.pipes.history.ColumnMetaDataOperations;


/** Given a target columns.tsv file to modify, and a patch columns.tsv file to pull authoritative Description and HumanReadableName from,
 *  update the target columns.tsv file and verify its contents (or if doing a test-only run, just report the updates that would occur) */
public class UpdateColumnsTsvCommand implements CommandPlugin {

    private static final String OPTION_COLUMNS_TSV_TO_MODIFY = "f";
    private static final String OPTION_COLUMNS_TSV_PATCH = "p";
    private static final String OPTION_TEST_ONLY = "t";
    
    static Logger sLogger = Logger.getLogger(UpdateColumnsTsvCommand.class);
    
    
    private String mCmdName;
    
    @Override
    public void init(Properties properties) throws Exception {
    	mCmdName = properties.getProperty("command.name");
    }

    
    @Override
    public void execute(CommandLine cl, Options options) throws Exception {
    	try {
	    	boolean isTestOnly = cl.hasOption(OPTION_TEST_ONLY);
	    	File columnsTsvToModify = getColsTsvFileToModify(cl, OPTION_COLUMNS_TSV_TO_MODIFY, isTestOnly);
	    	File patchFile = getPatchFile(cl, isTestOnly);
	
	    	updateColumnsTsv(columnsTsvToModify, patchFile, isTestOnly);
    	} catch(Exception e) {
    		sLogger.error("Error while executing command " + mCmdName, e);
    		throw e;
    	}
    }

    /** Get the patch file / file with authoritative column information */
	private File getPatchFile(CommandLine cl, boolean isTestOnly) throws Exception {
    	// If we are in test mode, then the patch file is not required (could be null), else it is
    	String patchFilePath = cl.getOptionValue(OPTION_COLUMNS_TSV_PATCH);
    	File patchFile = null;
    	if( isTestOnly ) {
    		// Patch file is not required when testing, but if it is provided, then set it
    		if( isGiven(patchFilePath) ) {
    			patchFile = new File(patchFilePath);
     		} else {
     			patchFile = null;
     		}
    	} else {
    		List<FileAttributes> fileAttrs = Arrays.asList(FileAttributes.EXISTS, FileAttributes.READABLE);
    		patchFile = handleFile(cl, OPTION_COLUMNS_TSV_PATCH, fileAttrs);
    	}
    	return patchFile;
	}


	private File getColsTsvFileToModify(CommandLine cl, String optionStr, boolean isTestOnly) throws Exception {
		// If in test-mode only, then the file only needs to exist and be readable,
		// But if we are modifying the original, then we need to have it writeable as well
        List<FileAttributes> fileAttrs = isTestOnly 
        		?  Arrays.asList(FileAttributes.EXISTS, FileAttributes.READABLE)
        		:  Arrays.asList(FileAttributes.EXISTS, FileAttributes.READABLE, FileAttributes.WRITEABLE);
        File file = handleFile(cl, optionStr, fileAttrs);
        return file;
	}

	
	private void updateColumnsTsv(File columnsTsvToModify, File patchFile, boolean isTestOnly) throws IOException {
		// Load the target columns.tsv
		ColumnMetaDataOperations colOps = new ColumnMetaDataOperations(columnsTsvToModify);
		List<ColumnMetaData> colsTarget = colOps.loadAsList();

		// Load the patch columns.tsv if it exists and is readable
		HashMap<String,ColumnMetaData>  colsPatchMap  = new HashMap<String,ColumnMetaData>();
		if( patchFile != null  &&  patchFile.exists()  &&  patchFile.canRead() ) {
			ColumnMetaDataOperations colOpsPatch = new ColumnMetaDataOperations(patchFile);
			colsPatchMap = colOpsPatch.load();
		}
		
		// Loop through the fields in the current/target columns.tsv, and check for any updates
		for(ColumnMetaData targetColMeta : colsTarget) {
			// Get the matching patch field if available
			ColumnMetaData patchColMeta = colsPatchMap.get(targetColMeta.getColumnName());
			String diffs  = getDiffs(targetColMeta, patchColMeta, isTestOnly);
			String verify = getVerifyMsgs(targetColMeta, isTestOnly);
			printMsgs(targetColMeta, diffs, verify);
		}
		
		// If we are NOT in test mode, then write the output back to the target columns.tsv
		if( ! isTestOnly ) {
			colOps.save(colsTarget, /*isForceOverwrite=*/true);
		} else {
			System.err.println("\nNOTE: The test-only flag was supplied, so changes will NOT go into the target columns.tsv\n");
		}
	}


	private void printMsgs(ColumnMetaData targetColMeta, String diffs, String verify) {
		String allMsgs = trimOutsideNewlines(diffs + verify);
		if( allMsgs.length() > 0 ) {
			System.err.println("\n--------------------------------------------\n"
					+          "Field: " + targetColMeta.getColumnName() + "\n"
					+          allMsgs
					);
		}
	}


	private String trimOutsideNewlines(String s) {
		StringBuilder str = new StringBuilder(s);
		
		while(str.length() > 0  &&  str.charAt(0) == '\n') {
			str.deleteCharAt(0);
		}
		while(str.length() > 0  &&  str.charAt(str.length()-1) == '\n') {
			str.deleteCharAt(str.length()-1);
		}
		return str.toString();
	}


	private String getDiffs(ColumnMetaData targetColMeta, ColumnMetaData patchColMeta, boolean isTestOnly) {
		// Keep a running list of the differences and print for each column
		StringBuilder msg = new StringBuilder();
		
		// If the patchColMeta is NOT null, then update the description and humanReadableName if necessary
		if( patchColMeta != null ) {
			msg.append(updateHumanReadableName(targetColMeta, patchColMeta, isTestOnly));
			msg.append(updateDescription(targetColMeta, patchColMeta, isTestOnly));
			msg.append(warnIfTypeDiff(targetColMeta, patchColMeta));
			msg.append(warnIfCountDiff(targetColMeta, patchColMeta));
		}
		
		return msg.toString();
	}


	private String getVerifyMsgs(ColumnMetaData targetColMeta, boolean isTestOnly) {
		StringBuilder msg = new StringBuilder();
		msg.append(warnAndFixDoubleQuotes(targetColMeta, isTestOnly));
		msg.append(warnAndFixBlankColumns(targetColMeta, isTestOnly));
		msg.append(errorOnBadColumnNameCountAndType(targetColMeta, isTestOnly));
		msg.append(warnHumanReadableNameSameAsColumnName(targetColMeta));
		return msg.toString();
	}



	// =============================================================================================================================
	// Methods dealing with differences between the target and patch files
	// =============================================================================================================================
	private String updateHumanReadableName(ColumnMetaData targetColMeta, ColumnMetaData patchColMeta, boolean isTestOnly) {
		String msg = "";
		if( isGiven(patchColMeta.getHumanReadableName())  &&  ! isSame(targetColMeta.getHumanReadableName(), patchColMeta.getHumanReadableName())) {
			msg =      "  HumanReadableName updated: [" + targetColMeta.getHumanReadableName() + "] ==> [" + patchColMeta.getHumanReadableName() + "]\n";
			targetColMeta.setHumanReadableName(patchColMeta.getHumanReadableName());
		}
		return msg;
	}

	private String updateDescription(ColumnMetaData targetColMeta, ColumnMetaData patchColMeta, boolean isTestOnly) {
		String msg = "";
		if( isGiven(patchColMeta.getDescription())  &&  ! isSame(targetColMeta.getDescription(), patchColMeta.getDescription())) {
			msg =      "  Description updated:       [" + targetColMeta.getDescription() + "] ==> [" + patchColMeta.getDescription() + "]\n";
			targetColMeta.setDescription(patchColMeta.getDescription());
		}
		return msg;
	}
	
	private boolean isSame(String s1, String s2) {
		if( s1 == null  &&  s2 == null ) {
			return true;
		} else if( s1 == null ) {
			return false;
		}
		return s1.equals(s2);
	}



	
	private String warnIfTypeDiff(ColumnMetaData targetColMeta, ColumnMetaData patchColMeta) {
		if( isDifferent(targetColMeta.getTypeStrOriginal(), patchColMeta.getTypeStrOriginal()) ) {
			return "  Warning: Type is different between target [" + targetColMeta.getTypeStrOriginal() + "] and patch [" + patchColMeta.getTypeStrOriginal() + "]\n";
		}
		return "";
	}

	private boolean isDifferent(String targetStr, String patchStr) {
		return isGiven(targetStr)  &&  isGiven(patchStr)  &&  ! targetStr.equals(patchStr);
	}


	private String warnIfCountDiff(ColumnMetaData targetColMeta, ColumnMetaData patchColMeta) {
		if( isDifferent(targetColMeta.getCount(), patchColMeta.getCount()) ) {
			return "  Warning: Count is different between target [" + targetColMeta.getCount() + "] and patch [" + patchColMeta.getCount() + "]\n";
		}
		return "";
	}



	// =============================================================================================================================
	// Methods for verifying target file
	// =============================================================================================================================


	private String warnAndFixDoubleQuotes(ColumnMetaData targetColMeta, boolean isTestOnly) {
		StringBuilder msg = new StringBuilder();
		if( targetColMeta.getColumnName().contains("\"") ) {
			msg.append("  ERROR: ColumnName cannot contain double-quotes.\n");
		}
		
		if( targetColMeta.getHumanReadableName().contains("\"") ) {
			msg.append("  Warning: HumanReadableName cannot contain double-quotes.  These have been converted to single quotes.\n");
			targetColMeta.setHumanReadableName(targetColMeta.getHumanReadableName().replaceAll("\"", "'"));
		}

		if( targetColMeta.getDescription().contains("\"") ) {
			msg.append("  Warning: Description cannot contain double-quotes.  These have been converted to single quotes.\n");
			targetColMeta.setDescription(targetColMeta.getDescription().replaceAll("\"", "'"));
		}

		return msg.toString();
	}
	

	private Object warnAndFixBlankColumns(ColumnMetaData targetColMeta, boolean isTestOnly) {
		StringBuilder msg = new StringBuilder();
		
		if( ! isGiven(targetColMeta.getColumnName()) ) {
			msg.append("  ERROR: ColumnName cannot be blank.\n");
		}

		if( ! isGiven(targetColMeta.getDescription())  ||  targetColMeta.getDescription().equals(".") ) {
			msg.append("  Warning: Description is missing.");
			if( ! targetColMeta.getDescription().equals(".")) {
				msg.append("  Setting this to dot.");
				targetColMeta.setDescription(".");
			}
			msg.append("\n");
		}

		if( ! isGiven(targetColMeta.getHumanReadableName())  ||  targetColMeta.getHumanReadableName().equals(".") ) {
			msg.append("  Warning: HumanReadableName is missing.   Setting this to to the same as the ColumnName.\n");
			targetColMeta.setHumanReadableName(targetColMeta.getColumnName());
		}
		
		return msg.toString();
	}
	

	private String errorOnBadColumnNameCountAndType(ColumnMetaData targetColMeta, boolean isTestOnly) {
		StringBuilder msg = new StringBuilder();
		// Type can be only one of several options
		if( ! isValidType(targetColMeta.getType()) ) {
			msg.append("  ERROR: Type [" + targetColMeta.getTypeStrOriginal() + "] is not valid.  Possible types include: " + getTypesAsList() + "\n");
		}
			
		// Count must be valid
		if( ! isValidCount(targetColMeta.getCount()) ) {
			msg.append("  ERROR: Count [" + targetColMeta.getCount() + "] is not valid.  It must be one of the values: '.', 'A', 'G', 'R', or a non-negative integer (0,1,2...)\n");
		}
		return msg.toString();
	}

	private boolean isValidType(ColumnMetaData.Type type) {
		// If it is not null, then it should fit into one of the other types
		if( type == null ) {
			return false;
		}
		return true;
	}
	
	private String getTypesAsList() {
		StringBuilder s = new StringBuilder();
		for(ColumnMetaData.Type type : ColumnMetaData.Type.values()) {
			if( s.length() > 0 ) {
				s.append(",  ");
			}
			s.append(type.name());
				
		}
		return s.toString();
	}
	
	private boolean isValidCount(String count) {
		return ".".equals(count)  ||  "A".equalsIgnoreCase(count) || "R".equalsIgnoreCase(count) || "G".equalsIgnoreCase(count) || isZeroOrPositiveInt(count);
	}
	
	private boolean isZeroOrPositiveInt(String count) {
		try {
			Integer i = Integer.parseInt(count);
			return i >= 0;
		} catch(Exception e) {
			return false;
		}
	}


	private String warnHumanReadableNameSameAsColumnName(ColumnMetaData targetColMeta) {
		String colName = targetColMeta.getColumnName();
		String humanReadableName = targetColMeta.getHumanReadableName();
		if( colName.equals(humanReadableName) ) {
			return "  Warning: HumanReadableName is same as ColumnName [" + colName + "].  Please update the HumanReadableName";
		}
		return "";
	}


	private boolean isGiven(String s) {
		return s != null &&  s.length() > 0;
	}
    

}
