package edu.mayo.bior.cli.cmd;

import java.io.BufferedReader;
import java.io.BufferedWriter;
import java.io.File;
import java.io.IOException;
import java.io.InputStreamReader;
import java.io.OutputStreamWriter;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Properties;

import org.apache.commons.cli.CommandLine;
import org.apache.commons.cli.Options;

import com.google.common.base.Charsets;
import com.google.common.io.Files;

import edu.mayo.cli.CommandPlugin;

/** Replaces one line with another on an input stream
 *  
 *  NOTE: You could also use sed to replace strings, and make it more readable using aliases:
 *    alias replaceFileFormat='sed '\''s/##fileformat=VCFv4.1/##fileformat=VCFv4.3/'\'''
 *    cat my.vcf | replaceFileFormat
 *  
 *  Given two text files:
 *   - replace the 1st line in the first file with the first line in the second;
 *   - then the 2nd line in the first file with the 2nd line in the second; etc
 *  This can be helpful for changing ##INFO lines in VCF files, for example, if you have this line in the input VCF:
 *    ##INFO=<ID=dbSNPBuildID,Number=.,Type=String,Description="First dbSNP Build for RS">
 *  And want this one:
 *    ##INFO=<ID=dbSNPBuildID,Number=1,Type=Integer,Description="First dbSNP Build for RS">
 *  You would add the first line to the replace.txt file, and the second line to the replaceWith.txt file
 * @author Michael Meiners (m054457), Jan 31, 2017
 *
 */
public class ReplaceLinesCommand implements CommandPlugin {

    private static final String OPTION_FILE_LINES_FROM = "f";
    private static final String OPTION_FILE_LINES_TO = "t";


	Map<String,String> replacementMap = new HashMap<String,String>();

    @Override
    public void init(Properties properties) throws Exception {
    }

    @Override
    public void execute(CommandLine cl, Options options) throws Exception {
    	File fileLinesFrom = new File(cl.getOptionValue(OPTION_FILE_LINES_FROM));
    	File fileLinesTo   = new File(cl.getOptionValue(OPTION_FILE_LINES_TO));

    	loadFiles(fileLinesFrom, fileLinesTo);
    	replaceLines();
	}

	public void loadFiles(File fileLinesToReplace, File fileLinesToReplaceWith) throws Exception {
		List<String> linesToReplace = Files.readLines(fileLinesToReplace, Charsets.UTF_8);
		List<String> linesToReplaceWith = Files.readLines(fileLinesToReplaceWith, Charsets.UTF_8);
		
		if( linesToReplace.size() != linesToReplaceWith.size() ) {
			throw new Exception("ReplaceLine: the two input files must have the same number of lines");
		}
		
		for(int i=0; i < linesToReplace.size(); i++) {
			replacementMap.put(linesToReplace.get(i), linesToReplaceWith.get(i));
		}
	}
	
	private void replaceLines() throws IOException {
		BufferedReader fin = new BufferedReader(new InputStreamReader(System.in));
		BufferedWriter fout = new BufferedWriter(new OutputStreamWriter(System.out));
		
		String line = null;
		while( (line = fin.readLine()) != null ) {
			String lineOut = line;
			String lineToReplaceWith = replacementMap.get(line);
			if( lineToReplaceWith != null ) {
				lineOut = lineToReplaceWith;
			}
			if( lineOut != null  &&  lineOut.length() > 0 )
				System.out.println(lineOut);
		}
		
		if( fin != null ) {
			fin.close();
		}

		if( fout != null ) {
			fout.close();
		}
	}
}
