package edu.mayo.bior.cli.cmd;

import java.io.File;
import java.util.Properties;

import org.apache.commons.cli.CommandLine;
import org.apache.commons.cli.Options;
import org.apache.log4j.Logger;

import edu.mayo.bior.buildcatalog.BuildCatalog;
import edu.mayo.bior.buildcatalog.BuildStepKey;
import edu.mayo.cli.CommandPlugin;

/**
 *
 * @author Julie Johnson (m042195)
 * 
 * The bior_build_catalog command does more than it's earlier counter-
 * part 'bior_create_catalog'. The 'bior_build_catalog' command creates
 * not only the catalog and associated tabix index file, but also builds
 * all of the supporting properties files, and indexes, as well as verifying
 * the catalog's and associated property files contents as the last step 
 * of the build process. 
 * 
 * The 'bior_build_catalog' command takes as input two things:
 * 		* a build information property file
 * 		* an input tjson formatted file containing the JSON
 *         column that will be used to build the catalog contents
 *         and other supporting property file information
 * 
 */
public class BuildCatalogCommand implements CommandPlugin {

	private static final Logger sLogger = Logger.getLogger(BuildCatalogCommand.class);

    private static final char OPTION_BUILD_INFO_FILE = 'b';
    private static final char OPTION_START_STEP 	 = 's';
    private static final char OPTION_END_STEP 		 = 'e';
    
    
    public void init(Properties props) {
    	// just going to put below for now.
    }

    public void execute(CommandLine cl, Options options) throws Exception {
    	
    	// The build_info.txt file path is required
        File buildInfoFile = new File(cl.getOptionValue(OPTION_BUILD_INFO_FILE));
        if( ! buildInfoFile.canRead() ) {
        	throw new IllegalArgumentException("Error: Specified build information file does not exist: " + buildInfoFile.getPath());
        }
        
        BuildStepKey stepStart = getStep(cl.getOptionValue(OPTION_START_STEP), BuildStepKey.MAKE_JSON);
        BuildStepKey stepEnd   = getStep(cl.getOptionValue(OPTION_END_STEP),   BuildStepKey.VERIFY);
        
        BuildCatalog biorBuildCatalog = new BuildCatalog();
        biorBuildCatalog.build(buildInfoFile.getCanonicalPath(), stepStart, stepEnd);
	}

	private BuildStepKey getStep(String stepStr,  BuildStepKey defaultStep) {
        // If null, this means it was not passed in as an option by the user
		// Start at the MAKE_JSON step by default.  Override this if user specified a step
		if( stepStr == null ) {
			return defaultStep;
		}
		
		// Go thru each of the BuildStepKey values and compare (case-insensitive), and return any that match
		for(BuildStepKey stepKey : BuildStepKey.values()) {
			if( stepKey.getDisplayName().equalsIgnoreCase(stepStr) ) {
				return stepKey;
			}
		}
		
		// Not found, so throw error 
		StringBuilder msg = new StringBuilder("Step not recognized: " + stepStr + ".\nValid steps are:\n");
		for(BuildStepKey key : BuildStepKey.values()) {
			msg.append("    " + key.getDisplayName() + "\n");
		}
		throw new IllegalArgumentException(msg.toString());
	}


}
