package edu.mayo.bior.catalog.markdown.transformer.comparison;

import edu.mayo.bior.catalog.markdown.transformer.MarkdownTransformer;
import edu.mayo.bior.catalog.markdown.transformer.TransformException;

import java.io.PrintWriter;
import java.io.StringWriter;
import java.util.ArrayList;
import java.util.Comparator;
import java.util.List;

import static edu.mayo.bior.catalog.markdown.MarkdownUtils.chompTrailingNewlines;
import static edu.mayo.bior.catalog.markdown.MarkdownUtils.escapeChrs;
import static edu.mayo.bior.catalog.markdown.transformer.comparison.ComparisonStatsImpl.NOT_AVAILABLE_MESG;
import static java.lang.String.format;
import static java.util.Collections.reverseOrder;
import static java.util.Collections.sort;
import static org.apache.commons.io.IOUtils.closeQuietly;

/**
 * Builds a markdown table about the TOP 5 updated columns.
 */
public class CatalogTop5ColumnUpdateImpl implements MarkdownTransformer {

    static final String TEMPLATE_KEYWORD = "COMPARISON_STATISTICS_TOP5_COL_UPDATE_TABLE";

    private CatalogDiffStatistics stats;

    enum ChangeType {ADD, REMOVE, VALUE_CHANGE, DATATYPE_CHANGE}

    CatalogTop5ColumnUpdateImpl(CatalogDiffStatistics stats) {
        this.stats = stats;
    }

    @Override
    public String transform(String markdown) throws TransformException {
        if (stats == null) {
            return markdown.replace(TEMPLATE_KEYWORD, NOT_AVAILABLE_MESG);
        } else {

            return markdown.replace(TEMPLATE_KEYWORD, chompTrailingNewlines(buildTable()));
        }
    }

    private String buildTable() {
        final int TOP_LIMIT = 5;
        final long UPDATED_ROWS =
                stats.getSummaryStats().getCatalogEntriesChanged()
                        - stats.getSummaryStats().getCatalogEntriesRemoved()
                        - stats.getSummaryStats().getCatalogEntriesAdded();

        List<DiscreteColumnChange> discreteColumnChanges = getDiscreteColumnChanges();

        StringWriter sWtr = new StringWriter();
        PrintWriter pWtr = new PrintWriter(sWtr);

        pWtr.println("Rank | Updated Column | Update Type | # Updated Rows | % Updated Rows");
        pWtr.println("--- | --- | --- | --- | ---");
        for (int rowIdx = 0; rowIdx < TOP_LIMIT; rowIdx++ ) {
            int rank = rowIdx + 1;
            if (rowIdx < discreteColumnChanges.size()) {
                DiscreteColumnChange discreteColumnChange = discreteColumnChanges.get(rowIdx);
                pWtr.println(
                        format("%d | %s | %s | %d | %.1f%%",
                                rank,
                                escapeChrs(discreteColumnChange.getName()),
                                escapeChrs(getDisplayString(discreteColumnChange.getChangeType())),
                                discreteColumnChange.getRowCount(),
                                (float)discreteColumnChange.getRowCount() / (float)UPDATED_ROWS * 100
                        )
                );
            } else {
                pWtr.println(format("%d | | | ", rank));
            }
        }

        closeQuietly(pWtr);
        closeQuietly(sWtr);
        return sWtr.toString();
    }

    /**
     * Gets a user readable string for the given {@link ChangeType}.
     * @param changeType The {@link ChangeType} to be used.
     * @return User display string.
     */
    private String getDisplayString(ChangeType changeType) {
        switch (changeType) {
            case ADD:             return "Column Added";
            case REMOVE:          return "Column Removed";
            case VALUE_CHANGE:    return "Column Value Updated";
            case DATATYPE_CHANGE: return "Column Datatype Changed";
            default: throw new RuntimeException(String.format("Unsupported change type %s", changeType.name()));
        }
    }

    /**
     * Fans out the column changes nested inside a single {@link CatalogDiffStatistics.ColumnStats} object into multiple
     * {@link DiscreteColumnChange} objects (one per type... add, remove, etc...).
     * @return List of the fanned out objects.
     */
    private List<DiscreteColumnChange> getDiscreteColumnChanges() {
        List<DiscreteColumnChange> list = new ArrayList<DiscreteColumnChange>();
        for (CatalogDiffStatistics.ColumnStats columnStats: stats.getColumnStats()) {
            final long UPDATED_ROWS_W_COLS_ADDED = columnStats.getAdded()     - stats.getSummaryStats().getCatalogEntriesAdded();
            final long UPDATED_ROWS_W_COLS_REMOVED = columnStats.getRemoved() - stats.getSummaryStats().getCatalogEntriesRemoved();

            if (UPDATED_ROWS_W_COLS_ADDED > 0)
                list.add(new DiscreteColumnChange(columnStats.getName(), UPDATED_ROWS_W_COLS_ADDED, ChangeType.ADD));

            if (UPDATED_ROWS_W_COLS_REMOVED > 0)
                list.add(new DiscreteColumnChange(columnStats.getName(), UPDATED_ROWS_W_COLS_REMOVED, ChangeType.REMOVE));

            if (columnStats.getValueChanged() > 0)
                list.add(new DiscreteColumnChange(columnStats.getName(), columnStats.getValueChanged(), ChangeType.VALUE_CHANGE));

            if (columnStats.getDatatypeChanged() > 0)
                list.add(new DiscreteColumnChange(columnStats.getName(), columnStats.getDatatypeChanged(), ChangeType.DATATYPE_CHANGE));
        }

        // sort highest to lowest
        sort(list, reverseOrder(new SortByRowCount()));

        return list;
    }

    private class DiscreteColumnChange {
        private String name;
        private Long rowCount;
        private ChangeType changeType;

        DiscreteColumnChange(String name, Long rowCount, ChangeType changeType) {
            this.name = name;
            this.rowCount = rowCount;
            this.changeType = changeType;
        }

        String getName() {
            return name;
        }

        Long getRowCount() {
            return rowCount;
        }

        ChangeType getChangeType() {
            return changeType;
        }
    }

    /**
     * Sort by {@link DiscreteColumnChange#getRowCount()} ()} lowest to highest.
     */
    class SortByRowCount implements Comparator<DiscreteColumnChange> {
        @Override
        public int compare(DiscreteColumnChange o1, DiscreteColumnChange o2) {
            return o1.getRowCount().compareTo(o2.getRowCount());
        }
    }
}