package edu.mayo.bior.catalog.markdown.transformer.comparison;

import edu.mayo.bior.catalog.markdown.transformer.MarkdownTransformer;
import edu.mayo.bior.catalog.markdown.transformer.TransformException;

import java.io.PrintWriter;
import java.io.StringWriter;
import java.util.ArrayList;
import java.util.List;

import static edu.mayo.bior.catalog.markdown.MarkdownUtils.chompTrailingNewlines;
import static edu.mayo.bior.catalog.markdown.MarkdownUtils.positiveNumPrint;
import static edu.mayo.bior.catalog.markdown.transformer.comparison.ComparisonStatsImpl.NOT_AVAILABLE_MESG;
import static java.lang.String.format;
import static org.apache.commons.io.IOUtils.closeQuietly;

/**
 * Builds a markdown table about catalog column changes.
 */
public class CatalogColumnChangesImpl implements MarkdownTransformer {

    static final String TEMPLATE_KEYWORD = "COMPARISON_STATISTICS_COL_CHANGES_TABLE";

    private CatalogDiffStatistics stats;

    CatalogColumnChangesImpl(CatalogDiffStatistics stats) {
        this.stats = stats;
    }

    @Override
    public String transform(String markdown) throws TransformException {
        if (stats == null) {
            return markdown.replace(TEMPLATE_KEYWORD, NOT_AVAILABLE_MESG);
        } else {

            return markdown.replace(TEMPLATE_KEYWORD, chompTrailingNewlines(buildTable()));
        }
    }

    /**
     * Gets all removed columns.  A column is defined to be removed if it's removed on all rows from the
     * previous catalog.
     * @return List of only the removed columns.
     */
    private List<CatalogDiffStatistics.ColumnStats> getRemovedColumns() {
        List<CatalogDiffStatistics.ColumnStats> list = new ArrayList<CatalogDiffStatistics.ColumnStats>();
        for (CatalogDiffStatistics.ColumnStats columnStats: stats.getColumnStats()) {
            if (columnStats.getRemoved() == stats.getSummaryStats().getCatalogOldEntries()) {
                list.add(columnStats);
            }
        }
        return list;
    }

    /**
     * Gets all added columns.  A column is defined to be added if it's added to AT LEAST all rows from the
     * previous catalog.  There may be cases where there are rows added to the previous catalog, which would
     * also show up.
     * @return List of only the added columns.
     */
    private List<CatalogDiffStatistics.ColumnStats> getAddedColumns() {
        List<CatalogDiffStatistics.ColumnStats> list = new ArrayList<CatalogDiffStatistics.ColumnStats>();
        for (CatalogDiffStatistics.ColumnStats columnStats: stats.getColumnStats()) {
            if (columnStats.getAdded() >= stats.getSummaryStats().getCatalogOldEntries()) {
                list.add(columnStats);
            }
        }
        return list;
    }

    private String buildTable() {

        List<CatalogDiffStatistics.ColumnStats> addedColumnList = getAddedColumns();
        List<CatalogDiffStatistics.ColumnStats> removedColumnList = getRemovedColumns();

        int supersetAllCols = stats.getColumnStats().length;
        int removedCols = removedColumnList.size();
        int addedCols = addedColumnList.size();
        int previousCols = supersetAllCols - addedCols;
        int currentCols = supersetAllCols - removedCols;
        int netDifference = currentCols - previousCols;

        StringWriter sWtr = new StringWriter();
        PrintWriter pWtr = new PrintWriter(sWtr);

        pWtr.println("Metric | Description | Count");
        pWtr.println("--- | --- | ---");

        pWtr.println(
                format("%s | %s | %d",
                        "Previous Columns",
                        "# of columns in the previous catalog",
                        previousCols
                )
        );
        pWtr.println(
                format("%s | %s | %s",
                        "Removed Columns",
                        "# of columns removed from the previous catalog",
                        positiveNumPrint(-1 * removedCols)
                )
        );
        pWtr.println(
                format("%s | %s | %s",
                        "Added Columns",
                        "# of columns added to the previous catalog",
                        positiveNumPrint(addedCols)
                )
        );
        pWtr.println(
                format("%s | %s | %d",
                        "Current Columns",
                        "# of columns in the current catalog",
                        currentCols
                )
        );
        pWtr.println(
                format("%s | %s | %s",
                        "Net Difference",
                        "Difference between Previous Columns and Current Columns",
                        positiveNumPrint(netDifference)
                )
        );
        closeQuietly(pWtr);
        closeQuietly(sWtr);
        return sWtr.toString();
    }
}