package edu.mayo.bior.buildcatalog;

import java.io.File;
import java.util.Arrays;
import java.util.List;

import org.apache.commons.lang.StringUtils;

import com.jayway.jsonpath.InvalidPathException;

/** Provide a standard way of getting a temp directory from a prioritized list */ 
public class TempDirectory {
	   private static final String SEP = File.separator;

	   /** A prioritized list of top-level temp directories to use.  These may or may not exist on the server where the code is run */
	   public static final List<String> TEMP_DIR_PRIORITY_LIST = Arrays.asList(
			   "/local2/tmp",
			   "/local1/tmp",
			   System.getProperty("java.io.tmpdir"),
			   "/tmp"
	   );

	   /** Find the preferred base temp directory based on a prioritized list
	    *  and the user's subdirectory within that.
	    *  Ex:  If preferred temp dir exists:  /local2/tmp
	    *       and user is:                   m001122
	    *       and user-supplied subdir is:   bior_build_catalog_tmp
	    *       Return path:                   /local2/tmp/m001122/bior_build_catalog_tmp
	    * @param  subdir  Subdirectory to create under the base temp directory (or pass null or "" to not add a subdirectory).  
	    *                 An example of this might be "bior_build_catalog_tmp"
	    * @return Base temp directory
	    * @throws BuildCatalogStepInputException
	    */
	   public static String findBaseTempDir(String subdir) throws BuildCatalogStepInputException
	   {
	      for ( String tempPath : TEMP_DIR_PRIORITY_LIST )
	      {
	    	  // Check to see if the path exists
	         String tempSubPath = createTempSubdir(tempPath, subdir);
	         
	         // If the path is valid and 
	         if (tempSubPath != null) {
	            return tempSubPath;
	         }
	      }

	      // totally shut out
	      throw new InvalidPathException("Couldn't find anywhere to make a temporary directory");
	   }

	   /** Try to create the "[USERNAME]/[subdir]" subdirectory.  If successful, return the path, else return null to signal that it failed */
	   private static String createTempSubdir(String path, String subdir)  {
	      File baseTempDir = new File(path);
	      if (baseTempDir.exists())
	      {
	         String userName = System.getProperty("user.name");
	         String testPath = baseTempDir 
	         					+ (StringUtils.isBlank(userName) ? "" : SEP + userName)
	         					+ (StringUtils.isBlank(subdir)   ? "" : SEP + subdir);
	         File testDir = new File(testPath);
	         // If exists and is readable and writeable, return it
	         if (testDir.exists() && testDir.canRead() && testDir.canWrite())  {
	            return testPath;
	         }
	         // If it doesn't exist, try creating it and if successful, return that path
	         if (testDir.mkdirs())  {
	            return testPath;
	         }
	      }
	      return null;
	   }

}
