package edu.mayo.bior.buildcatalog;

import java.io.File;
import java.io.IOException;
import java.io.StringWriter;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Iterator;
import java.util.LinkedHashMap;
import java.util.List;

import org.apache.commons.configuration.PropertiesConfiguration;
import org.apache.commons.io.FileUtils;

import edu.mayo.bior.catalog.CatalogDataSource;
import edu.mayo.bior.catalog.CatalogFormatException;
import edu.mayo.bior.catalog.DataSourceKey;
import edu.mayo.bior.catalog.verification.CatalogVerifier;
import edu.mayo.bior.catalog.verification.MessageLogger;



/** 3-way merge of columns.tsv files between the current (possibly manually edited) file,
 *  the default file from crawling the newly built catalog,
 *  and the previous catalog's file
 *  @author Michael Meiners (m05445) - 2016-05-06 
 *  
 *  An example datasource.properties file from dbSNP 142 GRCh37:  00-All.vcf.datasource.properties<br>
   		### Datasource properties file for Catalog. Please fill in the descriptions to the keys below.<br>
		ShortUniqueName=dbSNP_142_GRCh37p13
		Description=NCBI's dbSNP Variant Database
		Source=dbSNP
		Dataset=Variants
		## Version of the data source
		Version=142
		## The Genome build/assembly
		Build=GRCh37.p13
		## The BioR catalog compatibility format
		Format=1.1.0
*/
public class MergeDataSourceProperties {

	private BuildInfo  mBuildInfo;
	private StepLogger mStepLogger;
	private boolean    mIsUpdateNecessary = false;
		
	private LinkedHashMap<DataSourceKey, String> mKeyToDescMap = new LinkedHashMap<DataSourceKey,String>();
	
	private final List<DataSourceKey>  KEYS_TO_NOT_OVERRIDE_FROM_PREVIOUS_CATALOG = Arrays.asList(
			DataSourceKey.ShortUniqueName,
			DataSourceKey.Version,
			DataSourceKey.DataSourceReleaseDate
			);
	
	/** Do a 3-way merge for the datasource.properties.  If just trying to write the one file
	 *   without any dependencies on the default and previous catalogs, pass values for catalogDir and catalogShortName,
	 *   but leave previousCatalogDataSourceProperties and stepLogger null. 
	 * @param buildInfo  The buildInfo object created from data in the build_info.txt file
	 * @param stepLogger
	 */
	public MergeDataSourceProperties(BuildInfo buildInfo, StepLogger stepLogger) {
		mBuildInfo = buildInfo;
		mStepLogger = stepLogger;

		initDefaultDescriptionMap();
	}


	protected static File getDataSourcePropertiesFromCatalog(File prevCatalogFile) {
		if( prevCatalogFile == null )
			return null;
		return new File(prevCatalogFile.getParentFile(), prevCatalogFile.getName().replace(".tsv.bgz", ".datasource.properties"));
	}



	private void initDefaultDescriptionMap() {
		mKeyToDescMap.put(DataSourceKey.ShortUniqueName, "### Datasource properties file for Catalog - buildCatalog.allSteps.  Please fill in the descriptions to the keys below." + "\n" +
														 "## Short name that should be unique (or mostly unique except for fixes to existing catalog) across all catalogs. Ex: dbSNP_142_GRCh37p13");
		mKeyToDescMap.put(DataSourceKey.Description,     "## Description of catalog.  Ex: NCBI's dbSNP Variant Database");
		mKeyToDescMap.put(DataSourceKey.Source,			 "## Source of data, without point release, etc.  Ex: dbSNP");
		mKeyToDescMap.put(DataSourceKey.Dataset,		 "## Type of data.  Ex: Variants");
		mKeyToDescMap.put(DataSourceKey.Version, 		 "## Version of the data source.  Ex: 142");
		mKeyToDescMap.put(DataSourceKey.Build,			 "## The Genome build/assembly.  Ex: GRCh37.p13");
		mKeyToDescMap.put(DataSourceKey.Format,			 "## The BioR catalog compatibility format.  Ex: 1.1.1");
		mKeyToDescMap.put(DataSourceKey.DataSourceReleaseDate,"## The release date of the data source from the provider (not the BioR build date).  Ex: 2018-07-10");
	}

	public void mergeDatasourceProperties() throws Exception {
		File currentFile = new File(mBuildInfo.getTargetDirectoryFile(), mBuildInfo.getCatalogPrefix() + ".datasource.properties");
		File defaultFile = new File(mBuildInfo.getTargetDirectoryFile() + "/" + BuildCatalog.BUILD_SUBDIR,  mBuildInfo.getCatalogPrefix() + ".datasource.properties.default");
		File prevCtgFile = null;
		if( mBuildInfo.getPreviousCatalogPath() != null)
			prevCtgFile = getDataSourcePropertiesFromCatalog(new File(mBuildInfo.getPreviousCatalogPath()));

		// Backup the current datasource.properties file (using MergeColumnsTsv's backup method since it is re-usable here)
		BackupUtils.backupCurrentFile(currentFile);

		PropertiesConfiguration buildInfoProps = getPropsFromBuildInfo(mBuildInfo);
		PropertiesConfiguration currentProps = loadProps(currentFile, /*isShowErrorOnLoadFailure=*/true);
		PropertiesConfiguration defaultProps = loadProps(defaultFile, /*isShowErrorOnLoadFailure=*/true);
		PropertiesConfiguration prevCtgProps = loadProps(prevCtgFile, /*isShowErrorOnLoadFailure=*/false);

		// Keep track of the list of keys in the default and previousCatalog.  We will print out any extras.
		List<String> defaultPropKeysUnused = toList(defaultProps.getKeys());
		List<String> prevCtgPropKeysUnused = toList(prevCtgProps.getKeys());
		
		
		Iterator<String> currentKeys = currentProps.getKeys();
		// Go thru each of the required keys and add them as necessary
		for( DataSourceKey key : DataSourceKey.values() ) {
			// Determine the best value from amongst the build_info.txt,  current proper
			String val = getBestValue(key, buildInfoProps, currentProps, defaultProps, prevCtgProps);
			currentProps.setProperty(key.toString(), val);
			// Remove the key from the list for defaults and prevCtg lists since we know we've encountered it now
			defaultPropKeysUnused.remove(key.toString());
			prevCtgPropKeysUnused.remove(key.toString());

			String desc = getBestDescription(key, currentProps, defaultProps, prevCtgProps);
			currentProps.getLayout().setComment(key.toString(), desc);
		}
		
		
		warnIfAnyDefaultKeysUnused(defaultPropKeysUnused, currentFile, defaultFile);
		
		warnIfAnyPreviousCatalogKeysUnused(prevCtgPropKeysUnused, currentFile, prevCtgFile);
		
		warnIfCurrentDiffersFromPrevious(currentProps, prevCtgProps);

		// Write the properties file to disk
		currentProps.save(currentFile);
		
		addTopMostCommentIfMissing(currentFile);
		
		removeSpacesAroundEquals(currentFile);

		String path = currentFile.getPath();
		if( BackupUtils.isCurrentSameAsLastBackup(currentFile) ) {
			mStepLogger.logAndSummary(String.format("No changes made to '%s'", path));
			BackupUtils.removeLastBackup(currentFile);
		} else {
			mStepLogger.logAndSummary(String.format("Some changes made to '%s'", path));
		}

		CatalogDataSource catalogDataSrc;
		try
		{
			catalogDataSrc = new CatalogDataSource(currentFile);
		}
		catch (CatalogFormatException e)
		{
			mStepLogger.logAndSummary(String.format("Datasource.properties file '%s' has some critical issues. See verify output for details.", path));
			return;
		}

		MessageLogger verifyLogger = new MessageLogger(new StringWriter());
		catalogDataSrc.verify(CatalogVerifier.VAL_TYPE.STRICT, verifyLogger);
		if (verifyLogger.hasErrors() || verifyLogger.hasWarnings())
		{
			mStepLogger.logAndSummary(String.format("Datasource.properties file '%s' has some issues. See verify output for details.", path));
		}
	}


	private PropertiesConfiguration getPropsFromBuildInfo(BuildInfo buildInfo) throws IllegalStateException {
		PropertiesConfiguration buildInfoProps = new PropertiesConfiguration();
		buildInfoProps.setProperty(DataSourceKey.ShortUniqueName.toString(),		getShortUniqueName(buildInfo));
		buildInfoProps.setProperty(DataSourceKey.Source.toString(),  				buildInfo.getDataSourceName());
		buildInfoProps.setProperty(DataSourceKey.Version.toString(),				buildInfo.getDataSourceVersion());
		buildInfoProps.setProperty(DataSourceKey.Build.toString(),					buildInfo.getDataSourceBuild());
		buildInfoProps.setProperty(DataSourceKey.Description.toString(),			blankIfNull(buildInfo.getDataSourceDescription()));
		buildInfoProps.setProperty(DataSourceKey.Dataset.toString(),				blankIfNull(buildInfo.getDataSourceDataset()));
		buildInfoProps.setProperty(DataSourceKey.DataSourceReleaseDate.toString(),	blankIfNull(buildInfo.getDataSourceReleaseDate()));
		// NOTE: "Format" value should be filled in by the code elsewhere and set to the latest catalog-format-version
		buildInfoProps.setProperty(DataSourceKey.Format.toString(),					"");
		
		verifyAllDataSourceKeysSetInBuildInfoProps(buildInfoProps);
		
		return buildInfoProps;
	}


	private String blankIfNull(String s) {
		return (s == null)  ?  ""  :  s;
	}


	private void verifyAllDataSourceKeysSetInBuildInfoProps(PropertiesConfiguration buildInfoProps) throws IllegalStateException {
		for(DataSourceKey key : DataSourceKey.values()) {
			String val = (String)buildInfoProps.getProperty(key.name());
			if( val == null )
				throw new IllegalStateException("Programming error:  DataSourceKey." + key.name() + " should be added from the build_info.txt data in MergeDataSourceProperties.getPropsFromBuildInfo().  It seems to have been forgotten.");
		}
	}


	/** Built from: DATA_SOURCE + _ + DATA_SOURCE_VERSION + _ + DATA_SOURCE_BUILD
	 *  Remove all non-alphanumeric characters from the version and build
	    Warn if any are missing and use the defaults in that case
	    Then remove all non-alphanumeric and underscore characters
	    Ex: DATA_SOURCE = dbSNP
	        DATA_SOURCE_VERSION = 142
	        DATA_SOURCE_BUILD = GRCh37.p13
	        ==> ShortUniqueName=dbSNP_142_GRCh37p13
	    Ex: DATA_SOURCE = omim
	        DATA_SOURCE_VERSION = 2018_08_05
	        DATA_SOURCE_BUILD = GRCh37.p13
	        ==> ShortUniqueName=omim_20180805_GRCh37p13
	    */
	protected String getShortUniqueName(BuildInfo buildInfo) {
		final String DEFAULT_SOURCE  = "DATASOURCENAME";
		final String DEFAULT_VERSION = "VERSION";
		final String DEFAULT_BUILD   = "BUILD";
		
		String source = DEFAULT_SOURCE;
		if( isGiven(buildInfo.getDataSourceName()) ) {
			source = buildInfo.getDataSourceName();
		} else {
			mStepLogger.log("Warning: (datasource.properties): Building ShortUniqueName, but could not find a value for "
					+ BuildInfoKey.DATA_SOURCE.name() + " in the build_info.txt file.  Using default: " + DEFAULT_SOURCE
					+ ".  Please update this in the datasource.properties file.");
		}
		
		String version = DEFAULT_VERSION;
		if( isGiven(buildInfo.getDataSourceVersion()) ) {
			version = buildInfo.getDataSourceVersion().replaceAll("[^A-Za-z0-9]", "");
		} else {
			mStepLogger.log("Warning: (datasource.properties): Building ShortUniqueName, but could not find a value for "
					+ BuildInfoKey.DATA_SOURCE_VERSION.name() + " in the build_info.txt file.  Using default: " + DEFAULT_VERSION
					+ ".  Please update this in the datasource.properties file.");
		}

		// Add build:  "datasourcename_version_BUILD"
		String build = DEFAULT_BUILD;
		if( isGiven(buildInfo.getDataSourceBuild()) ) {
			build = buildInfo.getDataSourceBuild().replaceAll("[^A-Za-z0-9]", "");
		} else {
			mStepLogger.log("Warning: (datasource.properties): Building ShortUniqueName, but could not find a value for "
					+ BuildInfoKey.DATA_SOURCE_BUILD.name() + " in the build_info.txt file.  Using default: " + DEFAULT_BUILD
					+ ".  Please update this in the datasource.properties file.");
		}

		String shortUniqueName = source + "_" + version + "_" + build;
		return shortUniqueName;
	}


	private void warnIfCurrentDiffersFromPrevious(PropertiesConfiguration currentProps, PropertiesConfiguration prevCtgProps) {
		Iterator<String> keys = currentProps.getKeys();
		while(keys.hasNext() ) {
			String key = keys.next();
			String currVal = currentProps.getString(key);
			String prevVal = prevCtgProps.getString(key);
			if( isGiven(currVal) && isGiven(prevVal)  &&  ! currVal.equals(prevVal) )
				mStepLogger.log("Note: (datasource.properties): '" + key + "' value is different between current ('" + currVal + "') and previous ('" + prevVal + "') catalog.");
		}
	}


	private void removeSpacesAroundEquals(File currentPropsFile) throws IOException {
		List<String> lines = FileUtils.readLines(currentPropsFile);
		for(int i=0; i < lines.size(); i++) {
			String line = lines.get(i);
			
			// Skip if line is a comment or does not have an equals sign 
			if( line.startsWith("#")  ||  line.indexOf("=") == -1 )
				continue;
			
			// If the first equals sign (note: could have multiples if extras exist in the value) has spaces around it, then trim those out
			if( line.indexOf("=") > 0  &&  (line.indexOf("=")  ==  (1 + line.indexOf(" = "))) )
				lines.set(i, line.replace(" = ", "="));
		}
		FileUtils.writeLines(currentPropsFile, lines);
	}


	private void warnIfAnyPreviousCatalogKeysUnused(List<String> prevCtgPropKeysUnused, File currentPropsFile, File prevCtgPropsFile)	throws IOException {
		// Log any values that were in the previous catalog, but not in the current
		if( prevCtgPropKeysUnused.size() > 0 ) {
			log("These keys were found in the previous catalog file:");
			log("    " + prevCtgPropsFile.getCanonicalPath());
			log("but not in the current one:");
			log("    " + currentPropsFile.getCanonicalPath());
			log("--------:");
			for(String key : prevCtgPropKeysUnused)
				log("    " + key);
		}
	}


	private void warnIfAnyDefaultKeysUnused(List<String> defaultPropKeysUnused, File currentPropsFile, File defaultPropsFile) throws IOException {
		// Log any values that were in the defaults, but not in the current
		if( defaultPropKeysUnused.size() > 0 ) {
			mIsUpdateNecessary = true;
			log("These keys were found in the defaults file:");
			log("    " + defaultPropsFile.getCanonicalPath());
			log("but not in the current one:");
			log("    " + currentPropsFile.getCanonicalPath());
			log("--------:");
			for(String key : defaultPropKeysUnused)
				log("    " + key);
		}
	}


	/** Add the "### Datasource properties file for Catalog - ..." to top if it doesn't exist 
	 * @throws IOException */
	private void addTopMostCommentIfMissing(File currentPropsFile) throws IOException {
		// If the very top comment about the catalog shortname is not in there, then add it
		String currentProps = FileUtils.readFileToString(currentPropsFile);
		String topMostComment =       "### Datasource properties file for Catalog - " + mBuildInfo.getCatalogPrefix() + ".  Please fill in the descriptions to the keys below.\n";
		if( ! currentProps.startsWith("### Datasource properties file for Catalog -") )
			currentProps = topMostComment + currentProps;
		
		// If it has the substitution variable for the catalog prefix, then add the real prefix
		final String CATALOG_PREFIX_VAR = "_CATALOG_PREFIX_";
		if( currentProps.contains(CATALOG_PREFIX_VAR) )
			currentProps = currentProps.replace(CATALOG_PREFIX_VAR, mBuildInfo.getCatalogPrefix());
	
		FileUtils.write(currentPropsFile, currentProps);
	}


	/** Adds comments above a particular property if it is not given */
	private String getBestDescription(DataSourceKey key,
			PropertiesConfiguration currentProps,
			PropertiesConfiguration defaultProps,
			PropertiesConfiguration prevCtgProps)
	{
		// If the description is not in the current properties, then add it (from previous catalog preferably, else the defaults)
		String desc = currentProps.getLayout().getComment(key.toString());
		if( ! isGiven(desc) ) {
			log("Description/comment missing for key: '" + key.toString() + "'");
			
			// If description is present in the previous catalog, use that
			String prevCtgDesc = prevCtgProps.getLayout().getComment(key.toString());
			String defaultDesc = defaultProps.getLayout().getComment(key.toString());
			if( isGiven(prevCtgDesc) ) {
				desc = prevCtgDesc;
				log("    Using previous catalog description/comment: " + prevCtgDesc);
			}
			// Else if the description is present in the defaults, use that
			else if( isGiven(defaultDesc) ) {
				desc = defaultDesc;
				log("    Using default description/comment: " + defaultDesc);
			}
			// Else use the hard-coded description
			else {
				desc = mKeyToDescMap.get(key);
				log("    Using hard-coded description/comment: " + desc);
			}
		}
		return desc;
	}


	/** Modify the currentProps as necessary from the defaultProps and prevCtgProps.
	 *  If any keys are missing (the key must be present, but its value blank), 
	 *  add them from the previous catalog or default properties, 
	 *  BUT, WARN the user as this will likely have to be changed!
	 *  Also, warn if any defaults or previous keys exist that are NOT in the current props! 
	 *
	 */
	private String getBestValue(DataSourceKey key,
			PropertiesConfiguration buildInfoProps,
			PropertiesConfiguration currentProps,
			PropertiesConfiguration defaultProps,
			PropertiesConfiguration prevCtgProps )
	{
		String val = currentProps.getString(key.toString());
		if( ! isGiven(key, currentProps)  ) {
			// If the key is present in the build_info.txt file, use that value
			if( isGiven(key, buildInfoProps) ) {
				val = buildInfoProps.getString(key.toString());
				// Just use the log (and not logAndSummary) since this is the expected path when running updates
				log("Note: (datasource.properties): Value for key '" + key + "' coming from build_info.txt file with value: " + val);
				
				return val;
			}
			
			if( KEYS_TO_NOT_OVERRIDE_FROM_PREVIOUS_CATALOG.contains(key) )
				return val;
			
			mIsUpdateNecessary = true;
			log("Warning: (datasource.properties): Key '" + key + "' missing, or value is empty.  Attempting to add key and value...");
			
			// If the key is present in the previous catalog, use the value from there
			if( isGiven(key, prevCtgProps) ) {
				val = prevCtgProps.getString(key.toString());
				log("    Using the previous catalog key and value: " + val);
			}
			// Else if the key is present in the defaults catalog, use the value from there
			else if( isGiven(key, defaultProps) ) {
				val = defaultProps.getString(key.toString());
				log("    Using the default catalog key and value: " + val);
			}
			// Else add it as a blank line with just the key
			else {
				val = "";
				log("    Could not find a value in the previous catalog or defaults, so setting it to empty");
			}
		}
		return val;
	}



	private List<String> toList(Iterator<String> keys) {
		List<String> keyList = new ArrayList<String>();
		while(keys.hasNext()) {
			keyList.add(keys.next());
		}
		return keyList;
	}


	/** Return true if the key is present in the properties object and is not null and not blank */
	private boolean isGiven(DataSourceKey key,  PropertiesConfiguration props) {
		if( props == null )
			return false;
		
		String val = props.getString(key.toString());
		return  val != null  &&  val.trim().length() > 0;
	}
	
	/** Return true if the string is not null and not blank */
	private boolean isGiven(String s) {
		return s != null  &&  s.trim().length() > 0;
	}


	private PropertiesConfiguration loadProps(File propsFile, boolean isShowErrorOnLoadFailure) throws Exception {
		// NOTE: Do NOT call the constructor with the file passed in, or it will double all comment lines!
		//       (Or at least it does if calling the empty constructor first, then trying to call the constructor with the file next.)
		PropertiesConfiguration props = new PropertiesConfiguration();
		try {
			if( propsFile != null ) {
				props.load(propsFile);
			}
			replaceAnyDoubleQuotes(props, propsFile);
		} catch(Exception e) {
			String msg = "Could not load the data source properties file: " + (propsFile == null  ?  "(null)"  :  propsFile.getCanonicalPath());
			if( isShowErrorOnLoadFailure ) {
				mIsUpdateNecessary = true;
				logAndSummary("ERROR: (datasource.properties): " + msg);
				throw e;
			} else {
				log("Warning: (datasource.properties): " + msg);
			}
		}
		return props;
	}

	/** If there are any double-quotes in the datasource.properties file, replace them with single quotes 
	 * @throws IOException */
	private void replaceAnyDoubleQuotes(PropertiesConfiguration props, File propsFile) throws IOException {
		Iterator<String> keyIter = props.getKeys();
		while( keyIter.hasNext() ) {
			String key = (String)(keyIter.next());
			String val= props.getString(key);
			if( val.contains("\"") ) {
				if( isQuotesJustOnEnds(val) ) {
					val = val.substring(1, val.length()-1);
				} else {
					String errorMsg = "Warning: the " + key + " value in datasource.properties contains a double-quote which could mess up any downstream VCF validators.  Converting these to single-quotes.  Offending value: [" + val + "] in file: " + propsFile.getCanonicalPath();
					log(errorMsg);
					System.err.println(errorMsg);
					val = val.replaceAll("\"", "'");
				}
				props.setProperty(key, val);
			}
		}
	}


	private boolean isQuotesJustOnEnds(String val) {
		if( val == null ) {
			return false;
		}
		return val.length() >= 2  &&  val.charAt(0) == '\"'  &&  val.charAt(val.length()-1) == '\"';
	}


	private void log(String msg) {
		if( mStepLogger == null )
			return;
		mStepLogger.log(msg);
	}

	private void logAndSummary(String msg) {
		if( mStepLogger == null )
			return;
		mStepLogger.logAndSummary(msg);
	}

}
