package edu.mayo.bior.util;

import java.io.File;
import java.util.*;

import edu.mayo.bior.buildcatalog.BuildCatalog;
import edu.mayo.bior.buildcatalog.BuildInfo;
import edu.mayo.bior.buildcatalog.BuildInfoKey;
import edu.mayo.bior.buildcatalog.BuildStepKey;
import edu.mayo.pipes.history.ColumnMetaData;
import edu.mayo.pipes.history.History;
import edu.mayo.pipes.history.HistoryMetaData;
import org.apache.commons.io.FileUtils;
import org.junit.rules.TemporaryFolder;

import static edu.mayo.bior.catalog.CatalogFileUtils.findAllTsvBgzFiles;
import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertTrue;

/** Provide helper methods to other Test classes */
public class TestUtil {

	public static String concat(String... cols) {
		StringBuilder str = new StringBuilder();
		for(int i=0; i < cols.length; i++) {
			if( i > 0 )
				str.append("\t");
			str.append(cols[i]);
		}
		return str.toString();
	}
	
	public static String swapQuotes(String s) {
		return s.replaceAll("'", "\"");
	}
	
	/** Create a History object with the merged header (separated by tabs) */
	protected History createHistoryWithHeader(String mergedHeader) {
		History history = new History();
		history.setMetaData(new HistoryMetaData(null));
		history.getMetaData().setOriginalHeader(null);
		if( mergedHeader == null || mergedHeader.length() == 0 )
			return history;
		
		List<String> header = new ArrayList<String>();
		header.addAll(Arrays.asList(mergedHeader.split("\t")));
		HistoryMetaData historyMetaData = new HistoryMetaData(header);
		historyMetaData.setOriginalHeader(header);
		history.setMetaData(historyMetaData);
		for(String headerName : header) {
			historyMetaData.getColumns().add(new ColumnMetaData(headerName));
		}
		return history;
	}

	/**
	 * Helper method that creates a Catalog that has all of the standard files but has no real data.  This can be
	 * helpful when writing tests that reference Catalogs but don't dive into the Catalog data itself.
	 * @param targetDir   Directory where the Catalog will be created
	 * @param source      Corresponding property from datasource.properties
	 * @param version     Corresponding property from datasource.properties
	 * @param build       Corresponding property from datasource.properties
	 * @param description Corresponding property from datasource.properties
	 * @param dataset     Corresponding property from datasource.properties
	 * @param releaseDate Corresponding property from datasource.properties
	 * @param tempFolder  JUNIT {@link TemporaryFolder} used for creating temp dirs and files
	 * @return {@link File} that points to the created .tsv.bgz data file under targetDir
	 */
	public static File buildEmptyCatalog(
			File targetDir,
			String source,
			String version,
			String build,
			String description,
			String dataset,
			String releaseDate,
			TemporaryFolder tempFolder) throws Exception {

		// no-op script that does nothing other than create an empty tjson file
		File makeJsonScript = tempFolder.newFile("make_json.sh");
		assertTrue(makeJsonScript.setExecutable(true));
		FileUtils.write(makeJsonScript, "echo -e '1\t1\t1\t{}' > $MAKE_JSON_OUTPUT_FILE_PATH");

		Map<String, String> buildInfoMap = new HashMap<String, String>();
		buildInfoMap.put(BuildInfoKey.MAKE_JSON_SCRIPT_PATH.name(),    makeJsonScript.getCanonicalPath());
		buildInfoMap.put(BuildInfoKey.MAKE_JSON_ARGS.name(),           "");
		buildInfoMap.put(BuildInfoKey.CATALOG_PREFIX.name(),           source + "_no_data.tsv");
		buildInfoMap.put(BuildInfoKey.TARGET_DIR.name(),               targetDir.getAbsolutePath());
		buildInfoMap.put(BuildInfoKey.TEMP_DIR.name(),                 tempFolder.newFolder().getAbsolutePath());
		buildInfoMap.put(BuildInfoKey.DATA_SOURCE.name(),              source);
		buildInfoMap.put(BuildInfoKey.DATA_SOURCE_VERSION.name(),      version);
		buildInfoMap.put(BuildInfoKey.DATA_SOURCE_BUILD.name(),        build);
		buildInfoMap.put(BuildInfoKey.DATA_SOURCE_DESCRIPTION.name(),  description);
		buildInfoMap.put(BuildInfoKey.DATA_SOURCE_DATASET.name(),      dataset);
		buildInfoMap.put(BuildInfoKey.DATA_SOURCE_RELEASE_DATE.name(), releaseDate);

		BuildInfo buildInfo = new BuildInfo(buildInfoMap);
		File buildInfoFile = tempFolder.newFile("build_info.txt");
		buildInfo.save(buildInfoFile.getAbsolutePath());

		BuildCatalog buildCatalog = new BuildCatalog();
		buildCatalog.build(buildInfoFile.getAbsolutePath(), BuildStepKey.MAKE_JSON, BuildStepKey.MAKE_INDEXES);

		List<File> tsvBgzFiles = findAllTsvBgzFiles(targetDir, false);
		assertEquals(1, tsvBgzFiles.size());
		return tsvBgzFiles.get(0);
	}

	/**
	 * Helper to make sub-directory and avoid static code analysis warnings
	 * @param parent parent dir
	 * @param name name of sub-directory
	 * @return New sub-directory
	 */
	public static File mkdir(File parent, String name) {
		File dir = new File(parent, name);
		assertTrue(dir.mkdir());
		return dir;
	}
}