package edu.mayo.bior.cli.func;

import static org.junit.Assert.assertEquals;

import java.io.File;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.net.URISyntaxException;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.HashMap;
import java.util.List;
import java.util.Properties;

import org.apache.commons.configuration.ConfigurationException;
import org.apache.commons.io.FileUtils;
import org.junit.After;
import org.junit.Assert;
import org.junit.Before;
import org.junit.BeforeClass;
import org.junit.Test;
import org.junit.rules.TemporaryFolder;

import edu.mayo.bior.buildcatalog.BuildCatalog;
import edu.mayo.bior.buildcatalog.BuildStepKey;
import edu.mayo.bior.buildcatalog.StepLogger;
import edu.mayo.bior.cli.cmd.CreateCatalogPropsCommand;
import edu.mayo.bior.pipeline.createCatalogProps.ColumnMetaFromCatalogCrawling;
import edu.mayo.cli.InvalidDataException;
import edu.mayo.cli.InvalidOptionArgValueException;
import edu.mayo.pipes.history.ColumnMetaData;
import edu.mayo.pipes.history.ColumnMetaDataOperations;
import edu.mayo.pipes.util.PropertiesFileUtil;


public class CreateCatalogPropsCommandITCase extends BaseFunctionalTest {
	
	private File mTempDir;
	private File mTargetDir;
	
	// Directories containing the individual catalog (.tsv.bgz) for each case
	private File mGeneCtg;
	private File mDeepJsonCtg;
	private File mDeepJsonTsv;
	private File mLongDotsCtg;
	private File mLongDotsVcf;

	private File mExpectedFilesDir;
	private File mExpectedFilesDirWithVcfInfo;
	
	
	private enum Prop {
		columns,
		datasource
	};
	
	@BeforeClass
	public static void beforeAll() throws FileNotFoundException {
		BaseFunctionalTest.setBiorToolkitCmdsRequired(true);
	}
	
	@Before
	public void beforeEach() throws IOException {
		TemporaryFolder tempFolder = new TemporaryFolder();
		tempFolder.create();
		
		mTempDir     = tempFolder.newFolder();
		mTargetDir   = tempFolder.newFolder();
		mExpectedFilesDir = new File("src/test/resources/metadata/createCatalogProps/");
		mExpectedFilesDirWithVcfInfo = new File(mExpectedFilesDir, "testTargetFolder");
		
		File mGeneDir     = tempFolder.newFolder();
		File mDeepJsonDir = tempFolder.newFolder();
		File mLongDotsDir = tempFolder.newFolder();
		
		mGeneCtg 		= new File(mGeneDir, 		"genes.tsv.bgz");
		mDeepJsonCtg 	= new File(mDeepJsonDir, 	"testJsonDepth.tsv.bgz");
		mDeepJsonTsv    = new File(mDeepJsonDir,    "testJsonDepth.tsv");  // Slightly different from the .bgz catalog
		mLongDotsCtg	= new File(mLongDotsDir,	"ALL.wgs.phase1_release_v3.20101123.snps_indels_sv.sites_GRCh37.tsv.bgz");
		mLongDotsVcf 	= new File(mLongDotsDir,	"ALL.wgs.phase1_release_v3.20101123.snps_indels_sv.sites_GRCh37.vcf");
		
		// Copy the catalog from the test dir to a temp dir (so we can build files without fear of existing ones, or overwriting)
		FileUtils.copyFile(new File(mExpectedFilesDir, mGeneCtg.getName()), 	   mGeneCtg);
		FileUtils.copyFile(new File(mExpectedFilesDir, mDeepJsonCtg.getName()), mDeepJsonCtg);
		FileUtils.copyFile(new File(mExpectedFilesDir, mDeepJsonTsv.getName()), mDeepJsonTsv);
		FileUtils.copyFile(new File(mExpectedFilesDir, mLongDotsCtg.getName()), mLongDotsCtg);
		FileUtils.copyFile(new File(mExpectedFilesDir, mLongDotsVcf.getName()), mLongDotsVcf);
	}

	@Test
	public void testCmd() throws IOException, InterruptedException {
        System.out.println("CreateCatalogPropsCommandITCase.testCmd");
	    CommandOutput out = executeScript("bior_create_catalog_props", null, "-d", mGeneCtg.getCanonicalPath());
        assertEquals(out.stderr, 0, out.exit);
        assertEquals("", out.stderr);
        
        // Compare datasource and columns props file
        assertPropertiesSame(mGeneCtg, mExpectedFilesDir, mGeneCtg.getParentFile());
	}

	@Test
	public void testCmd_vcfAndTargetDir() throws IOException, InterruptedException {
        System.out.println("CreateCatalogPropsCommandITCase.testCmd_vcfAndTargetDir");
        // Specify a different taret directory from the one where the catalog file resides
	    CommandOutput out = executeScript("bior_create_catalog_props", null, "-d", mLongDotsCtg.getCanonicalPath(), "-v", mLongDotsVcf.getCanonicalPath(), "-t", mTargetDir.getCanonicalPath());
        assertEquals(out.stderr, 0, out.exit);
        assertEquals("", out.stderr);
        
        // Compare datasource and columns props file
        // NOTE: Use mTargetDir here since we changed the directory.
        // Also, since we specified a VCF for more info (description and name), our expected output file is different
        assertPropertiesSame( mLongDotsCtg, mExpectedFilesDirWithVcfInfo, mTargetDir );
	}
	
	@Test
	public void testNoCmd() throws IOException, InterruptedException, InvalidOptionArgValueException, InvalidDataException, URISyntaxException, ConfigurationException {
        System.out.println("CreateCatalogPropsCommandITCase.testNoCmd");

		CreateCatalogPropsCommand creator = new CreateCatalogPropsCommand();
		creator.setStepLogger(new StepLogger(BuildStepKey.MAKE_PROP_FILES.toString()));
		creator.execNoCmd(mGeneCtg.getCanonicalPath(), /*catalogOriginalVcfPath=*/null, /*targetDirPath=*/null, /*isVcfSpecified=*/false, /*progressHandler=*/null);
        
        // Compare datasource and columns props file
        assertPropertiesSame(mGeneCtg, mExpectedFilesDir, mGeneCtg.getParentFile());
	}

	@Test
	public void testNoCmd_largeRowCounts() throws IOException, InterruptedException, InvalidOptionArgValueException, InvalidDataException, URISyntaxException, ConfigurationException {
        System.out.println("CreateCatalogPropsCommandITCase.testNoCmd");
        
        // Set it to max int value so we'll test the boundary of Integer range when we scan the second line
        ColumnMetaFromCatalogCrawling.NUM_LINES_START = Integer.MAX_VALUE;
        // Set reporting on each line so we can test the Integer.MAX_VALUE limit and ensure we can exceed that without generating negative numbers
        CreateCatalogPropsCommand.PROGRESS_HANDLER_NUM_LINES_CALLBACK_ON = 1;
        
        CommandOutput out = runCmdApp(new CreateCatalogPropsCommand(), "bior_create_catalog_props", "-d", mGeneCtg.getCanonicalPath());
        
        assertEquals(0, out.exit);
        String stderr = removeSlf4jLinesFromStderr(out.stderr);
        assertEquals("", stderr);
        // 21 lines scanned (2,147,483,647 - 21st line =      2147483667
        assertContains(out.stdout,  "# catalog lines crawled: 2147483667");
	}

	
	private String removeSlf4jLinesFromStderr(String stderr) {
		List<String> lines = new ArrayList<String>();
		for(String s : stderr.split("\n")) {
			if( ! s.startsWith("SLF4J: ") )
				lines.add(s);
		}
		return String.join("\n", lines);
	}

	@Test
	public void testNoCmd_LongNameWithDots() throws IOException, InterruptedException, InvalidOptionArgValueException, InvalidDataException, URISyntaxException, ConfigurationException {
        System.out.println("CreateCatalogPropsCommandITCase.testNoCmd_LongNameWithDots");

        CreateCatalogPropsCommand creator = new CreateCatalogPropsCommand();
        creator.setStepLogger(new StepLogger(BuildStepKey.MAKE_PROP_FILES.toString()));
		creator.execNoCmd(mLongDotsCtg.getCanonicalPath(), /*catalogOriginalVcfPath=*/null, /*targetDirPath=*/null, /*isVcfSpecified=*/false, /*progressHandler=*/null);
        
        // Compare datasource and columns props file
        assertPropertiesSame(mLongDotsCtg, mExpectedFilesDir, mLongDotsCtg.getParentFile());
	}

	@Test
	/** Test multiple JSON levels as well as a catalog that has a header line, and a catalog that has no 
	 * This will also test JSONArrays of size 0,1,2 which should all return "." for count,
	 * as well as all matter of other combinations! */
	public void testNoCmd_DeepJson() throws IOException, InterruptedException, InvalidOptionArgValueException, InvalidDataException, URISyntaxException, ConfigurationException {
        System.out.println("CreateCatalogPropsCommandITCase.testNoCmd_DeepJson");

		CreateCatalogPropsCommand creator = new CreateCatalogPropsCommand();
		creator.setStepLogger(new StepLogger(BuildStepKey.MAKE_PROP_FILES.toString()));
		creator.execNoCmd(mDeepJsonTsv.getCanonicalPath(), /*catalogOriginalVcfPath=*/null, /*targetDirPath=*/null, /*isVcfSpecified=*/false, /*progressHandler=*/null);
        
        // Compare datasource and columns props files
        assertPropertiesSame(mDeepJsonTsv, mExpectedFilesDir, mDeepJsonTsv.getParentFile());
	}

	@Test
	public void testNoCmd_vcfAndTargetDir() throws IOException, InterruptedException, InvalidOptionArgValueException, InvalidDataException, URISyntaxException, ConfigurationException {
        System.out.println("CreateCatalogPropsCommandITCase.testNoCmd_vcfAndTargetDir");
		
		CreateCatalogPropsCommand creator = new CreateCatalogPropsCommand();
		creator.setStepLogger(new StepLogger(BuildStepKey.MAKE_PROP_FILES.toString()));
		creator.execNoCmd(mLongDotsCtg.getCanonicalPath(), mLongDotsVcf.getCanonicalPath(), mTargetDir.getCanonicalPath(), /*isVcfSpecified=*/true, /*progressHandler=*/null);
        
        // Compare datasource and columns props file
		// NOTE: We must use mExpectedFilesDirWithVcfInfo here because we gave it a VCF path which 
		//       will fill in the description in columns.tsv and thus the output will be slightly
		//       different from running the command without the VCF file path
		// NOTE: We must use mTargetDir instead of the catalog dir since we gave it a different target path to put the files
        assertPropertiesSame(mLongDotsCtg, mExpectedFilesDirWithVcfInfo, mTargetDir);
	}

	/** Remove the catalog extension ".tsv.bgz" or ".tsv" (in the case of the ASCII text catalog) */
	private String removeCtgExt(File catalog) {
		return catalog.getName().replace(".tsv.bgz", "").replace(".tsv", "");
	}
	


	/** Compare the properties files in the expectedDir (the expected output) and actualTargetDir (the results directory) */
	private void assertPropertiesSame(File catalog, File dirWithExpectedFiles, File dirWithActualFiles) throws IOException {
		// Compare DataSource properties
		String namePrefix = removeCtgExt(catalog);
		String dataSourceActualPath   = new File(dirWithActualFiles,   namePrefix + ".datasource.properties").getCanonicalPath();
		String dataSourceExpectedPath = new File(dirWithExpectedFiles, namePrefix + ".datasource.properties.expected").getCanonicalPath();
        Properties propsActual   = new PropertiesFileUtil(dataSourceActualPath).getProperties();
        Properties propsExpected = new PropertiesFileUtil(dataSourceExpectedPath).getProperties();
        assertPropertiesSame(propsExpected, propsActual);
        
        // Compare column properties
        File columnsActualFile   = new File(dirWithActualFiles,   namePrefix + ".columns.tsv");
        File columnsExpectedFile = new File(dirWithExpectedFiles, namePrefix + ".columns.tsv.expected");
        HashMap<String,ColumnMetaData> colMetaMapActual   = new ColumnMetaDataOperations(columnsActualFile).load();
        HashMap<String,ColumnMetaData> colMetaMapExpected = new ColumnMetaDataOperations(columnsExpectedFile).load();
        assertColsSame(colMetaMapExpected, colMetaMapActual);
        
        // Compare the blacklist and blacklist.biorweb files
        String blacklistActual   = FileUtils.readFileToString(new File(dirWithActualFiles,   namePrefix + ".columns.tsv.blacklist"));
        String blacklistExpected = FileUtils.readFileToString(new File(dirWithExpectedFiles, namePrefix + ".columns.tsv.blacklist.expected"));
        assertEquals(blacklistExpected, blacklistActual);
        
        String blacklistBiorWebActual   = FileUtils.readFileToString(new File(dirWithActualFiles,   namePrefix + ".columns.tsv.blacklist.biorweb"));
        String blacklistBiorWebExpected = FileUtils.readFileToString(new File(dirWithExpectedFiles, namePrefix + ".columns.tsv.blacklist.biorweb.expected"));
        assertEquals(blacklistBiorWebExpected, blacklistBiorWebActual);
}
	
	private void assertColsSame(
			HashMap<String, ColumnMetaData> colMetaMapExpected,
			HashMap<String, ColumnMetaData> colMetaMapActual)
	{
		assertEquals("Size of columns properties files are different", colMetaMapExpected.size(), colMetaMapActual.size());
		System.out.println("Expected keys: " + colMetaMapExpected.keySet());
		System.out.println("Actual   keys: " + colMetaMapActual.keySet());

		for(String key : colMetaMapExpected.keySet()) {
			ColumnMetaData expected = colMetaMapExpected.get(key);
			ColumnMetaData actual   = colMetaMapActual.get(key);
			Assert.assertNotNull("Expected value for key " + key + " was null", expected);
			Assert.assertNotNull("Actual   value for key " + key + " was null", actual);
			assertEquals("Column row does not match:\n  Expected: " + expected.toString() + "\n  Actual:   " + actual.toString(),
					expected.toString(), actual.toString());
		}
	}

	private void assertPropertiesSame(Properties expected, Properties actual) {
		String[] keysExpected = expected.keySet().toArray(new String[0]);
		String[] keysActual   = actual.keySet().toArray(new String[0]);
		assertEquals("Size of properties are not the same", keysExpected.length, keysActual.length);
		
		// Validate that all keys are the same
		for(int i=0; i < keysActual.length; i++) {
			assertEquals("Key[" + i + "] not same (expected=" + keysExpected[i] + ", actual=" + keysActual[i] + ")",
				keysExpected[i], keysActual[i]);
		}
		
		// Validate all values are the same
		for(int i=0; i < keysActual.length; i++) {
			String key = keysActual[i];
			assertEquals("Values[" + i + "] for key [" + key + "] don't match (expected=" 
				+ expected.getProperty(key) + ", actual=" + actual.getProperty(key) + ")",
				expected.getProperty(key), actual.getProperty(key)); 
		}
	}
}
