package edu.mayo.bior.cli.func;

import org.apache.commons.io.FileUtils;
import org.apache.commons.lang.StringUtils;
import org.junit.BeforeClass;
import org.junit.Test;

import edu.mayo.pipes.util.test.PipeTestUtils;

import java.io.File;
import java.io.FileFilter;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;

import static org.hamcrest.CoreMatchers.is;
import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertThat;

public class CmdHelpITCase  extends BaseFunctionalTest {

	private final List<String> EXPECTED_CMDS = Arrays.asList(
			"_bior_catalog_list",
			"_bior_catalog_markdown_with_comparison_stats",
			"_bior_check_java_version",
			"_bior_fasta_to_catalog",
			"_bior_get_tabix_path",
			"_bior_index_stats",
			"_bior_update_columns_tsv",
			"bior_bed_to_tjson",
			"bior_build_catalog",
			"bior_catalog_latest",
			"bior_catalog_markdown",
            "bior_catalog_remove_duplicates",
            "bior_catalog_stats",
			"bior_chunk",
			"bior_compress",
			"bior_concat",
			"bior_count_catalog_misses",
			"bior_create_catalog",
			"bior_create_catalog_props",
			"bior_create_config_for_tab_to_tjson",
			"bior_drill",
			"bior_gbk_to_tjson",
			"bior_gff3_to_tjson",
			"bior_index_catalog",
			"bior_lookup",
			"bior_merge",
			"bior_modify_tjson",
			"bior_overlap",
			"bior_pretty_print",
			"bior_ref_allele",
			"bior_replace_lines",
            "bior_rsid_to_tjson",
			"bior_same_variant",
			"bior_tab_to_tjson",
			"bior_tjson_to_vcf",
			"bior_trim_spaces",
            "bior_variant_to_tjson",
			"bior_vcf_to_tjson",
			"bior_verify_catalog"
	);
	
	private class Flag {
		public String flagName;
		public String longOpt;
		public List<String> lines;
	}
	
	private class Version {
		public int major;
		public int minor;
		public int patch;
	}
	
	
	@BeforeClass
	public static void beforeAll() throws FileNotFoundException {
		BaseFunctionalTest.setBiorToolkitCmdsRequired(true);
	}
	
	@Test
	public void testAllCmdsCovered() throws IOException {
		List<String> actual = getBiorScripts();
        PipeTestUtils.assertListsEqual(EXPECTED_CMDS, actual);
	}




	@Test
	public void testHelp() throws IOException, InterruptedException {
		for( String cmd : EXPECTED_CMDS ) {

            // TODO: FIX ME - these are skipped for now because they do not use the mayo-commons-cli framework
            if (cmd.equals("_bior_sort_catalog")) {
                System.err.println("WARNING - the following command has been skipped: " + cmd);
                continue;
            }

			System.out.print("Cmd: " + cmd + "  :  ");
			
			CommandOutput actual = getOutput(cmd);

			final String EXPECTED_STDOUT = loadHelpFile(cmd);
			
			// bior_annotate and bior_annotate_blaster and bior_remove_duplicates will be deprecated in v5.0.0
			String EXPECTED_STDERR = "";
			// NOTE: bior_annotate and bior_catalog_remove_duplicates will NOT print the warning to STDERR because the -h flag exits before calling the command code.
			if( /*cmd.equals("bior_annotate")  ||*/  cmd.equals("bior_annotate_blaster")  || cmd.equalsIgnoreCase("_bior_annotate_blaster_chunker")  || cmd.equalsIgnoreCase("_bior_annotate_blaster_concat")) {
				EXPECTED_STDERR = getAnnotateDeprecationWarning();
			} /*else if( cmd.equals("bior_catalog_remove_duplicates") ) {
				EXPECTED_STDERR = getRemoveDupesDeprecationWarning();
			}*/
			
			boolean isOk = actual.exit == 0  &&  actual.stdout.equals(EXPECTED_STDOUT)  &&  actual.stderr.equals(EXPECTED_STDERR);

			System.out.println( isOk  ?  "OK"  :  "DIFFERENT");
			
			final String EXIT_CODE_MSG = "  Error: exit code should be 0 when running help on a command.\nCommand: " + cmd + ".\nExit code was " + actual.exit + "\nError msg: " + actual.stderr;
			assertEquals(EXIT_CODE_MSG,	0,  actual.exit);
			
			assertEquals(EXPECTED_STDERR, actual.stderr);
			
			final String ERROR_MSG = "WARNING:  Comparison failed!  Verify that you have run 'mvn clean package' after modifying any properties files.  "
					+    "This is required to get the most up-to-date help output based on the properties files!\n"
					+    "EXPECTED ===========================================================================\n"
					+    EXPECTED_STDOUT + "\n"
					+    "ACTUAL =============================================================================\n"
					+    actual.stdout + "\n"
					+    "====================================================================================\n";
			assertEquals(ERROR_MSG,  EXPECTED_STDOUT, actual.stdout);
		}
	}
	
	
	@Test
	/** Verify that all properties files that have the --database flag have an arg of "CATALOG_PATH":
	 * flag.catalog.file=\
		{\
  			"opt": "d",\
  			"longOpt": "database",\
  			"argName": "CATALOG_PATH",\
  			"description": "The catalog file (bgzip format) containing the JSON data.  Note that a catalog columns.tsv file is required in the same directory.",\
  			"required": true,\
  			"numArgs": 1\
		}
	 */
	public void testAllDatabaseArgsAreCatalogPath() throws IOException {
		List<String> databaseFlagWarnings = new ArrayList<String>();
		File[] biorCmdPropertiesFiles = getCmdPropertiesFiles();
		for(File propsFile : biorCmdPropertiesFiles) {
			List<Flag> allFlagsForFile = getAllFlags(propsFile);
			List<Flag> databaseFlags = filterFlagsByLongOpt(allFlagsForFile, "database");
			for(Flag flag : databaseFlags) {
				String databaseArg = getVal(flag, "argName");
				if( ! "CATALOG_PATH".equals(databaseArg) )
					databaseFlagWarnings.add(propsFile.getPath() + ":  " + databaseArg);
			}
		}
		String errorMsg = "These '--database' flags do NOT have argument of 'CATALOG_PATH':\n" + joinStr(databaseFlagWarnings, "\n") + "\n";
		assertEquals(errorMsg, 0, databaseFlagWarnings.size());
	}
	
	
	@Test
	/** Verify that all properties files that have the --path flag have an arg of "JSON_PATH":
		flag.indexKey=\
		{							\
  			"opt": "p",				\
  			"longOpt": "path",		\
  			"argName": "PATH",    \
  			"description": "The JSON path used in the index to match locations within the catalog where the same JSON path occurs."  \
  			"required": false,			\
  			"numArgs": 1				\
		}	 */
	public void testAllPathArgsArePath() throws IOException {
		final String ARG_NAME = "JSON_PATH";
		final String FLAG     = "path";
		List<String> errorList = new ArrayList<String>();
		File[] biorCmdPropertiesFiles = getCmdPropertiesFiles();
		for(File propsFile : biorCmdPropertiesFiles) {
			List<Flag> allFlagsForFile = getAllFlags(propsFile);
			List<Flag> pathFlags = filterFlagsByLongOpt(allFlagsForFile, FLAG);
			for(Flag flag : pathFlags) {
				String pathArg = getVal(flag, "argName");
				if( ! ARG_NAME.equals(pathArg) )
					errorList.add(propsFile.getPath() + ":  " + pathArg);
			}
		}
		String errorMsg = "These '--" + FLAG + "' flags do NOT have argument of '" + ARG_NAME + "':\n" + joinStr(errorList, "\n") + "\n";
		assertEquals(errorMsg, 0, errorList.size());
	}
	

	
	@Test
	public void testArgNamesShouldBeOnlyCapsAndUnderscores() throws IOException {
		List<String> errorList = new ArrayList<String>();
		File[] biorCmdPropertiesFiles = getCmdPropertiesFiles();
		for(File propsFile : biorCmdPropertiesFiles) {
			List<Flag> allFlagsForFile = getAllFlags(propsFile);
			for(Flag flag : allFlagsForFile) {
				String argVal = getVal(flag, "argName");
				if( ! isArgAllCapsAndDigitsAndUnderscores(argVal) )
					errorList.add(propsFile.getPath() + ":  " + flag.longOpt + " = " + argVal);
			}
		}
		String errorMsg = "These arguments should have ONLY uppercase letters and underscores:\n" + joinStr(errorList, "\n") + "\n";
		assertEquals(errorMsg, 0, errorList.size());
	}
	
	private String joinStr(List<String> strList, String delim) {
		StringBuilder sb = new StringBuilder();
		for(String s : strList) {
			if(sb.length() > 0)
				sb.append(delim);
			sb.append(s);
		}
		return sb.toString();
	}

	@Test
	public void testLongOptionsShouldBeOnlyLettersOrDigits() throws IOException {
		List<String> errorList = new ArrayList<String>();
		
		File[] biorCmdPropertiesFiles = getCmdPropertiesFiles();
		for(File propsFile : biorCmdPropertiesFiles) {
			List<Flag> allFlagsForFile = getAllFlags(propsFile);
			for(Flag flag : allFlagsForFile) {
				String longOptVal = getVal(flag, "longOpt");
				if( ! isOnlyLettersAndDigitsAndFirstLetterIsLowercase(longOptVal) ) {
					String cmdClassAndLongOpt = propsFile.getPath() + ":  " + flag.longOpt;
					if( isTemporaryCamelCaseExemption(propsFile, longOptVal) )
						System.err.println("WARNING:  cmd flag longOpt is not standard.  It should be camelCase (ex: 'numberOfValues').  This should be fixed in BioR v6.0.0: " + cmdClassAndLongOpt);
					else
						errorList.add(cmdClassAndLongOpt);
				}
			}
		}
		String errorMsg = "These long options should have ONLY letters and numbers (no hyphens, etc):\n" + joinStr(errorList, "\n") + "\n";
		assertEquals(errorMsg, 0, errorList.size());
	}




	@Test
	public void testGetVal() {
		assertEquals("Description: one (val) in \"two\", or 3.",  getVal("   \"description\": \"Description: one (val) in \"two\", or 3.\",\\"));
		assertEquals("true", getVal("   \"required\": true,\\"));
		assertEquals("1",    getVal("   \"numArgs\": 1\\"));
		assertEquals("CONFIG FILE", getVal("  \"argName\": \"CONFIG FILE\", \\"));
	}
	
	@Test
	public void testGetKey() {
		assertEquals("description",  getKey("   \"description\": \"Description: one (val) in \"two\", or 3.\",\\"));
		assertEquals("required",     getKey("   \"required\": true,\\"));
		assertEquals("numArgs",      getKey("   \"numArgs\": 1\\"));
		assertEquals("argName",      getKey("   \"argName\": \"CONFIG FILE\", \\"));
	}


	//-------------------------------------------------------------------------------------

	/** If this is an exemption from the longOpt camel-casing rule
	 *  (at least until bior v6.0.0 when all of those will be corrected), return true 
	 * @throws IOException */
	private boolean isTemporaryCamelCaseExemption(File propsFile, String longOptVal) throws IOException {
		boolean isBeforeV6 = isBeforeBiorVersion6();
		String filenameLongOpt = propsFile.getPath() + ":  " + longOptVal;
		List<String> exemptionList = getCamelCaseExemptions();
		return isBeforeV6 && exemptionList.contains(filenameLongOpt);
	}


	private boolean isBeforeBiorVersion6() throws IOException {
		Version version = getBiorVersionFromPomXml();
		return version.major < 6;
	}


	/** Should see bior version from pom.xml within first 20 lines
		<groupId>edu.mayo.bior</groupId>
		<artifactId>bior-toolkit</artifactId>
		<version>4.4.0-SNAPSHOT</version>
		<name>bior-toolkit</name>
	 * @throws IOException 
	*/
	private Version getBiorVersionFromPomXml() throws IOException {
		List<String> lines = FileUtils.readLines(new File("pom.xml"));
		String versionStr = "";
		for(int i=0; i < 20; i++) {
			String line = lines.get(i);
			if( line.contains("<version>")  && line.contains("</version>") )
				versionStr = line.trim().replace("<version>", "").replace("</version>", "").replace("-SNAPSHOT", "");
		}
		return getVersionFromStr(versionStr);
	}

	/** Convert "1.0.2" to Version object */
	private Version getVersionFromStr(String versionStr) {
		int idx1 = versionStr.indexOf(".");
		int idx2 = versionStr.indexOf(".", idx1+1);
		if( idx1 == -1  ||  idx2 == -1 )
			throw new IllegalArgumentException("Version is not valid: " + versionStr);
		
		Version ver = new Version();
		ver.major = Integer.parseInt(versionStr.substring(0, idx1));
		ver.minor = Integer.parseInt(versionStr.substring(idx1+1, idx2));
		ver.patch = Integer.parseInt(versionStr.substring(idx2+1));
		return ver;
	}

	private List<String> getCamelCaseExemptions() {
		String dir = "src/main/resources/cli/";
		return Arrays.asList(
				dir + "CatalogStatsCommand.properties:  number-of-values",
				dir + "CatalogStatsCommand.properties:  output-dir",
				dir + "DrillCommand.properties:  keep-json",
				dir + "DrillCommand.properties:  array-delimiter",
				dir + "DrillCommand.properties:  skip-nulls",
				dir + "ModifyTjsonCommand.properties:  config-file",
				dir + "PrettyPrintCommand.properties:  row-number",
				dir + "VCF2VariantCommand.properties:  add-flag-falses"
				);
	}

	private boolean isArgAllCapsAndDigitsAndUnderscores(String argVal) {
		for(int i=0; i < argVal.length(); i++) {
			char c = argVal.charAt(i);
			boolean isUppercaseDigitUnderscore = Character.isUpperCase(c)  ||  Character.isDigit(c)  ||  c == '_';
			if( ! isUppercaseDigitUnderscore )
				return false;
		}
		return true;
	}

	private boolean isOnlyLettersAndDigitsAndFirstLetterIsLowercase(String s) {
		if( s == null  ||  s.trim().length() == 0 )
			return true;
		
		if( ! Character.isLowerCase(s.charAt(0)) )
			return false;
		
		for(int i=0; i < s.length(); i++) {
			char c = s.charAt(i);
			if( ! Character.isLetterOrDigit(c) )
				return false;
		}
		return true;
	}


	private String getVal(Flag flag, String key) {
		for(String line : flag.lines) {
			String key2 = getKey(line);
			if( key.equals(key2) )
				return getVal(line);
		}
		// Not found, return ""
		return "";
	}


	private List<Flag> filterFlagsByLongOpt(List<Flag> allFlags, String longOptVal) {
		List<Flag> flagsFiltered = new ArrayList<Flag>();
		for(Flag flag : allFlags) {
			for(String line : flag.lines) {
				String val = getVal(line);
				if( val.equals(longOptVal) )
					flagsFiltered.add(flag);
			}
		}
		return flagsFiltered;
	}


	private String getKey(String line) {
		String key = "";
		int idxColon = line.indexOf(":");
		if( idxColon != -1 ) {
			key = line.substring(0, idxColon).trim();
			if( key.startsWith("\"") && key.endsWith("\"") )
				key = key.substring(1, key.length()-1).trim();
		}
		return key;
	}




	private List<Flag> getAllFlags(File propsFile) throws IOException {
		List<Flag> flags = new ArrayList<Flag>();
		List<String> lines = FileUtils.readLines(propsFile);
		boolean isInFlag = false;
		Flag flag = new Flag();
		for(String line : lines) {
			if( line.trim().startsWith("flag.") ) {
				isInFlag = true;
				flag = new Flag();
				flag.flagName = line;
				flag.lines = new ArrayList<String>();
				flags.add(flag);
			}
			
			if( isInFlag ) {
				flag.lines.add(line);
				if( line.contains("longOpt") )
					flag.longOpt = getLongOpt(line);
				if( line.trim().equals("}") )
					isInFlag = false;
			}
		}
		return flags;
	}

	/** From:
	 * 		"longOpt": "database",\
	 *  Get:
	 *  	database
	 * @param line
	 * @return
	 */
	private String getLongOpt(String line) {
		String opt = getVal(line);
		return opt;
	}

	/** From:
	 * 		"description": "The catalog file (bgzip format) containing the JSON data.  Note that a catalog columns.tsv file is required in the same directory.",\
	 *  Get
	 *  	The catalog file (bgzip format) containing the JSON data.  Note that a catalog columns.tsv file is required in the same directory.
	 *  Or, from:
	 *  	"required": true,\
	 *  Get
	 *  	true
	 *  Or, from:
  			"numArgs": 1\
  		Get
  			1
	 * @param line
	 * @return
	 */
	private String getVal(String line) {
		String val = "";
		int idxColon = line.indexOf(":");
		if( idxColon != -1  && idxColon < line.length()-1 ) {
			val = line.substring(idxColon+1).trim();
			if( val.endsWith("\\") )
				val = val.substring(0, val.length()-1).trim();
			if( val.endsWith(",") )
				val = val.substring(0, val.length()-1).trim();
			if( val.startsWith("\"") && val.endsWith("\"") )
				val = val.substring(1, val.length()-1);
		}
		return val;
	}




	private File[] getCmdPropertiesFiles() {
		File[] propsFiles =  new File("src/main/resources/cli").listFiles(new FileFilter() {
			public boolean accept(File pathname) {
				return pathname.getName().endsWith(".properties");
			}
		});
		return propsFiles;
	}

	private List<String> getBiorScripts() throws FileNotFoundException {
		// dir = target/test-classes
		String dir = this.getClass().getResource("/").getPath();
		// We need:  target/bior-toolkit-xxxxxxx/bin/
		File parentDir = new File(dir).getParentFile();
		File biorPipelineDir = getBiorToolkitDistributionDir(parentDir);
		
		File scriptsDir = new File(biorPipelineDir, "bin");
		if( ! scriptsDir.exists() ) {
			throw new FileNotFoundException("BioR distribution bin dir containing the BioR scripts does not exist: " + scriptsDir.getAbsolutePath());
		}
		File[] scripts = scriptsDir.listFiles();
		List<String> biorScripts = removeTabixCmdsAndGroovyScripts(scripts);
		Collections.sort(biorScripts);
		return biorScripts;
	}


	private File getBiorToolkitDistributionDir(File parentDir) {
		File biorPipelineDir = null;
		for(File f : parentDir.listFiles()) {
			if( f.isDirectory() && f.getName().startsWith("bior-toolkit-") && !f.getName().endsWith("distribution") )
				biorPipelineDir = f;
		}
		return biorPipelineDir;
	}

	/** Remove from the list any files beginning with "tabix" as those are not bior-provided scripts*/
	private List<String> removeTabixCmdsAndGroovyScripts(File[] scriptsInBinDir) {
        List<String> biorScripts = new ArrayList<String>();
        for (File script: scriptsInBinDir) {
        	if( ! script.getName().startsWith("tabix")  &&  ! script.getName().endsWith(".groovy"))
        		biorScripts.add(script.getName());
        }
        return biorScripts;
	}
	private String getAnnotateDeprecationWarning() {
		return "----------------------------------------------------------\n"
			+  "WARNING:  THIS COMMAND WILL BE REMOVED IN BIOR v5.0.0!  Use the more flexible bior_annotate.sh script as a replacement\n"
			+  "    https://github.com/Steven-N-Hart/bior_annotate\n"
			+  "More info at:\n"
			+  "    http://bsiweb.mayo.edu/bior_annotate\n"
			+  "You can continue to use older versions of BioR to run this command\n"
			+  "----------------------------------------------------------\n";
	}
	
	private String getRemoveDupesDeprecationWarning() {
		return "WARNING:  THIS COMMAND WILL BE REMOVED IN BIOR v5.0.0!\n"
			+  "You can continue to use older versions of BioR to run this command\n";
	}

	private String loadHelpFile(String cmd) throws IOException {
		File cmdHelpFile = new File(this.getClass().getResource("/help").getFile(),  cmd + ".help");
		return FileUtils.readFileToString(cmdHelpFile);
	}

	private CommandOutput getOutput(String cmd) throws IOException, InterruptedException {
		CommandOutput out = executeScript(cmd, "", "-h");
		return out;
	}
}
