package edu.mayo.bior.cli.cmd;

import static org.junit.Assert.assertEquals;
import static org.junit.Assert.fail;

import java.io.File;
import java.io.IOException;
import java.io.UnsupportedEncodingException;
import java.util.Arrays;
import java.util.List;

import org.apache.commons.io.FileUtils;
import org.apache.commons.lang.StringUtils;
import org.junit.After;
import org.junit.Before;
import org.junit.Ignore;
import org.junit.Test;

import com.tinkerpop.pipes.util.Pipeline;

import edu.mayo.bior.cli.func.BaseFunctionalTest;
import edu.mayo.bior.cli.func.CommandOutput;
import edu.mayo.cli.CommandLineApp;
import edu.mayo.cli.CommandPlugin;
import edu.mayo.cli.InvalidDataException;
import edu.mayo.pipes.JSON.InjectIntoJsonPipe;
import edu.mayo.pipes.JSON.inject.Injector;
import edu.mayo.pipes.UNIX.CatPipe;
import edu.mayo.pipes.history.HistoryInPipe;
import edu.mayo.pipes.history.HistoryOutPipe;
import edu.mayo.pipes.util.metadata.Metadata;
import edu.mayo.pipes.util.test.PipeTestUtils;
import junit.framework.Assert;
/**
 * Test the bior_tab_to_tjson command
 * Created with IntelliJ IDEA.
 * User: m102417
 * Date: 7/24/13
 * Time: 2:22 PM
 * To change this template use File | Settings | File Templates.
 */
public class Tab2JSONCommandTest extends BaseFunctionalTest {
	
    protected static final String MOCK_SCRIPT_NAME = "bior_tab_to_tjson";
    protected CommandLineApp mApp = new CommandLineApp();
    protected CommandPlugin mMockPlugin;

	
	// NOTE: Make sure to reset the history as the static variable can mess with subsequent tests!!!
    @Before
    public void clearStaticHistoryBefore(){
    }

    @After
    public void clearStaticHistoryAfter(){
    }
	
    @Test
    public void testParseConfigFile(){
        Tab2JSONCommand t = new Tab2JSONCommand();
	    Injector[] injectors = null;
	    try {
	    	injectors = t.parseConfigFile("src/test/resources/Tab2JSONFiles/example.config");
	    } catch (InvalidDataException e) {
	    	Assert.fail("Unexpected Exception occurred");
	    }
        assertEquals(2, injectors.length);
    }

    @Test
    public void testParseConfigFileWithAltAlleles(){
        Tab2JSONCommand t = new Tab2JSONCommand();
	    Injector[] injectors = null;
	    try {
	    	injectors = t.parseConfigFile("src/test/resources/Tab2JSONFiles/example2.config");
	    } catch (InvalidDataException e) {
	    	Assert.fail("Unexpected Exception occurred");
	    }
        assertEquals(11, injectors.length);
    }
    
    @Test
    public void testConvertTab2JSON() throws IOException, InvalidDataException{
    	String input = FileUtils.readFileToString(new File("src/test/resources/Tab2JSONFiles/example.tsv"));
    	String config= FileUtils.readFileToString(new File("src/test/resources/Tab2JSONFiles/example.config"));
    	List<String> actual = getOutput(input, config);
        
        final List<String> expected = Arrays.asList(
        		"##BIOR=<ID=\"bior.ToTJson\",Operation=\"bior_tab_to_tjson\",DataType=\"JSON\",ShortUniqueName=\"ToTJson\">",
                concat("#gene", "rsID", "bior.ToTJson"),
                concat("MRPL39", "rs142513484", "{\"gene\":\"MRPL39\",\"rsID\":\"rs142513484\"}"),
                concat("MRPL39", "rs200187839", "{\"gene\":\"MRPL39\",\"rsID\":\"rs200187839\"}"),
                concat("PANX2", "rs191258266",  "{\"gene\":\"PANX2\",\"rsID\":\"rs191258266\"}"),
                concat("PANX2", "rs145036669",  "{\"gene\":\"PANX2\",\"rsID\":\"rs145036669\"}"),
                concat("BRCA1", "rs184237074",  "{\"gene\":\"BRCA1\",\"rsID\":\"rs184237074\"}"),
                concat("BRCA1", "rs189382442",  "{\"gene\":\"BRCA1\",\"rsID\":\"rs189382442\"}"),
                concat("BRCA1", "rs182218567",  "{\"gene\":\"BRCA1\",\"rsID\":\"rs182218567\"}")
        );
        PipeTestUtils.assertListsEqual(expected, actual);
    }

    @Test
    public void testConvertTab2JSON_withAltAlleles() throws IOException, InvalidDataException{
    	String input = FileUtils.readFileToString(new File("src/test/resources/Tab2JSONFiles/example2.tsv"));
    	String config= FileUtils.readFileToString(new File("src/test/resources/Tab2JSONFiles/example2.config"));
    	List<String> actual = getOutput(input, config);

        final List<String> expected = Arrays.asList(
        		"##BIOR=<ID=\"bior.ToTJson\",Operation=\"bior_tab_to_tjson\",DataType=\"JSON\",ShortUniqueName=\"ToTJson\">",
        		concat("#UNKNOWN_1", "#UNKNOWN_2", "#UNKNOWN_3", "#UNKNOWN_4", "#UNKNOWN_5", "#UNKNOWN_6", "bior.ToTJson"),
                concat("rs1111", "13", "11111", "gene111", "T", "A,C", swapQuotes("{'chr':'13','pos':11111,'gene':'gene111','ref':'T','alts':['A','C'],'rsid':'rs1111','_landmark':'13','_minBP':11111,'_refAllele':'T','_altAlleles':['A','C'],'_id':'rs1111'}")),
                concat("rs2222", "14", "22222", "gene222", "G", "A,C", swapQuotes("{'chr':'14','pos':22222,'gene':'gene222','ref':'G','alts':['A','C'],'rsid':'rs2222','_landmark':'14','_minBP':22222,'_refAllele':'G','_altAlleles':['A','C'],'_id':'rs2222'}")),
                concat("rs3333", "15", "33333", "gene333", "C", "G",   swapQuotes("{'chr':'15','pos':33333,'gene':'gene333','ref':'C','alts':['G'],'rsid':'rs3333','_landmark':'15','_minBP':33333,'_refAllele':'C','_altAlleles':['G'],'_id':'rs3333'}"))
        );

        PipeTestUtils.assertListsEqual(expected, actual);
    }
    
    @Test
    @Ignore  // Ignore for now, but if we want to move the computation of _maxBP to bior_tab_to_tjson, then we want to put this back in
    public void testComputeMaxBp() throws InvalidDataException, IOException {
    	String input =
    			"#CHROM	POS	N_ALLELES	N_CHR	Ref	Ref_AF	Alt	Alt_AF" + "\n" +
    			"1	65872	2	118	T	0.288136	G	0.711864" + "\n" +
    			"1	65893	2	4	GG	0.25	A	0.75" + "\n";
    	String config = 
    			"1	CHROM	STRING	COLUMN	.	_landmark" + "\n" +
    			"2	POS	NUMBER	COLUMN	.	_minBP" + "\n" +
    			"3	N_ALLELES	NUMBER	COLUMN	.	." + "\n" +
    			"4	N_CHR	NUMBER	COLUMN	.	." + "\n" +
    			"5	Ref	STRING	COLUMN	.	_refAllele" + "\n" +
    			"6	Ref_AF	NUMBER	COLUMN	.	." + "\n" +
    			"7	Alt	STRING	COLUMN	.	_altAlleles" + "\n" +
    			"7	_altAlleles	STRING	ARRAY	,	_altAlleles" + "\n" +
    			"8	Alt_AF	NUMBER	COLUMN	.	.";
    	
    	List<String> actual = getOutput(input, config);

        final List<String> expected = Arrays.asList(
        		"##BIOR=<ID=\"bior.ToTJson\",Operation=\"bior_tab_to_tjson\",DataType=\"JSON\",ShortUniqueName=\"ToTJson\">",
        		"#CHROM	POS	N_ALLELES	N_CHR	Ref	Ref_AF	Alt	Alt_AF	bior.ToTJson",
        		("1	65872	2	118	T	0.288136	G	0.711864	{'CHROM':'1','POS':65872,'N_ALLELES':2,'N_CHR':118,'Ref':'T','Ref_AF':0.288136,'Alt':'G','_altAlleles':['G'],'Alt_AF':0.711864,'_landmark':'1','_minBP':65872,'_refAllele':'T','_maxBP':65872}".replaceAll("'", "\"")),
        		("1	65893	2	4	G	0.25	A	0.75	{'CHROM':'1','POS':65893,'N_ALLELES':2,'N_CHR':4,'Ref':'G','Ref_AF':0.25,'Alt':'A','_altAlleles':['A'],'Alt_AF':0.75,'_landmark':'1','_minBP':65893,'_refAllele':'GG','_maxBP':65894}".replaceAll("'", "\""))
        );

        PipeTestUtils.assertListsEqual(expected, actual);

    }
    
	@Test
	public void testConvertTab2JSON_DupData_AndAltAlleleDelimTest() throws IOException, InvalidDataException{
    	String input = FileUtils.readFileToString(new File("src/test/resources/Tab2JSONFiles/example3.tsv"));
    	String config= FileUtils.readFileToString(new File("src/test/resources/Tab2JSONFiles/example3.config"));
    	List<String> actual = getOutput(input, config);

	    final String expected_substring1 = swapQuotes("'_altAlleles':['G','T']");   	    
	    final String expected_substring2 = swapQuotes("'MyData_JSON_F1':'ABCD','MyData_JSON_F2':'ABCD'");   
	    final String expected_second_cat_output = swapQuotes("13	4595959	G	C	4595959	variant	xyz	rs838383	{'CHROM':'13','POS':4595959,'REF':'G','ALT':['C'],'POS2':4595959,'TypeVar':'variant','MyData_JSON_F1':'xyz','MyData_JSON_F2':'xyz','rsid':'rs838383','_landmark':'13','_minBP':4595959,'_refAllele':'G','_altAlleles':['C'],'_maxBP':4595959,'_type':'variant','_id':'rs838383'}");
	    
	    Assert.assertTrue(actual.size() == 4);
	    Assert.assertTrue(actual.get(2).contains(expected_substring1));
	    Assert.assertTrue(actual.get(2).contains(expected_substring2));
	    Assert.assertTrue(actual.get(3).equals(expected_second_cat_output));
	}
	
	
	@Test
	/** Simple example with a column used twice (for _minBP and _maxBP) */
	public void columnUsedTwiceForTwoGoldenAttributes() throws InvalidDataException, IOException{
		String tsvIn = 
			concat("#CHROM", "POS",      "REF", "ALT") + "\n" +
			concat("chr1",   "14779",    "C",   "G")   + "\n" +
			concat("chr1",   "134989",   "A",   "T")   + "\n" +
			concat("chr1",   "135099",   "A",   "C")   + "\n" +
			concat("chr1",   "135100",   "G",   "T")   + "\n" +
			concat("chr1",   "135132",   "A",   "T")   + "\n" +
			concat("chr1",   "136603",   "A",   "G")   + "\n" +
			concat("chr1",   "136768",   "TTT", "C,AA,GGG")   + "\n" +
			concat("chr1",   "136962",   "C",   "T")   + "\n" +
			concat("chr1",   "137713",   "T",   "G");
		
		String config = 
			//     col  name     type      injector  delim goldenAttr
			concat("1", "CHROM", "STRING", "COLUMN", ".",  "_landmark")   + "\n" +
			concat("2", "POS",   "INTEGER","COLUMN", ".",  "_minBP")      + "\n" +
			concat("2", "POS",   "INTEGER","COLUMN", ".",  "_maxBP")      + "\n" +  // NOTE: Repeat use of column 2!
			concat("3", "REF",   "STRING", "COLUMN", ".",  "_refAllele")  + "\n" +
			concat("4", "ALT",   "STRING", "ARRAY",  ",",  "_altAlleles");
		
		List<String> actual = getOutput(tsvIn, config);
		
		List<String> expected = Arrays.asList(
			"##BIOR=<ID=\"bior.ToTJson\",Operation=\"bior_tab_to_tjson\",DataType=\"JSON\",ShortUniqueName=\"ToTJson\">",
			concat("#CHROM", "POS",      "REF", "ALT",	"bior.ToTJson"),
			concat("chr1",   "14779",    "C",   "G",	swapQuotes("{'CHROM':'chr1','POS':14779,'REF':'C','ALT':['G'],'_landmark':'chr1','_minBP':14779,'_maxBP':14779,'_refAllele':'C','_altAlleles':['G']}")),
			concat("chr1",   "134989",   "A",   "T",	swapQuotes("{'CHROM':'chr1','POS':134989,'REF':'A','ALT':['T'],'_landmark':'chr1','_minBP':134989,'_maxBP':134989,'_refAllele':'A','_altAlleles':['T']}")),
			concat("chr1",   "135099",   "A",   "C",	swapQuotes("{'CHROM':'chr1','POS':135099,'REF':'A','ALT':['C'],'_landmark':'chr1','_minBP':135099,'_maxBP':135099,'_refAllele':'A','_altAlleles':['C']}")),
			concat("chr1",   "135100",   "G",   "T",	swapQuotes("{'CHROM':'chr1','POS':135100,'REF':'G','ALT':['T'],'_landmark':'chr1','_minBP':135100,'_maxBP':135100,'_refAllele':'G','_altAlleles':['T']}")),
			concat("chr1",   "135132",   "A",   "T",	swapQuotes("{'CHROM':'chr1','POS':135132,'REF':'A','ALT':['T'],'_landmark':'chr1','_minBP':135132,'_maxBP':135132,'_refAllele':'A','_altAlleles':['T']}")),
			concat("chr1",   "136603",   "A",   "G",	swapQuotes("{'CHROM':'chr1','POS':136603,'REF':'A','ALT':['G'],'_landmark':'chr1','_minBP':136603,'_maxBP':136603,'_refAllele':'A','_altAlleles':['G']}")),
			concat("chr1",   "136768",   "TTT", "C,AA,GGG",	swapQuotes("{'CHROM':'chr1','POS':136768,'REF':'TTT','ALT':['C','AA','GGG'],'_landmark':'chr1','_minBP':136768,'_maxBP':136768,'_refAllele':'TTT','_altAlleles':['C','AA','GGG']}")),
			concat("chr1",   "136962",   "C",   "T",	swapQuotes("{'CHROM':'chr1','POS':136962,'REF':'C','ALT':['T'],'_landmark':'chr1','_minBP':136962,'_maxBP':136962,'_refAllele':'C','_altAlleles':['T']}")),
			concat("chr1",   "137713",   "T",   "G",	swapQuotes("{'CHROM':'chr1','POS':137713,'REF':'T','ALT':['G'],'_landmark':'chr1','_minBP':137713,'_maxBP':137713,'_refAllele':'T','_altAlleles':['G']}"))
			);

		PipeTestUtils.assertListsEqual(expected, actual);

		// Now try it with the mock
		File configFile = File.createTempFile("config", "tsv");
		FileUtils.writeStringToFile(configFile, config);
		CommandOutput out = runCmdApp(tsvIn, new Tab2JSONCommand(), "bior_tab_to_tjson", "-c", configFile.getCanonicalPath());
		assertEquals(out.stderr, 0, out.exit);
		assertEquals(out.stderr, "", out.stderr);
		PipeTestUtils.assertListsEqual(expected, Arrays.asList(out.stdout.split("\n")));

	}
	
	
	@Test
	/** Splitting by pipe and dot should work ok */
	public void splitArrayValues_pipeAndDot() throws InvalidDataException, IOException{
		String tsvIn = 
			concat("#AltPipe",  "AltDot",  "AltComma", "AltSemicolon", "AltColon") + "\n" +
			concat("A|B||C||",  ".1..2..", "Z,Y,,X,,", "L;M;;N;;",     "Q:R::S::") + "\n" +
			concat(".",         ".",       ".",        ".",            ".")   +  "\n" +
			concat(",",         ".",       ";",        ",",            ",")   + "\n" +
			concat("A",         "1",       "Z",        "L",            "Q")   + "\n" +
			concat("",          "",        "",         "",             "")    + "\n" ;
			
		
		String config = 
			//     col  name           type      injector  delim goldenAttr
			concat("1", "AltPipe",     "STRING", "ARRAY",  "|",  ".")  + "\n" +
			concat("2", "AltDot",      "NUMBER", "ARRAY",  ".",  ".")  + "\n" +
			concat("3", "AltComma",    "STRING", "ARRAY",  ",",  ".")  + "\n" +
			concat("4", "AltSemicolon","STRING", "ARRAY",  ";",  ".")  + "\n" +
			concat("5", "AltColon",    "STRING", "ARRAY",  ":",  ".");
		
		List<String> actual = getOutput(tsvIn, config);
		
		List<String> expected = Arrays.asList(
			"##BIOR=<ID=\"bior.ToTJson\",Operation=\"bior_tab_to_tjson\",DataType=\"JSON\",ShortUniqueName=\"ToTJson\">",
			concat("#AltPipe",  "AltDot",  "AltComma", "AltSemicolon", "AltColon", "bior.ToTJson"),
			concat("A|B||C||",  ".1..2..", "Z,Y,,X,,", "L;M;;N;;",    "Q:R::S::",  swapQuotes("{'AltPipe':['A','B','','C','',''],'AltDot':[,1,,2,,,],'AltComma':['Z','Y','','X','',''],'AltSemicolon':['L','M','','N','',''],'AltColon':['Q','R','','S','','']}")),
			concat(".",         ".",       ".",        ".",	           ".",        swapQuotes("{'AltPipe':[],'AltDot':[],'AltComma':[],'AltSemicolon':[],'AltColon':[]}")),
			concat(",",         ".",       ";",        ",",	           ",",        swapQuotes("{'AltPipe':[','],'AltDot':[],'AltComma':[';'],'AltSemicolon':[','],'AltColon':[',']}")),
			concat("A",         "1",       "Z",        "L",	           "Q",        swapQuotes("{'AltPipe':['A'],'AltDot':[1],'AltComma':['Z'],'AltSemicolon':['L'],'AltColon':['Q']}")),
			concat("",          "",        "",         "",	           "",         swapQuotes("{'AltPipe':[],'AltDot':[],'AltComma':[],'AltSemicolon':[],'AltColon':[]}"))
			);

		PipeTestUtils.assertListsEqual(expected, actual);

		// Now try it with the mock
		File configFile = File.createTempFile("config", "tsv");
		FileUtils.writeStringToFile(configFile, config);
		CommandOutput out = runCmdApp(tsvIn, new Tab2JSONCommand(), "bior_tab_to_tjson", "-c", configFile.getCanonicalPath());
		assertEquals(out.stderr, 0, out.exit);
		assertEquals(out.stderr, "", out.stderr);
		PipeTestUtils.assertListsEqual(expected, Arrays.asList(out.stdout.split("\n")));

	}
	
	@Ignore
	@Test // TODO: Need to account for this case yet
	/** Should throw an error (NOT give blank output) when lines separated by spaces */
	public void shouldThrowException_NotBlankOutput_whenInputContainsSpaces()  throws InvalidDataException, IOException{
		String tsvIn = 
			"#CHROM  POS    REF  ALT\n" +  // NOTE: Header contains spaces, which is ok, since this is ignored
			"chr1  14779  C  G\n"   +      // Should ERROR: Data row has spaces instead of tabs
			concat("chr1", "134989","A", "T");
		
		String config = 
			//     col  name     type      injector  delim goldenAttr
			concat("1", "CHROM", "STRING", "COLUMN", ".",  "_landmark")   + "\n" +
			concat("2", "POS",   "NUMBER", "COLUMN", ".",  "_minBP")      + "\n" +
			concat("3", "REF",   "STRING", "COLUMN", ".",  "_refAllele")  + "\n" +
			concat("4", "ALT",   "STRING", "ARRAY",  ",",  "_altAlleles");
		
		List<String> actual = getOutput(tsvIn, config);
		
		// Now try it with the mock
		File configFile = File.createTempFile("config", "tsv");
		FileUtils.writeStringToFile(configFile, config);
		CommandOutput out = runCmdApp(tsvIn, new Tab2JSONCommand(), "-c", configFile.getCanonicalPath());
		assertEquals(out.stderr, 0, out.exit);
		assertEquals(out.stderr, "(TODO: should be an error here!  Something about column is missing)", out.stderr);
		assertEquals("", out.stdout);
	}
	
	@Ignore
	@Test
	public void testConvertTab2JSON_IdentifySpaceSepData() throws IOException, InvalidDataException {
    	String input = FileUtils.readFileToString(new File("src/test/resources/Tab2JSONFiles/example6.tsv")); // multiple space-sep instead of tab-sep as should be
    	String config= FileUtils.readFileToString(new File("src/test/resources/Tab2JSONFiles/example5.config")); // its ok if we use the example5 config file
    	List<String> actual = getOutput(input, config);

		final List<String> expected = Arrays.asList(
				"##BIOR=<ID=\"bior.ToTJson\",Operation=\"bior_tab_to_tjson\",DataType=\"JSON\",ShortUniqueName=\"ToTJson\">",
				concat("#CHROM", "POS",   "REF", "ALT", "POS2",    "TypeVar",  "MyData", "rsid",  "bior.ToTJson"),
				concat("2",  "169870836", "A", "G/T", "169870836", "variant", "ABCD", "rs1234",   swapQuotes("{'CHROM':'2','POS':169870836,'REF':'A','ALT':['G','T'],'POS2':169870836,'TypeVar':'variant','MyData_JSON_F1':'ABCD','MyData_JSON_F2':'ABCD','rsid':'rs1234','_landmark':'2','_minBP':169870836,'_refAllele':'A','_altAlleles':['G','T'],'_maxBP':169870836,'_type':'variant','_id':'rs1234'}")),
				concat("13", "4595959",   "G", "C",   "4595959",   "variant", "xyz",  "rs838383", swapQuotes("{'CHROM':'13','POS':4595959,'REF':'G','ALT':['C'],'POS2':4595959,'TypeVar':'variant','MyData_JSON_F1':'xyz','MyData_JSON_F2':'xyz','rsid':'rs838383','_landmark':'13','_minBP':4595959,'_refAllele':'G','_altAlleles':['C'],'_maxBP':4595959,'_type':'variant','_id':'rs838383'}"))
		);

		PipeTestUtils.assertListsEqual(expected, actual);
	}
	
	@Test
	public void testConvertTab2JSON_InvalidGolden(){
	    Tab2JSONCommand t = new Tab2JSONCommand();
	    Metadata md = new Metadata("bior_tab_to_tjson");

	    Injector[] injectors = null;
	    try {
	    	injectors = t.parseConfigFile("src/test/resources/Tab2JSONFiles/example4.config");
	    	Assert.fail(); // shouldn't get here.
	    } catch (InvalidDataException e) {
	    	//System.out.println("Successfully reached the exception block for example4.config. " + e.getMessage());
	    	Assert.assertNotNull(e);
	    }
	}
	
	@Ignore
	@Test
	public void testConvertTab2JSON_HandleEmptyLinesInConfigAndTSV() throws IOException, InvalidDataException {
    	String input = FileUtils.readFileToString(new File("src/test/resources/Tab2JSONFiles/example5.tsv"));
    	String config= FileUtils.readFileToString(new File("src/test/resources/Tab2JSONFiles/example5.config"));
    	List<String> actual = getOutput(input, config);

		final List<String> expected = Arrays.asList(
				"##BIOR=<ID=\"bior.ToTJson\",Operation=\"bior_tab_to_tjson\",DataType=\"JSON\",ShortUniqueName=\"ToTJson\">",
				concat("#CHROM", "POS",   "REF", "ALT", "POS2",    "TypeVar",  "MyData", "rsid",  "bior.ToTJson"),
				concat("2",  "169870836", "A", "G/T", "169870836", "variant", "ABCD", "rs1234",   swapQuotes("{'CHROM':'2','POS':169870836,'REF':'A','ALT':['G','T'],'POS2':169870836,'TypeVar':'variant','MyData_JSON_F1':'ABCD','MyData_JSON_F2':'ABCD','rsid':'rs1234','_landmark':'2','_minBP':169870836,'_refAllele':'A','_altAlleles':['G','T'],'_maxBP':169870836,'_type':'variant','_id':'rs1234'}")),
				concat("13", "4595959",   "G", "C",   "4595959",   "variant", "xyz",  "rs838383", swapQuotes("{'CHROM':'13','POS':4595959,'REF':'G','ALT':['C'],'POS2':4595959,'TypeVar':'variant','MyData_JSON_F1':'xyz','MyData_JSON_F2':'xyz','rsid':'rs838383','_landmark':'13','_minBP':4595959,'_refAllele':'G','_altAlleles':['C'],'_maxBP':4595959,'_type':'variant','_id':'rs838383'}"))
		);

		PipeTestUtils.assertListsEqual(expected, actual);
	}

	

	@Test
	/** Test 0 and 1's converted to boolean in JSON if JSON type is specified as boolean.
	 *  Only these values accepted: true, false, 0, 1, null, dot.  Everything else will throw an exception. */
	public void testBooleanValues() throws InvalidDataException, IOException{
		String tsvIn = "#isInDbSNP" + "\n" +
					   "0" + "\n" +
					   "1" + "\n" +
					   "true" + "\n" +
					   "True" + "\n" + 
					   "TRUE" + "\n" +
					   "false" + "\n" + 
					   "False" + "\n" + 
					   "FALSE" + "\n" +
					   "." + "\n" +
					   "null" + "\n" +
					   "\n";  // This blank line should be removed
		
		String config = 
			//     col  name         type       injector  delim goldenAttr
			concat("1", "isInDbSNP", "BOOLEAN", "COLUMN", ".",  ".");
		
		List<String> actual = getOutput(tsvIn, config);
		
		List<String> expected = Arrays.asList(
			"##BIOR=<ID=\"bior.ToTJson\",Operation=\"bior_tab_to_tjson\",DataType=\"JSON\",ShortUniqueName=\"ToTJson\">",
			"#isInDbSNP" + "\t" + "bior.ToTJson",
			"0" + "\t" + "{\"isInDbSNP\":false}",
			"1" + "\t" + "{\"isInDbSNP\":true}",
			"true" + "\t" + "{\"isInDbSNP\":true}",
			"True" + "\t" + "{\"isInDbSNP\":true}",
			"TRUE" + "\t" + "{\"isInDbSNP\":true}",
			"false" + "\t" + "{\"isInDbSNP\":false}",
			"False" + "\t" + "{\"isInDbSNP\":false}",
			"FALSE" + "\t" + "{\"isInDbSNP\":false}",
			"." + "\t" + "{}",    // Not added in this case
			"null" + "\t" + "{}"  // Not added in this case
			);

		PipeTestUtils.assertListsEqual(expected, actual);
	}
	
	
	@Test
	/** Test bad values for booleans: "xyz", "NA", 2
		All should throw exception. */
	public void testBooleanValues_exceptions() throws InvalidDataException, IOException{
		final String HEADER = "#isInDbSNP" + "\n";
		final String[] VALS = new String[] { "xyz", "NA", "2" };

		String config = 
				//     col  name         type       injector  delim goldenAttr
				concat("1", "isInDbSNP", "BOOLEAN", "COLUMN", ".",  ".");

			// Each one should throw an exception
		for(int i=0; i < VALS.length; i++) {
			try {
				String tsvIn = HEADER + VALS[i] + "\n";
				// Each of the values should throw an exception
				getOutput(tsvIn, config);
				fail("Should have reached an exception here!");
			} catch(Exception e) {
				assertEquals(IllegalArgumentException.class, e.getClass());
				assertEquals("Value could not be converted to a Boolean: " + VALS[i] + ".  Key: isInDbSNP", e.getMessage());

			}
		}
	}
	
	
	
	@Test
	/** Verify that large floats (ex:  123456789.1234567890123456789) are preserved EXACTLY when adding to JSON */
	public void testLargeFloats() throws InvalidDataException, IOException{
		String tsvIn = "#Num\n"
					+  "12345.12345678901" 		+ "\n"	// Double
					+  "1234567.12345" 			+ "\n"	// Double
					+  "123456789.12345" 		+ "\n"	// BigDecimal
					+  "0.1234567890123456" 	+ "\n"	// Double
					+  "1.1234567890123456"		+ "\n"	// BigDecimal
					+  "12345678901234567890.12345678901234567899";	// BigDecimal
		
		String config = 
			//     col  name     type      injector  delim goldenAttr
			concat("1", "Num",   "NUMBER", "COLUMN", ".",  ".");
		
		List<String> actual = getOutput(tsvIn, config);
		
		List<String> expected = Arrays.asList(
			"##BIOR=<ID=\"bior.ToTJson\",Operation=\"bior_tab_to_tjson\",DataType=\"JSON\",ShortUniqueName=\"ToTJson\">",
			concat("#Num",				"bior.ToTJson"),
			concat("12345.12345678901",	"{\"Num\":12345.12345678901}"),
			concat("1234567.12345",		"{\"Num\":1234567.12345}"),
			concat("123456789.12345",	"{\"Num\":123456789.12345}"),
			concat("0.1234567890123456","{\"Num\":0.1234567890123456}"),
			concat("1.1234567890123456","{\"Num\":1.1234567890123456}"),
			concat("12345678901234567890.12345678901234567899",	"{\"Num\":12345678901234567890.12345678901234567899}")
			);

		PipeTestUtils.assertListsEqualExactStringMatch(expected, actual);

		// Now try it with the mock
		File configFile = File.createTempFile("config", "tsv");
		FileUtils.writeStringToFile(configFile, config);
		CommandOutput out = runCmdApp(tsvIn, new Tab2JSONCommand(), "bior_tab_to_tjson", "-c", configFile.getCanonicalPath());
		assertEquals(out.stderr, 0, out.exit);
		assertEquals(out.stderr, "", out.stderr);
		PipeTestUtils.assertListsEqualExactStringMatch(expected, Arrays.asList(out.stdout.split("\n")));
	}
	
	@Test
	/** Verify that there is some output when there is only one input column
	 *  (this was a bug before v4.3.0)
	 *  UPDATE:  This was caused by the previous use of history.hasNext() which caused multiple reads to happen before history.next() was called, and ate all the lines 
	 */
	public void testInputOnlyHasOneColumnAndNoHeader() throws InvalidDataException, IOException {
		String tsvIn = "27" + "\n" +
				       "34" + "\n";
		
		String config = 
				//     col  name         type       injector  delim goldenAttr
				concat("1", "c",         "NUMBER",  "COLUMN", ".",  ".");
	
		// Test thru pipes
		List<String> actual = getOutput(tsvIn, config);
	
		List<String> expected = Arrays.asList(
				"##BIOR=<ID=\"bior.ToTJson\",Operation=\"bior_tab_to_tjson\",DataType=\"JSON\",ShortUniqueName=\"ToTJson\">",
				concat("#UNKNOWN_1", "bior.ToTJson"),
				concat("27", "{\"c\":27}"),
				concat("34", "{\"c\":34}")
				);

		PipeTestUtils.assertListsEqual(expected, actual);
		
		//---------------------------------------------------------------
		// Now test it thru command mock
		File tempFile = File.createTempFile("myinput", "config");
		FileUtils.write(tempFile, config);
		CommandOutput out = runCmdApp(tsvIn, new Tab2JSONCommand(), "bior_tab_to_tjson", "-c", tempFile.getCanonicalPath());
		assertEquals("", out.stderr);
		assertEquals(StringUtils.join(expected, "\n") + "\n", out.stdout);
		assertEquals(0, out.exit);
	}
	
	@Test
	/** Test that _altAlleles correctly goes from multiple values to arrays.
	 * Previously it was just quoting array values, instead of converting them to JSON arrays
	 *   Ex:  "_altAlleles"  "A,C"     should become ["A","C"],    not:  "A,C"
	 *  */
	public void testAltAllelesGoldenAttributeToArray() throws InvalidDataException, IOException{
		String tsvIn = concat("#CHROM",  "POS",  "ID",  "REF",  "ALTS") + "\n" +
					   concat("1",       "100",  "rs1", "A",    "C") + "\n" +
					   concat("2",       "200",  "rs2", "C",    "G,T");
		
		String config = 
			//     col  name      type       injector   delim goldenAttr
			concat("1", "chrom",  "STRING",  "COLUMN",  ".",  ".") + "\n" +
			concat("2", "pos",    "NUMBER",  "COLUMN",  ".",  ".") + "\n" +
			concat("3", "id",     "STRING",  "COLUMN",  ".",  "_id") + "\n" + 
			concat("4", "ref",    "STRING",  "COLUMN",  ".",  "_refAllele") + "\n" +
			// Add "alts" as a String column (not array)
			concat("5", "alts",   "STRING",  "COLUMN",  ".",  ".") + "\n" +
			// Add "alts" column as "_altAlleles" and add to array (comma-separated)
			concat("5", "_altAlleles","STRING","ARRAY", ",",  "_altAlleles") + "\n";
			
		
		List<String> actual = getOutput(tsvIn, config);
		
		List<String> expected = Arrays.asList(
			"##BIOR=<ID=\"bior.ToTJson\",Operation=\"bior_tab_to_tjson\",DataType=\"JSON\",ShortUniqueName=\"ToTJson\">",
			concat("#CHROM",  "POS",  "ID",  "REF",  "ALTS",  "bior.ToTJson"),
			concat("1",       "100",  "rs1", "A",    "C",     "{'chrom':'1','pos':100,'id':'rs1','ref':'A','alts':'C','_id':'rs1','_refAllele':'A','_altAlleles':['C']}".replaceAll("'", "\"")),
			concat("2",       "200",  "rs2", "C",    "G,T",   "{'chrom':'2','pos':200,'id':'rs2','ref':'C','alts':'G,T','_id':'rs2','_refAllele':'C','_altAlleles':['G','T']}".replaceAll("'", "\""))
			);

		PipeTestUtils.assertListsEqual(expected, actual);
	}
	
	
	
	@Test
	/** Test the addition of the "INTEGER" and "FLOAT" types to replace "NUMBER" (NOTE: "NUMBER" should also still work for backward-compatibility) */
	public void testIntegerAndFloatTypes()  throws InvalidDataException, IOException{
		String tsvIn = concat("#CHROM",  "POS",  "REF",  "FloatCol", "IntCol",  "NumCol") + "\n" +
					   concat("1",       "100",  "A",    ".",        ".",       ".") + "\n" +
					   concat("2",       "200",  "C",    "0",        "0",       "0") + "\n" +
					   concat("3",       "300",  "G",    "0.39",     "2",       "9.4");
		
		String config = 
			//     col  name       type       injector   delim goldenAttr
			concat("1", "chrom",   "STRING",  "COLUMN",  ".",  ".") + "\n" +
			concat("2", "pos",     "INTEGER", "COLUMN",  ".",  ".") + "\n" +
			concat("3", "ref",     "STRING",  "COLUMN",  ".",  ".") + "\n" +
			concat("4", "FloatCol","FLOAT",   "COLUMN",  ".",  ".") + "\n" +
			concat("5", "IntCol",  "INTEGER", "COLUMN",  ".",  ".") + "\n" +
			// Still allow the "NUMBER" type for backwards-compatibility
			concat("6", "NumCol",  "NUMBER",  "COLUMN",  ".",  ".");
		
		List<String> actual = getOutput(tsvIn, config);
		
		List<String> expected = Arrays.asList(
			"##BIOR=<ID=\"bior.ToTJson\",Operation=\"bior_tab_to_tjson\",DataType=\"JSON\",ShortUniqueName=\"ToTJson\">",
			concat(       "#CHROM",  "POS",  "REF", "FloatCol", "IntCol", "NumCol", "bior.ToTJson"),
			swapQuotes(concat("1",       "100",  "A",   ".",        ".",      ".",      "{'chrom':'1','pos':100,'ref':'A'}")),
			swapQuotes(concat("2",       "200",  "C",   "0",        "0",      "0",      "{'chrom':'2','pos':200,'ref':'C','FloatCol':0,'IntCol':0,'NumCol':0}")),
			swapQuotes(concat("3",       "300",  "G",   "0.39",     "2",      "9.4",    "{'chrom':'3','pos':300,'ref':'G','FloatCol':0.39,'IntCol':2,'NumCol':9.4}"))
			);

		PipeTestUtils.assertListsEqual(expected, actual);
		
		//---------------------------------------------------------------
		// Now test it thru command mock
		File tempFile = File.createTempFile("myinput", "config");
		FileUtils.write(tempFile, config);
		CommandOutput out = runCmdApp(tsvIn, new Tab2JSONCommand(), "bior_tab_to_tjson", "-c", tempFile.getCanonicalPath());
		assertEquals("", out.stderr);
		assertEquals(StringUtils.join(expected, "\n") + "\n", out.stdout);
		assertEquals(0, out.exit);
	}
	
	
	//===================================================================================================================
	
	/** Get output from the pipes directly (NOT thru the mock) */
	private List<String> getOutput(String tsvIn, String configFileContents) throws InvalidDataException, IOException {
		File tsvInFile = File.createTempFile("input","tsv");
		FileUtils.writeStringToFile(tsvInFile, tsvIn);
		
		File configFile = File.createTempFile("config","tsv");
		FileUtils.writeStringToFile(configFile, configFileContents);
		
		
		Injector[] injectors = new Tab2JSONCommand().parseConfigFile(configFile.getCanonicalPath());
		InjectIntoJsonPipe inject = new InjectIntoJsonPipe(true, injectors);
		
		Pipeline pipe = new Pipeline(
				new CatPipe(),
				new HistoryInPipe(new Metadata("bior_tab_to_tjson")),
				inject,
				new HistoryOutPipe()
				//new PrintPipe()
		    );
		pipe.setStarts(Arrays.asList(tsvInFile.getCanonicalPath()));
		    
		return PipeTestUtils.getResults(pipe);
	}
	
	


	
}
