package edu.mayo.bior.cli.cmd;

import static org.junit.Assert.assertEquals;
import static org.junit.Assert.fail;

import java.io.BufferedReader;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.io.InputStreamReader;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;

import org.junit.Before;
import org.junit.BeforeClass;
import org.junit.Test;
import org.junit.rules.TemporaryFolder;

import edu.mayo.bior.cli.cmd.MergeCommand.ChromRange;
import edu.mayo.pipes.util.test.FileCompareUtils;
import edu.mayo.pipes.util.test.PipeTestUtils;

public class MergeCommandTest {
	
	private final String DIR = "src/test/resources/Merge/";
	private String mTempOutGz;
	private String mTempOutAscii;
	
	@Before
	public void beforeEach() throws IOException {
		TemporaryFolder tempFolder = new TemporaryFolder();
		tempFolder.create();
		File tempDir = tempFolder.newFolder();
		mTempOutGz 		= new File(tempDir, "out.vcf.gz").getCanonicalPath();
		mTempOutAscii 	= new File(tempDir, "out.vcf").getCanonicalPath();
	}
	
	@Test 
	public void twoFiles_oneAscii_oneGz() throws Exception {
		MergeCommand merge = new MergeCommand();
		merge.mergeVcfs(Arrays.asList(DIR + "1.vcf",  DIR + "2.vcf.gz"), mTempOutGz, true, false);
		compareFileContents(DIR + "expected.vcf.gz.1-2", mTempOutGz);
	}

	@Test 
	public void twoFiles_oneAscii_oneGz_removeDuplicates() throws Exception {
		MergeCommand merge = new MergeCommand();
		merge.mergeVcfs(Arrays.asList(DIR + "1.vcf",  DIR + "2.vcf.gz"), mTempOutGz, false, false);
		compareFileContents(DIR + "expected.vcf.gz.1-2", mTempOutGz);
	}

	@Test 
	public void twoFiles_oneAscii_oneGz_keepDuplicates() throws Exception {
		MergeCommand merge = new MergeCommand();
		merge.mergeVcfs(Arrays.asList(DIR + "1.vcf",  DIR + "2.vcf.gz"), mTempOutGz, false, true);
		compareFileContents(DIR + "expected.vcf.1-2.dupes", mTempOutGz);
	}

	@Test 
	public void twoFiles_oneAscii_oneGz_unscramble() throws Exception {
		MergeCommand merge = new MergeCommand();
		merge.mergeVcfs(Arrays.asList(DIR + "1.vcf",  DIR + "1.scrambled.vcf"), mTempOutGz, false, false);
		compareFileContents(DIR + "expected.vcf.1-2.scrambled", mTempOutGz);
	}

	@Test 
	public void twoFiles_oneAscii_oneGz_chrPrefix() throws Exception {
		MergeCommand merge = new MergeCommand();
		merge.mergeVcfs(Arrays.asList(DIR + "1.chr.vcf",  DIR + "2.vcf.gz"), mTempOutGz, false, true);
		compareFileContents(DIR + "expected.vcf.chr1-2", mTempOutGz);
	}

	@Test
	public void twoFiles_oneAscii_oneGz_getChromRanges() throws IOException {
		MergeCommand merge = new MergeCommand();
		List<ChromRange> chromRanges = merge.getAllChromRangesAllFilesSorted(Arrays.asList(DIR + "1.vcf",  DIR + "2.vcf.gz"));
		
		List<String> actual = new ArrayList<String>();
		for(ChromRange range : chromRanges) 
			actual.add(range.toString());
		
		List<String> expected = Arrays.asList(
			"[1:100-100, src/test/resources/Merge/1.vcf:97 (1 lines)]",
			"[1:100-100, src/test/resources/Merge/2.vcf.gz:103 (1 lines)]",
			"[2:200-200, src/test/resources/Merge/1.vcf:119 (1 lines)]",
			"[2:201-201, src/test/resources/Merge/2.vcf.gz:125 (1 lines)]",
			"[3:299-299, src/test/resources/Merge/1.vcf:141 (1 lines)]",
			"[3:300-300, src/test/resources/Merge/2.vcf.gz:147 (1 lines)]"
			);
		
		PipeTestUtils.assertListsEqual(expected, actual);
	}

	@Test
	public void chromRanges_7files() throws IOException {
		MergeCommand merge = new MergeCommand();
		final String CONCAT_DIR = "src/test/resources/Concat/";
		List<ChromRange> chromRanges = merge.getAllChromRangesAllFilesSorted(Arrays.asList(
				CONCAT_DIR + "my.vcf.1",
				CONCAT_DIR + "my.vcf.2",
				CONCAT_DIR + "my.vcf.3",
				CONCAT_DIR + "my.vcf.4",
				CONCAT_DIR + "my.vcf.5",
				CONCAT_DIR + "my.vcf.6",
				CONCAT_DIR + "my.vcf.done"
				));
		List<String> actual = new ArrayList<String>();
		for(ChromRange range : chromRanges) 
			actual.add(range.toString());
		List<String> expected = Arrays.asList(
			"[1:100-200, src/test/resources/Concat/my.vcf.1:39 (3 lines)]",
			"[2:80-100, src/test/resources/Concat/my.vcf.1:93 (2 lines)]",
			"[2:300-401, src/test/resources/Concat/my.vcf.2:39 (2 lines)]",
			"[3:800-810, src/test/resources/Concat/my.vcf.2:75 (2 lines)]",
			"[4:1000-1000, src/test/resources/Concat/my.vcf.2:111 (1 lines)]",
			"[4:9000-9000, src/test/resources/Concat/my.vcf.3:39 (1 lines)]",
			"[5:700-700, src/test/resources/Concat/my.vcf.3:58 (1 lines)]",
			"[6:1000-1000, src/test/resources/Concat/my.vcf.3:76 (1 lines)]",
			"[7:55-55, src/test/resources/Concat/my.vcf.3:95 (1 lines)]",
			"[8:10000-10000, src/test/resources/Concat/my.vcf.3:112 (1 lines)]",
			"[9:900-900, src/test/resources/Concat/my.vcf.4:39 (1 lines)]",
			"[10:30-8100, src/test/resources/Concat/my.vcf.4:57 (4 lines)]",
			"[10:30000-30000, src/test/resources/Concat/my.vcf.5:39 (1 lines)]",
			"[11:700-1400, src/test/resources/Concat/my.vcf.5:60 (4 lines)]",
			"[11:23000-23000, src/test/resources/Concat/my.vcf.6:39 (1 lines)]"
			);
		PipeTestUtils.assertListsEqual(expected, actual);
		
	}
	
	@Test
	public void merge_7sequentialfiles() throws Exception {
		MergeCommand merge = new MergeCommand();
		final String CONCAT_DIR = "src/test/resources/Concat/";
		List<String> inVcfs = Arrays.asList(
				CONCAT_DIR + "my.vcf.1",
				CONCAT_DIR + "my.vcf.2",
				CONCAT_DIR + "my.vcf.3",
				CONCAT_DIR + "my.vcf.4",
				CONCAT_DIR + "my.vcf.5",
				CONCAT_DIR + "my.vcf.6",
				CONCAT_DIR + "my.vcf.done");
		merge.mergeVcfs(inVcfs, mTempOutGz, false, false);
		compareFileContents(CONCAT_DIR + "expected.my.vcf", mTempOutGz);

	}
	
	@Test
	public void junk() throws IOException {
		BufferedReader fin = new BufferedReader(new InputStreamReader(new FileInputStream(DIR + "1.vcf")));
		fin.mark(100);
		String line1 = fin.readLine();
		fin.skip(101);
		String line2 = fin.readLine();
		fin.reset();
		String line3 = fin.readLine();
		System.out.println("Done.");
	}
	
	@Test 
	public void fourFiles_3Ascii_1Gz() throws Exception {
		MergeCommand merge = new MergeCommand();
		merge.mergeVcfs(Arrays.asList(DIR + "1.vcf",  DIR + "2.vcf.gz",  DIR + "3.vcf",  DIR + "4.vcf"), mTempOutAscii, false, false);
		compareFileContents(DIR + "expected.vcf.1-4", mTempOutAscii);
	}

	//===========================================================================
	private void compareFileContents(String expectedFilePath, String actualFilePath) throws IOException {
		List<String> expected = FileCompareUtils.loadFile(expectedFilePath);
		List<String> actual   = FileCompareUtils.loadFile(actualFilePath);
		PipeTestUtils.assertListsEqual(expected, actual);
	}

}
