package edu.mayo.bior.cli.cmd;

import static org.junit.Assert.fail;

import java.io.File;
import java.io.FilenameFilter;
import java.io.IOException;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.Comparator;
import java.util.List;

import org.apache.commons.io.FileUtils;
import org.junit.Test;


/**Scan the list of COMMAND.properties files and the src/main/scripts/bior* scripts to get the 
 * flags for all commands so we can compare them and figure out if there are overloading of flags for different values between commands
 * @author m054457
 *
 */
public class CommandLineFlagRoundup {
	   
	public class CmdFlag {
		public String flagShort;
		public String flagLong;
		public String name;
		public String desc;
		public String cmd;
		public String filename;
		public boolean isStandardMayoCommonsCliCmd;
		
		public CmdFlag() { }
		
		public CmdFlag(String flagShort, String flagLong, String name, String cmd, String filename, boolean isStdToolkitCmd, String desc) {
			this.flagShort = flagShort;
			this.flagLong  = flagLong;
			this.name      = name;
			this.desc      = desc;
			this.cmd       = cmd;
			this.filename  = filename;
			this.isStandardMayoCommonsCliCmd = isStdToolkitCmd;
		}
		
		public String toString() {
			return  isStandardMayoCommonsCliCmd ? "" : "*"
					+ (isGiven(flagShort) ? ("-" + flagShort + "\t") : "")
					+ "--" + flagLong + "\t"
					+ name + "\t"
					+ cmd
					+ " (" + filename + ")\t"
					+ desc;
		}

		private boolean isGiven(String s) {
			return s != null && s.length() > 0;
		}
	}

	@Test
	public void roundUpThemCommandFlags_YeeHaw() throws IOException {
		// ----- Command properties files ------------------------------------
		List<File> propFiles = new ArrayList<File>();
		propFiles.addAll(Arrays.asList(getPropsFilesFromDir("src/main/resources/cli")));
		propFiles.addAll(Arrays.asList(getPropsFilesFromDir("src/main/resources/tools")));
		
		List<CmdFlag> cmdFlags = new ArrayList<CmdFlag>();
		for(File f : propFiles) {
			cmdFlags.addAll(getCmdFlagsFromPropsFiles(f));
		}
		
		// ----- Scripts -----------------------------------------------------
		List<File> scriptFiles = getScriptFiles("src/main/scripts");
		for(File f : scriptFiles) {
			cmdFlags.addAll(getCmdFlagsFromScript(f));
		}

		// ----- Built-in flags -----------------------------------------------------
		cmdFlags.addAll(getCmdFlagsBuiltIn());
		
		// ----- Sort and print ----------------------------------------------
		Collections.sort(cmdFlags, getCmdFlagsComparator());
		printCmdFlags(cmdFlags);
		
		// ----- Verify ------------------------------------------------------
		fail("NOT FINISHED YET");

	}


	private List<CmdFlag> getCmdFlagsBuiltIn() {
		List<CmdFlag> flags = new ArrayList<CmdFlag>();
		flags.add(new CmdFlag("h", "help",    "help",   "(all)", "(N/A)", true, "Display help text and exit"));
		flags.add(new CmdFlag("l", "log",     "log",    "(all)", "(N/A)", true, "Generate the default log file 'bior.log' in the current directory.  By default, the log file is not generated"));
		flags.add(new CmdFlag("lf","logfile", "logfile","(all)", "(N/A)", true,"Generate the log file as specified by <logfile_name>, which overrides the default log file name, as shown in the -l, --log description."));
		flags.add(new CmdFlag("v", "version", "version","(all)", "(N/A)", true,"Display the version of the application and exit"));
		flags.add(new CmdFlag("remote", "remote-debug", "remote-debug", "(all)", "(N/A)", true, "Allow attaching Java remote debugger to the command (hidden)"));
		return flags;
	}


	private File[] getPropsFilesFromDir(String dir) {
		return new File(dir).listFiles(new FilenameFilter() {
			public boolean accept(File parentDir, String filename) {
				return filename.endsWith(".properties");
			}
		});
	}
	   
	private List<CmdFlag>  getCmdFlagsFromPropsFiles(File f) throws IOException {
		CmdFlag cmdFlag = null;
		List<CmdFlag> cmdFlags = new ArrayList<CmdFlag>();
		List<String> lines = FileUtils.readLines(f);
		String cmd = f.getName();  // overwrite with command.name if available
		String filename = f.getName();
		CmdFlag flag = new CmdFlag();
		boolean isInFlag = false;
		for(String line : lines) {
			line = line.trim();
			if( line.contains("command.name") ) {
				cmd = line.replace("command.name=", "").trim();
			} else if( line.startsWith("flag.") ) {
				flag = new CmdFlag();
				flag.isStandardMayoCommonsCliCmd = true;
				flag.cmd = cmd;
				flag.filename = filename;
				flag.name = line.substring(line.indexOf(".")+1, line.indexOf("="));
				isInFlag = true;
			} else if( isInFlag && line.startsWith("\"opt\"") ) {
				flag.flagShort = getVal(line);
			} else if( isInFlag && line.startsWith("\"longOpt\"") ) {
				flag.flagLong = getVal(line);
			} else if( isInFlag && line.startsWith("\"description\"") ) {
				flag.desc = getVal(line);
				cmdFlags.add(flag);
			} else if( isInFlag && line.equals("}") ) {
				isInFlag = false;
			}
		}
		return cmdFlags;
	}

	/** Given a line like:  '   "longOpt": "keep-json",       \'
	 *  Return the "keep-json" part (without quotes)	 */
	private String getVal(String line) {
		String val = line.substring(line.indexOf(":")+1, line.lastIndexOf("\\")).replace("\"", "").replace("\\n", "").replace(",", "").trim();
		return val;
	}

	private List<File> getScriptFiles(String dir) {
		return Arrays.asList(new File(dir).listFiles(new FilenameFilter() {
			public boolean accept(File parentDir, String filename) {
				return filename.startsWith("bior_")  || filename.startsWith("_bior_");
			}
		}));
	}

	/** Get flags from odd scripts.
	 * Ex:
	 *       echo "	-d, --database <DATABASE>"
	 *       echo "		The catalog file (bgzip format) containing the JSON data to lookup."
	 * Or:
	 *       echo "  [--tempDir tempDir]  : Temporary directory to write the output files."
	 * @param scriptFile
	 * @return
	 * @throws IOException 
	 */
	private List<CmdFlag> getCmdFlagsFromScript(File scriptFile) throws IOException {
		List<String> lines = FileUtils.readLines(scriptFile);
		List<CmdFlag> cmdFlags = new ArrayList<CmdFlag>();
		for(int i=0; i<lines.size(); i++) {
			String line = lines.get(i).trim();
			String nextLine = lines.size() > (i+1)  ?  lines.get(i+1).trim()  :  "";
			if( line.startsWith("echo \"  -") ||  line.startsWith("echo \"\t-") ) {
				cmdFlags.add(getCmdFlagType1(scriptFile, line, nextLine));
			} else if( line.startsWith("echo \"  [--") ) {
				cmdFlags.add(getCmdFlagType2(scriptFile, line));
			}
		}
		return cmdFlags;
	}

	/** Get flags from odd scripts with lines of this format:
	 *       echo "	-d, --database <DATABASE>"
	 *       echo "		The catalog file (bgzip format) containing the JSON data to lookup."
	 */
	private CmdFlag getCmdFlagType1(File scriptFile, String line, String nextLine) {
		CmdFlag cmdFlag = new CmdFlag();
		cmdFlag.isStandardMayoCommonsCliCmd = false;
		cmdFlag.cmd = scriptFile.getName();
		cmdFlag.filename = scriptFile.getName();
		cmdFlag.flagShort = getShortFlag_type1(line);
		cmdFlag.flagLong  = getLongFlag_type1(line);
		cmdFlag.name = cmdFlag.flagLong;
		cmdFlag.desc = nextLine.replace("echo \"", "").replaceAll("\"", "").trim();
		return cmdFlag;
	}
	
	/** Get flags from odd scripts with lines of this format:
	 *       echo "  [--tempDir tempDir]  : Temporary directory to write the output files."
	 */
	private CmdFlag getCmdFlagType2(File scriptFile, String line) {
		CmdFlag cmdFlag = new CmdFlag();
		cmdFlag.isStandardMayoCommonsCliCmd = false;
		cmdFlag.cmd = scriptFile.getName();
		cmdFlag.filename = scriptFile.getName();
		cmdFlag.flagShort = "";
		cmdFlag.flagLong  = getLongFlag_type2(line);
		cmdFlag.name = cmdFlag.flagLong;
		cmdFlag.desc = getDesc_type2(line);
		return cmdFlag;
	}




	/** Given the following line, get the description from the end of it
	 *      echo "  [--tempDir tempDir]  : Temporary directory to write the output files."   */
	private String getDesc_type2(String line) {
		int idxColon = line.indexOf(":");
		if( idxColon == -1 ) 
			return "";
		return line.substring(idxColon+1).replace("\"", "").trim();
	}


	/** Given a flag line like this, extract the short flag ("d")
	 *       echo "	-d, --database <DATABASE>"	 */
	private String getShortFlag_type1(String line) {
		int idxDash = line.indexOf("-");
		int idxSpace     = line.indexOf(" ", idxDash);
		if( idxSpace == -1 )
			idxSpace = line.length();
		if( idxDash == -1 )
			return "";
		return line.substring(idxDash, idxSpace).replace(",", "").replace(" ", "").replace("-", "").replace("]", "");
	}

	/** Given a flag line like this, extract the long flag ("database")
	 *       echo "	-d, --database <DATABASE>"
	 *  Or, given 	 */
	private String getLongFlag_type1(String line) {
		int idxDash = line.indexOf("--");
		int idxSpace     = line.indexOf(" ", idxDash);
		if( idxSpace == -1 )
			idxSpace = line.length();
		return line.substring(idxDash, idxSpace).replace(",", "").replace(" ", "").replace("--", "").replace("]", "").replace("\"", "");
	}

	/** Given a flag line like this, extract the long flag ("tempDir")
	 *    echo "  [--tempDir tempDir]  : Temporary directory to write the output files."   */
	private String getLongFlag_type2(String line) {
		return getLongFlag_type1(line);
	}

	private Comparator<? super CmdFlag> getCmdFlagsComparator() {
		return new Comparator<CmdFlag>() {
			public int compare(CmdFlag f1, CmdFlag f2) {
				// Try comparing by short flag first, but if same or one is not given, then compare by long flag
				if( ! isGiven(f1.flagShort)  ||  ! isGiven(f2.flagShort) )
					return f1.flagLong.compareTo(f2.flagLong);
				
				int compare = f1.flagShort.compareToIgnoreCase(f2.flagShort);
				if( compare != 0 )
					return compare;
				
				return f1.flagLong.compareToIgnoreCase(f2.flagLong);
			}

			private boolean isGiven(String s) {
				return s != null && s.trim().length() > 0;
			}
		};
	}


	/** Print a table of command line flags:
	 *   flagShort, flagLong, name, cmd, filename, desc
	 */
	private void printCmdFlags(List<CmdFlag> cmdFlags) {
		
		int[] maxColSizes = getMaxColSizes(cmdFlags);
		
		System.out.println("* = Not a standard mayo-commons-cli command with properties file");
		System.out.println( getRowFillOut( Arrays.asList("FlagShort", "FlagLong", "Name", "Command", "Filename", "Description"), maxColSizes, " "));
		System.out.println( getRowFillOut( Arrays.asList("", "", "", "", "", ""), maxColSizes, "-"));
		
		
		for(CmdFlag cmdFlag : cmdFlags) {
			String oddCmd = cmdFlag.isStandardMayoCommonsCliCmd ? "" : "* ";
			System.out.println( getRowFillOut( Arrays.asList(
					oddCmd + cmdFlag.flagShort,
					cmdFlag.flagLong,
					cmdFlag.name,
					cmdFlag.cmd,
					cmdFlag.filename,
					cmdFlag.desc
					), maxColSizes,  " ") );
		}
		
	}

	private String getRowFillOut(List<String> cellVals, int[] maxColSizes, String fillOutWithChr) {
		StringBuilder s = new StringBuilder();
		for(int i=0; i < cellVals.size(); i++) {
			s.append(pad(cellVals.get(i), maxColSizes[i], fillOutWithChr));
		}
		return s.toString();
	}


	private String pad(String s, int colWidth) {
		return pad(s, colWidth, " ");
	}
	
	private String pad(String s, int colWidth, String chr) {
		StringBuilder str = new StringBuilder(s);
		while(str.length() < colWidth)
			str.append(chr);
		return str.toString();
	}


	/** Max col sizes for: (+2 for space at end)
	 *   flagShort, flagLong, name, cmd, filename, desc
	 */
	private int[] getMaxColSizes(List<CmdFlag> cmdFlags) {
		int[] max = new int[6];
		for(CmdFlag flag : cmdFlags) {
			if(flag.flagShort.length() > max[0])
				max[0] = flag.flagShort.length();
			
			if(flag.flagLong.length() > max[1])
				max[1] = flag.flagLong.length();

			if(flag.name.length() > max[2])
				max[2] = flag.name.length();

			if(flag.cmd.length() > max[3])
				max[3] = flag.cmd.length();

			if(flag.filename.length() > max[4])
				max[4] = flag.filename.length();

			if(flag.desc.length() > max[5])
				max[5] = flag.desc.length();
		}
		
		for(int i=0; i < max.length; i++) {
			max[i] = max[i] + 2;
		}
		
		return max;
	}





}
