package edu.mayo.bior.cli.cmd;

import static org.junit.Assert.assertTrue;
import static org.junit.Assert.fail;

import java.io.File;
import java.io.FileNotFoundException;
import java.io.FileReader;
import java.io.IOException;
import java.util.List;
import java.util.Properties;

import org.apache.commons.io.FileUtils;
import org.junit.Test;

import com.google.gson.Gson;
import com.google.gson.JsonElement;
import com.google.gson.JsonObject;
import com.google.gson.JsonParser;

public class CmdPropertiesFileValidatorTest {

	
	@Test
	public void validateAllCmdPropFiles() throws FileNotFoundException, IOException {
		File[] propFiles = getPropFiles();
		int numProps = 0;
		for(File propFile : propFiles) {
			assertNoFilesHaveSpacesAfterSlashes(propFile);
			Properties props = loadPropertiesFile(propFile);
			for(Object keyObj : props.keySet()) {
				numProps++;
				String key = (String)keyObj;
				String val = props.getProperty(key);
				try {
					//System.out.println("===========================================");
					//System.out.print(keyObj + " = ");
					if(val.trim().length() == 0)
						throw new Exception("Value is empty.  Did you forget a slash at the end of a line?");
					validatePropKey(key);
					validatePropValue(val);
				} catch(Exception e) {
					e.printStackTrace();
					fail("Error occurred processing properties file:\n"
							+ propFile.getCanonicalPath() + "\n"
							+ "Key:   " + keyObj + "\n"
							+ "Value: " + val + "\n"
							+ e.getMessage() );
				}
			}
		}
		System.out.println(propFiles.length + " properties files processed, containing " + numProps + " properties");
	}

	/** Lines in a properties file similar to " this. \ " will cause problems because of the whitespace following the slash */
	private void assertNoFilesHaveSpacesAfterSlashes(File propFile) throws IOException {
		List<String> lines = FileUtils.readLines(propFile);
		StringBuilder str = new StringBuilder();
		for(int i=0; i < lines.size(); i++) {
			String line = lines.get(i);
			if( line.trim().endsWith("\\")  &&  ! line.endsWith("\\") )
				str.append("ERROR:  Line ending with slash has whitespace after it!:  \n    line " + (i+1) + ". [" + line + "]\n");
		}
		if( str.length() > 0 )
			fail(str.toString());
	}


	private File[] getPropFiles() {
		File dir = new File("src/main/resources/cli");
		File[] propFiles = dir.listFiles();
		//File[] propFiles = new File[] {
		//		new File("src/main/resources/cli/TjsonToVcfCommand.properties")
		//};
		return propFiles;
	}

	private void validatePropKey(String key) throws Exception {
		boolean isValidKeyWhole = key.equals("command.class")
				||  key.equals("command.name")
				||  key.equals("short.description")
				||  key.equals("long.description")
				||  key.equals("jvm.opts")
				||  key.startsWith("SAGE_")
				||  key.equals("DEFAULT_VALUE_COUNT");  // Added for CatalogStatsCommand.properties
		boolean isValidKeyPrefix = key.startsWith("flag.")
				||  key.startsWith("arg.")
				||  key.startsWith("example.");
		if( ! (isValidKeyWhole || isValidKeyPrefix) )
			throw new Exception("Key or key prefix not recognized: " + key 
				+ ".  It could be that you have a space after one of the backslashes at the end of a line");
	}

	private void validatePropValue(String propValue) {
		String out = propValue;
		if( propValue.trim().startsWith("{") ) {
			JsonElement json = new JsonParser().parse(propValue);
			out = json.toString();
		}
		//System.out.println(out);
	}

	
	private Properties loadPropertiesFile(File propFile) throws FileNotFoundException, IOException {
		Properties props = new Properties();
		props.load(new FileReader(propFile));
		return props;
	}
}
